/*************************************************
*            BK Emulator for Window 3.0          *
*************************************************/

/*
File: TapeManagerDlg.cpp
Version: 1.0.0
Written by: Yuriy Kalmykov <kalmykov@stoik.com>
    Copyright (c) 2002-2004 Yuriy Kalmykov

    BK Emulator is a program emulated hardware environment for running
code for BK 0010(01) in different configurations. 
           
    This code may be used in compiled form in any way you desire.
This file or it's parts can't be redistributed without the authors
written consent, but can be modified for your private needs.
    Providing that this notice and the authors name and all copyright
notices remains intact.

    Please, an email me to know how you are using it and where. You can
ask me for any information about this below code or any attendant
knowledge.
    
    This file is provided "as is" with no expressed or implied warranty.
The author accepts no liability for any damage or loss of business that
this product may cause.
*/




// TapeManagerDlg.cpp : implementation file
//

#include "stdafx.h"
#include "bk.h"
#include "Globals.h"
#include "TapeManagerDlg.h"
#include "BKFileDialog.h"
#include "BrowseForFolder.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

#define WM_BUFFER_READY (WM_USER + 101)
#define WM_INFO_READY (WM_USER + 102)

/////////////////////////////////////////////////////////////////////////////
// CTapeManagerDlg dialog


CTapeManagerDlg::CTapeManagerDlg(CWnd* pParent /*=NULL*/)
	: CBKDialog(CTapeManagerDlg::IDD, pParent)
{
	//{{AFX_DATA_INIT(CTapeManagerDlg)
		// NOTE: the ClassWizard will add member initialization here
	//}}AFX_DATA_INIT
	m_nOutStringsCount = OUTSTRINGS_COUNT;
	m_bCaptureMode = FALSE;
	m_nHeaderBuffer = 0;

	m_pTapeUnit = NULL;

	::GetCurrentDirectory (_MAX_PATH, m_strCurrentFolder.GetBufferSetLength (_MAX_PATH)); 
	m_strCurrentFolder.ReleaseBuffer ();
	NormalizePath (m_strCurrentFolder);
}


CTapeManagerDlg::~CTapeManagerDlg ()
{
}



void CTapeManagerDlg::DoDataExchange(CDataExchange* pDX)
{
	CBKDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CTapeManagerDlg)
	DDX_Control(pDX, IDC_TM_LIST, m_list);
	//}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(CTapeManagerDlg, CBKDialog)
	//{{AFX_MSG_MAP(CTapeManagerDlg)
	ON_MESSAGE (WM_BUFFER_READY, OnBufferReady)
	ON_MESSAGE (WM_INFO_READY, OnInfoReady)
	ON_BN_CLICKED(IDC_TM_LOAD, OnTmLoad)
	ON_NOTIFY(LVN_GETDISPINFO, IDC_TM_LIST, OnGetdispinfoTmList)
	ON_BN_CLICKED(IDC_TM_BROWSE, OnTmBrowse)
	ON_BN_CLICKED(IDC_TM_REMOVE, OnTmRemove)
	ON_UPDATE_COMMAND_UI (IDC_TM_REMOVE, OnUpdateTmRemove)
	ON_CBN_SELENDOK(IDC_TM_SAVE_AS, OnSelendokTmSaveAs)
	ON_UPDATE_COMMAND_UI (IDC_TM_SAVE_AS, OnUpdateSelendokTmSaveAs)
	ON_WM_TIMER()
	ON_NOTIFY(NM_RCLICK, IDC_TM_LIST, OnRclickTmList)
	ON_COMMAND(ID_TM_SELECTALL, OnTmSelectall)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CTapeManagerDlg message handlers

BOOL CTapeManagerDlg::InitDirectSound ()
{
	CDSTHResult hr;

	try
	{
		// Create Capture object
		hr = DirectSoundCaptureCreate (NULL, &m_pDirectSoundCapture, NULL);

		// Query capture format
		DSCCAPS dsccaps;
		dsccaps.dwSize = sizeof(DSCCAPS);

		hr = m_pDirectSoundCapture->GetCaps (&dsccaps);
		
		if ((dsccaps.dwFormats & WAVE_FORMAT_4M08) == 0)
			return FALSE;

		ASSERT (m_wfx.nAvgBytesPerSec % EVENTS_NUM == 0);

		m_wfx.cbSize			= sizeof (WAVEFORMATEX);
		m_wfx.wFormatTag		= WAVE_FORMAT_PCM;
		m_wfx.nSamplesPerSec	= BUFFER_FREQUENCY;
		m_wfx.wBitsPerSample	= BUFFER_BPS;
		m_wfx.nChannels			= 1;
		m_wfx.nBlockAlign		= (m_wfx.wBitsPerSample >> 3) * m_wfx.nChannels;
		m_wfx.nAvgBytesPerSec	= m_wfx.nSamplesPerSec * m_wfx.nBlockAlign;

		// Creare Capture Buffer
		memset (&m_BD, 0, sizeof (DSCBUFFERDESC));
		m_BD.dwSize = sizeof(DSCBUFFERDESC);
		m_BD.dwFlags = 0;
		m_BD.dwBufferBytes = m_wfx.nAvgBytesPerSec;
		m_BD.dwReserved = 0;
		m_BD.lpwfxFormat = &m_wfx;
 
		hr = m_pDirectSoundCapture->CreateCaptureBuffer ((LPCDSCBUFFERDESC)&m_BD, &m_pCaptureBuffer, NULL);
		hr = m_pCaptureBuffer->QueryInterface (IID_IDirectSoundNotify, (void**)&m_pNotify);

		InitEvents ();
		InitBuffers ();
				
		hr = m_pNotify->SetNotificationPositions (EVENTS_NUM, m_pNotifyPositions);

		// Start notify event thread
		m_pNotifyThread = AfxBeginThread ((AFX_THREADPROC)EventThreadWrapper, (void*)this, THREAD_PRIORITY_NORMAL);
		m_pNotifyThread->m_bAutoDelete = FALSE;

		m_hBufferReadyEvent = ::CreateEvent (NULL, FALSE, FALSE, NULL);
	}
	catch (CDSTFailQueryInterfaceException)
	{
		return FALSE;
	}

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::InitEvents ()
{
	m_pNotifyPositions = new DSBPOSITIONNOTIFY[EVENTS_NUM];
	m_pNotifyEvents = new HANDLE[EVENTS_NUM + 1]; // + Exit event

	for (int i = 0; i < EVENTS_NUM + 1; i++)
		m_pNotifyEvents[i] = ::CreateEvent (NULL, FALSE, FALSE, NULL);
	
	int position = 0;
	int step = BUFFER_FREQUENCY * m_wfx.nBlockAlign / EVENTS_NUM;

	for (i = 0; i < EVENTS_NUM; i++)
	{
		m_pNotifyPositions[i].dwOffset = position;
		m_pNotifyPositions[i].hEventNotify = m_pNotifyEvents[i + 1];

		position += step;
	}
}



//////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::InitBuffers ()
{
	RemoveBuffers ();
	ASSERT (m_wfx.nAvgBytesPerSec % EVENTS_NUM == 0);

	for (int n = 0; n < CAPTURE_BUFFERS_NUM; n++)
	{
		BYTE* pBuff = new BYTE[m_wfx.nAvgBytesPerSec / EVENTS_NUM];
		memset (pBuff, 0, m_wfx.nAvgBytesPerSec / EVENTS_NUM);
		m_captureBuffers.AddTail (pBuff);
	}
}



//////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::RemoveBuffers ()
{
	// delete all buffers memory
	POSITION pos = m_captureBuffers.GetHeadPosition ();

	while (pos)
	{
		delete []m_captureBuffers.GetNext (pos);
	}
}



//////////////////////////////////////////////////////////////////////
int CTapeManagerDlg::EventThreadWrapper (CTapeManagerDlg* pThis)
{
	pThis->EventThread ();

	return 0;
}



/////////////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::EventThread ()
{
	CDSTHResult hr;
	
	while (1)
	{
		int object = ::WaitForMultipleObjects (EVENTS_NUM + 1, m_pNotifyEvents, FALSE, INFINITE);

		// If exit event
		if (object == WAIT_OBJECT_0)
			break;

		// Calculate buffer offset 
		int nBufferLength = m_wfx.nAvgBytesPerSec / EVENTS_NUM;
		int nOffset = (object - WAIT_OBJECT_0 - 1 - 1) * nBufferLength;

		if (nOffset < 0)
			nOffset += nBufferLength * EVENTS_NUM;

		TRACE ("\n\n Capture offset = %i", nOffset);

		CSingleLock lock (&m_csModifyBuffer, TRUE);

		// Get current buffer
		BYTE* pBuff;
		DWORD nSize;
		hr = m_pCaptureBuffer->Lock (nOffset, nBufferLength, (void**)&pBuff, &nSize, NULL, NULL, 0);

		TRACE ("\n Capture Lock size = %i", nSize);
		
		if (pBuff)
		{
			BYTE* pTailBuff = m_captureBuffers.RemoveHead ();
			memcpy (pTailBuff, pBuff, nBufferLength);
			m_captureBuffers.AddTail (pTailBuff);
		}

		hr = m_pCaptureBuffer->Unlock (pBuff, nSize, NULL, NULL);

		lock.Unlock ();

		// Set "Buffer ready" event
		::SetEvent (m_hBufferReadyEvent);

		SendMessage (WM_BUFFER_READY); // Changed from PostMessage
	}
}



/////////////////////////////////////////////////////////////////////////////
BOOL CTapeManagerDlg::OnInitDialog() 
{
	CBKDialog::OnInitDialog();
	
	InitDirectSound ();
	SetTimer (CTRL_UPDATE_TIMER, 100, NULL);

	if (!m_pCaptureBuffer)
	{
		// Initialization error - close dialog
		CString strMessage;
		strMessage.LoadString (IDS_TAPAMNGR_CAPTURE_ERROR);
		
		CBKMessageBox (strMessage, MB_OK|MB_ICONSTOP);
	}

	if (m_pCaptureBuffer)
		m_pCaptureBuffer->Start (DSCBSTART_LOOPING);

	// Start status event thread
	m_pStatusThread = AfxBeginThread ((AFX_THREADPROC)StatusThreadWrapper, (void*)this, THREAD_PRIORITY_NORMAL);
	m_pStatusThread->m_bAutoDelete = FALSE;

	InitListCtrl ();
	InitComboCtrl ();
	SetFolder (m_strCurrentFolder);
	
	return TRUE;  // return TRUE unless you set the focus to a control
	              // EXCEPTION: OCX Property Pages should return FALSE
}



/////////////////////////////////////////////////////////////////////////////
BOOL CTapeManagerDlg::WaitForBufferReady (int time)
{
	//TRACE ("\nWait buffer time = %i", time);

	if (::WaitForSingleObject (m_hBufferReadyEvent, time) == WAIT_OBJECT_0)
		return TRUE;

	return FALSE;
}



/////////////////////////////////////////////////////////////////////////////
DWORD CTapeManagerDlg::GetAllBuffersLength ()
{
	return GetBufferLength () * CAPTURE_BUFFERS_NUM;
}



/////////////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::GetAllBuffers (BYTE* pBuffer)
{
	CSingleLock lock (&m_csModifyBuffer, TRUE);

	POSITION pos = m_captureBuffers.GetHeadPosition ();
	int nBuffer = 0;
	int nBufferLength = GetBufferLength ();

	while (pos)
	{
		BYTE* pBuff = m_captureBuffers.GetNext (pos);
		memcpy (pBuffer + nBuffer * nBufferLength, pBuff, nBufferLength);
		nBuffer++;
	}
}



/////////////////////////////////////////////////////////////////////////////
DWORD CTapeManagerDlg::GetBufferLength ()
{
	return m_wfx.nAvgBytesPerSec / EVENTS_NUM;
}



/////////////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::GetBuffer (BYTE* pBuffer)
{
	CSingleLock lock (&m_csModifyBuffer, TRUE);

	BYTE* pTailBuff = m_captureBuffers.GetTail ();
	memcpy (pBuffer, pTailBuff, GetBufferLength ());
}



/////////////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::ClearBuffers ()
{
	CSingleLock lock (&m_csModifyBuffer, TRUE);

	POSITION pos = m_captureBuffers.GetHeadPosition ();
	int nBufferLength = GetBufferLength ();

	while (pos)
	{
		BYTE* pBuff = m_captureBuffers.GetNext (pos);
		memset (pBuff, 128, nBufferLength);
	}
}



/////////////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::OutString (CString strOut)
{
	if (m_outStrings.GetCount () >= OUTSTRINGS_COUNT)
		m_outStrings.RemoveHead ();

	strOut = CString ("\r\n") + strOut;

	m_outStrings.AddTail (strOut);

	CString strOutput;

	POSITION pos = m_outStrings.GetHeadPosition ();

	while (pos)
		strOutput += m_outStrings.GetNext (pos);

	SetDlgItemText (IDC_TM_OUT, strOutput);

	CEdit* pEdit = (CEdit*)GetDlgItem (IDC_TM_OUT);
	pEdit->LineScroll (OUTSTRINGS_COUNT);
}



/////////////////////////////////////////////////////////////////////////////
HRESULT CTapeManagerDlg::OnBufferReady (WPARAM wParam, LPARAM lParam)
{
	WaitForBufferReady ();

	if (!m_nHeaderBuffer)
	{
		if (m_bCaptureMode == FALSE)
		{
			// Allocate new buffer for wave header
			DWORD nLength = GetAllBuffersLength ();
			m_headReader.AllocWaveBuffer (nLength);

			// Copy captured wave to tape parser
			GetAllBuffers (m_headReader.GetWaveBuffer ());

			// Try to find file header
			TAPE_FILE_INFO tfi;
			m_headReader.GetWaveFile (&tfi, TRUE);

			//TRACE ("\n tfi.start_tuning = %i", tfi.start_tuning);
			//TRACE ("\n tfi.synchro_header = %i", tfi.synchro_header);

			DebugDraw ();

			if (tfi.synchro_data != -1)
			{
				// If detect header - prepare context for capture
				m_captureOffset = tfi.start_tuning;
				m_captureLength = 0;

				CString strName((LPTSTR)tfi.name, 16);
				CString strNameFormat;
				strNameFormat.Format (IDS_TAPEMNGR_NAME_FORMAT, strName, tfi.address, tfi.length);

				OutString (strNameFormat);

				// Open file For capture
				m_strCaptureFileName = GetTmpFilePath ();
				if (m_captureFile.Open (m_strCaptureFileName, CFile::modeCreate|CFile::modeWrite))
				{
					m_bCaptureMode = TRUE;
			
					GetBuffer (m_captureBuffer);
					WriteBuffer (m_headReader.GetWaveBuffer (), nLength);
					ClearBuffers ();
					//m_headReader.AllocWaveBuffer (nLength);
				}
			}
		}
		else
		{
			// Start capture wave data
			GetBuffer (m_captureBuffer);
			WriteBuffer (m_captureBuffer, GetBufferLength ());
		}
	}

	return TRUE;
}



/////////////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::DebugDraw ()
{
//#ifdef _DEBUG
	CWnd* pMonitor = GetDlgItem (IDC_TM_MONITOR);

	if (!pMonitor)
		return;

	CClientDC dc(this);
	
	CRect rcMonitor;
	pMonitor->GetWindowRect (&rcMonitor);
	ScreenToClient (&rcMonitor);
	rcMonitor.DeflateRect (1, 1);

	int x_offs = rcMonitor.left;
	int y_offs = rcMonitor.top;
	int width = rcMonitor.Width ();
	int height = rcMonitor.Height ();
	int height_half = height / 2;
	int nBufferLength = BUFFER_FREQUENCY * CAPTURE_BUFFERS_NUM / 2;
	BYTE pWave[BUFFER_FREQUENCY * CAPTURE_BUFFERS_NUM / 2];

	GetAllBuffers (pWave);

	dc.PatBlt (x_offs, y_offs, width, height, WHITENESS);

	for (int x = 0; x < width; x++)
	{
		BYTE sample = pWave[x * nBufferLength / width];
		int y = y_offs + height_half - height_half * (sample - 127)  / 128;

		dc.MoveTo (x + x_offs, y);
		dc.LineTo (x + x_offs, y + height * (sample - 127) / 128);
	}
//#endif
}



/////////////////////////////////////////////////////////////////////////////
CString CTapeManagerDlg::GetTmpFilePath ()
{
	/*
	TCHAR pTmpPath[_MAX_PATH];
	::GetTempPath (_MAX_PATH, pTmpPath);
	*/

	TCHAR pTmpFilePath[_MAX_PATH];
	//::GetTempFileName (pTmpPath, _T("tap"), 0, pTmpFilePath);

	::GetTempFileName (GetCurrentPath (), _T("tap"), 0, pTmpFilePath);
	
	return CString (pTmpFilePath);
}



/////////////////////////////////////////////////////////////////////////////
BOOL CTapeManagerDlg::WriteBuffer (BYTE* pBuff, DWORD nSize)
{
	if (m_captureFile.m_hFile == CFile::hFileNull)
		return FALSE;

	m_captureLength += nSize;

	if (m_captureLength < m_captureOffset)
		return TRUE;

	DWORD nOffs = max (0, (int)(nSize - (m_captureLength - m_captureOffset)));
	DWORD nSizeToWrite = min (nSize, m_captureLength - m_captureOffset);

	m_captureFile.Write (pBuff + nOffs, nSizeToWrite);

	if (IsCaptureEnd (pBuff + nOffs, nSizeToWrite))
	{
		m_captureFile.Close ();
		m_bCaptureMode = FALSE;

		CString strOut;
		strOut.LoadString (IDS_TAPEMNGR_END_OF_FILE);
		OutString (strOut);

		CTapeUnit* pTapeUnit = new CTapeUnit;
		pTapeUnit->SetTmpFile (m_strCaptureFileName);
		InsertNewUnit (pTapeUnit);

		FileListChanged (m_list.GetItemCount ());
		RetrieveInfo ((CTapeUnit*)m_list.GetItemData (m_tapePos));
	}

	return TRUE;
}



/////////////////////////////////////////////////////////////////////////////
BOOL CTapeManagerDlg::IsCaptureEnd (BYTE* pBuff, DWORD nSize)
{
	int nAverageSum = 0;
	int nSilenceLength = 0;
	for (int n = 0; n < nSize; n++)
	{
		int nAverage = 0;

		if (n > 0)
			nAverage = nAverageSum / n;

		char sample = pBuff[n] - 128;
		if (sample - nAverage <= SILENCE_THRESHOLD && sample - nAverage >= -SILENCE_THRESHOLD)
			nSilenceLength++;
		else
			nSilenceLength = 0;

		nAverageSum += sample;
	}

	//TRACE ("\n nAverage = %i", nAverageSum / nSize);
	//TRACE ("\n Silence length = %i", nSilenceLength);

	if (nSilenceLength >= SILENCE_LENGTH)
		return TRUE;

	return FALSE;
}



/////////////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::InitListCtrl ()
{
	CString strColumn;

	m_list.SetExtendedStyle (LVS_EX_FULLROWSELECT);

	strColumn.LoadString (IDS_TAPEMNGR_TYPE);
	m_list.InsertColumn (COLUMN_TYPE, strColumn, LVCFMT_LEFT, 20);
	strColumn.LoadString (IDS_TAPEMNGR_NAME);
	m_list.InsertColumn (COLUMN_NAME, strColumn, LVCFMT_LEFT, 100);
	strColumn.LoadString (IDS_TAPEMNGR_ADDR);
	m_list.InsertColumn (COLUMN_ADDR, strColumn, LVCFMT_RIGHT, 60);
	strColumn.LoadString (IDS_TAPEMNGR_LENGTH);
	m_list.InsertColumn (COLUMN_LEN, strColumn, LVCFMT_RIGHT, 60);
	strColumn.LoadString (IDS_TAPEMNGR_TIME);
	m_list.InsertColumn (COLUMN_TIME, strColumn, LVCFMT_RIGHT, 40);
	strColumn.LoadString (IDS_TAPEMNGR_WAVELENGTH);
	m_list.InsertColumn (COLUMN_WAVELEN, strColumn, LVCFMT_RIGHT, 80);
	strColumn.LoadString (IDS_TAPEMNGR_CRC);
	m_list.InsertColumn (COLUMN_CRC, strColumn, LVCFMT_LEFT, 40);
	strColumn.LoadString (IDS_TAPEMNGR_PATH);
	m_list.InsertColumn (COLUMN_PATH, strColumn, LVCFMT_LEFT, 300);
	//m_list.SetColumnWidth (COLUMN_PATH, LVSCW_AUTOSIZE_USEHEADER);

	m_imgList.Create (16, 16, ILC_COLOR, 3, 1);
	m_imgList.Add (::LoadIcon (AfxGetInstanceHandle (), MAKEINTRESOURCE (IDR_MAINFRAME)));
	m_imgList.Add (::LoadIcon (AfxGetInstanceHandle (), MAKEINTRESOURCE (IDI_WAVE)));
	m_imgList.Add (::LoadIcon (AfxGetInstanceHandle (), MAKEINTRESOURCE (IDI_TAPE)));
	m_imgList.Add (::LoadIcon (AfxGetInstanceHandle (), MAKEINTRESOURCE (IDI_TAPE)));

	m_list.SetImageList (&m_imgList, LVSIL_SMALL);
}



/////////////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::InitComboCtrl ()
{
	CString strItem;

	CComboBox* pCombo = (CComboBox*)GetDlgItem (IDC_TM_SAVE_AS);

	strItem.LoadString (IDS_TAPEMNGR_SAVE_AS);
	pCombo->InsertString (0, strItem);
	pCombo->SetItemData (0, COMBO_NONE); 
	strItem.LoadString (IDS_FILEFILTER_TAP_ONLY);
	strItem = strItem.Left (strItem.Find (_T('|')));
	pCombo->InsertString (1, strItem);
	pCombo->SetItemData (1, COMBO_SAVE_AS_MSF);
	strItem.LoadString (IDS_FILEFILTER_WAV);
	strItem = strItem.Left (strItem.Find (_T('|')));
	pCombo->InsertString (2, strItem);
	pCombo->SetItemData (2, COMBO_SAVE_AS_WAV);
	strItem.LoadString (IDS_FILEFILTER_BIN);
	strItem = strItem.Left (strItem.Find (_T('|')));
	pCombo->InsertString (3, strItem);
	pCombo->SetItemData (3, COMBO_SAVE_AS_BIN);

	CSize   sz;
	int     width = 0;
	CDC*    pDC = pCombo->GetDC ();
	for (int i = 0;i < pCombo->GetCount (); i++)
	{
		pCombo->GetLBText (i, strItem);
		sz = pDC->GetTextExtent (strItem);

		if (sz.cx > width)
			width = sz.cx;
	}

	width += ::GetSystemMetrics(SM_CXVSCROLL) + 2 * ::GetSystemMetrics(SM_CXEDGE);

	pCombo->SetDroppedWidth (width);
	pCombo->ReleaseDC (pDC);
	pCombo->SetCurSel (0);
}



/////////////////////////////////////////////////////////////////////////////
int CTapeManagerDlg::StatusThreadWrapper (CTapeManagerDlg* pThis)
{
	pThis->StatusThread ();

	return 0;
}



/////////////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::StatusThread ()
{
	m_hStatusExit = ::CreateEvent (NULL, FALSE, FALSE, NULL);
	m_hStatusNew = ::CreateEvent (NULL, FALSE, FALSE, NULL);

	HANDLE pEvents[] = {m_hStatusExit, m_hStatusNew};
	int nEvents = sizeof (pEvents) / sizeof (pEvents[0]);

	while (1)
	{
		int object = ::WaitForMultipleObjects (nEvents, pEvents, FALSE, INFINITE);

		// If exit event
		if (object == WAIT_OBJECT_0)
			break;

		BOOL bSkeep = FALSE;

		// New tape unit set
		TRACE ("\nTread CS IN");
		CSingleLock lock (&m_csModifyStatus, TRUE);
		CTapeUnit tapeUnit = *m_pTapeUnit;
	
		if (tapeUnit.GetWaveLength () != -1)
			bSkeep = TRUE;
		
		lock.Unlock ();
		TRACE ("\nTread CS OUT");

		// Retrieve information of it
		if (!bSkeep)
			tapeUnit.RetrieveInfo ();

		TRACE ("\nTread CS IN");
		lock.Lock ();
		if (m_pTapeUnit) // NULL means that unit list changed
		{
			*m_pTapeUnit = tapeUnit; // Move info back to unit
			lock.Unlock ();

			TRACE ("\nTread PostMessage IN");
			PostMessage (WM_INFO_READY, (WPARAM)bSkeep);
			TRACE ("\nTread PostMessage OUT");
		}
		else
			lock.Unlock ();
		TRACE ("\nTread CS OUT");
		
	}
}



/////////////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::RetrieveInfo (CTapeUnit* pUnit)
{
	CSingleLock lock (&m_csModifyStatus, TRUE);

	// Set new unit for porocess
	m_pTapeUnit = pUnit;

	if (m_pTapeUnit)
		::SetEvent (m_hStatusNew);
}



/////////////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::FileListChanged (int nTapeCount)
{
	RetrieveInfo (NULL);
	m_tapePos = 0;
	m_tapeCount = nTapeCount;
}



/////////////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::InsertNewUnit (CTapeUnit* pUnit)
{
	int nItem = m_list.GetItemCount ();

	for (int n = COLUMN_TYPE; n <= COLUMN_PATH; n++)
	{
		LVITEM lvi;
		memset (&lvi, 0, sizeof (LVITEM));
		lvi.mask = LVIF_IMAGE|LVIF_TEXT|LVIF_PARAM;
		lvi.iItem = nItem;
		lvi.iSubItem = n;
		lvi.pszText = LPSTR_TEXTCALLBACK;
		lvi.iImage = pUnit->GetType ();
		lvi.lParam = (LPARAM)pUnit;

		m_list.InsertItem (&lvi);
	}
}



/////////////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::OnTimer(UINT nIDEvent) 
{
	switch (nIDEvent)
	{
	case CTRL_UPDATE_TIMER:
		UpdateDialogControls (GetParent (), FALSE);
	}
	
	CBKDialog::OnTimer(nIDEvent);
}



/////////////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::OnTmLoad() 
{
	CString strFilterManager;
	strFilterManager.LoadString (IDS_FILEFILTER_TAPEMNGR_LOAD);

	CBKFileDialog dlg (TRUE, NULL, NULL, OFN_ALLOWMULTISELECT|OFN_HIDEREADONLY|OFN_OVERWRITEPROMPT, strFilterManager);
	if (dlg.DoModal () != IDOK)
		return;

	// Get files path
	BOOL bSingleFile = TRUE;
	int pos = 0;
	CString strPath = dlg.m_ofn.lpstrFile + pos;

	pos += strPath.GetLength () + 1;

	CStringList strFiles;
	while (dlg.m_ofn.lpstrFile[pos] != _T('\0'))
	{
		if (bSingleFile)
		{
			bSingleFile = FALSE;
			NormalizePath (strPath);
		}

		// Get next file name
		CString strFilePath = dlg.m_ofn.lpstrFile + pos;
		pos += strFilePath.GetLength () + 1;

		// Create tape unit from file
		CTapeUnit* pTapeUnit = new CTapeUnit;
		if (pTapeUnit->SetFile (strPath + strFilePath))
			InsertNewUnit (pTapeUnit);
	}

	if (bSingleFile)
	{
		// If only one file selected
		CTapeUnit* pTapeUnit = new CTapeUnit;
		if (pTapeUnit->SetFile (strPath))
			InsertNewUnit (pTapeUnit);
	}

	FileListChanged (m_list.GetItemCount ());
	RetrieveInfo ((CTapeUnit*)m_list.GetItemData (m_tapePos));
}



/////////////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::OnTmBrowse() 
{
	/*
	CComPtr <IShellFolder> pShellFolder;
	CComBSTR root(m_strCurrentFolder);

	ITEMIDLIST* pList;

	::SHGetDesktopFolder (&pShellFolder);
	pShellFolder->ParseDisplayName (NULL, NULL, root
		, NULL, &pList, 0);
	*/
    
	CBrowseForFolder browse (this->m_hWnd, NULL, m_strCurrentFolder,  _T("Select folder to copy"));
	if (!browse.SelectFolder ())
		return;

	SetFolder (browse.GetSelectedFolder ());
}



/////////////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::OnCancel() 
{
	if (m_pCaptureBuffer)
	{
		m_pCaptureBuffer->Stop ();

		// Waif for event thread finish
		::SetEvent (m_pNotifyEvents[0]);
		::WaitForSingleObject (*m_pNotifyThread, INFINITE);
		delete m_pNotifyThread;
		
		// Release Direct Sound objects
		m_pNotify.Release ();

		for (int i = 0; i < EVENTS_NUM + 1; i++) // + 1 Exit event
			::CloseHandle (m_pNotifyEvents[i]);

		::CloseHandle (m_hBufferReadyEvent);
		m_hBufferReadyEvent = NULL;

		delete []m_pNotifyPositions;
		delete []m_pNotifyEvents;

		m_pCaptureBuffer.Release ();
		m_pDirectSoundCapture.Release ();
	}

	RemoveBuffers ();

	// Wait while status thread finishing
	::SetEvent (m_hStatusExit);
	::WaitForSingleObject (*m_pStatusThread, INFINITE);
	delete m_pStatusThread;	

	DeleteUnits (FALSE);
	
	CBKDialog::OnCancel();
}



/////////////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::OnUpdateTmRemove (CCmdUI* pCmdUI)
{
	pCmdUI->Enable (m_list.GetSelectedCount ());
}



/////////////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::OnTmRemove() 
{
	DeleteUnits (TRUE);
}




/////////////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::OnUpdateSelendokTmSaveAs (CCmdUI* pCmdUI)
{
	pCmdUI->Enable (m_list.GetSelectedCount ());
}



/////////////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::OnSelendokTmSaveAs() 
{
	CComboBox* pCombo = (CComboBox*)GetDlgItem (IDC_TM_SAVE_AS);

	int nItem = pCombo->GetCurSel ();
	int type = CTapeUnit::TYPE_NONE;
	
	if (nItem >= 0)
	{
		int nSaveAs = pCombo->GetItemData (nItem);

		switch (nSaveAs)
		{
		case COMBO_SAVE_AS_BIN:
			type = CTapeUnit::TYPE_BIN;
			break;

		case COMBO_SAVE_AS_WAV:
			type = CTapeUnit::TYPE_WAV;
			break;

		case COMBO_SAVE_AS_MSF:
			type = CTapeUnit::TYPE_MSF;
			break;
		}
	}

	pCombo->SetCurSel (0);
	pCombo->ShowDropDown (FALSE);

	SaveUnitsAs (type);
}



/////////////////////////////////////////////////////////////////////////////
HRESULT CTapeManagerDlg::OnInfoReady (WPARAM wParam, LPARAM lParam)
{
	BOOL bSkeep = (BOOL)wParam;

	if (!bSkeep)
	{
		m_list.Invalidate (FALSE);

		if (!m_pTapeUnit->GetCRC ())
		{
			CString strError;
			strError.Format (IDS_TAPEMNGR_CRC_ERROR, m_pTapeUnit->GetName ());
			OutString (strError);
		}
	}
		
	if (++m_tapePos < m_tapeCount)
	{
		TRACE ("\nRetriveInfo CS IN");
		RetrieveInfo ((CTapeUnit*)m_list.GetItemData (m_tapePos));
		TRACE ("\nRetriveInfo CS OUT");
	}
	
	return 0;
}



/////////////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::OnGetdispinfoTmList(NMHDR* pNMHDR, LRESULT* pResult) 
{
	LV_DISPINFO* pDispInfo = (LV_DISPINFO*)pNMHDR;
	
	if (pDispInfo->item.mask & LVIF_TEXT)
	{
		//pDispInfo->item.mask |= LVIF_DI_SETITEM;
		CTapeUnit* pTapeUnit = (CTapeUnit*)pDispInfo->item.lParam;

		switch (pDispInfo->item.iSubItem)
		{
		case COLUMN_NAME:
			{
				CString strName = pTapeUnit->GetName ();

				if (strName.IsEmpty ())
					strName.LoadString (IDS_TAPEMNGR_UNKNOWN);

				memcpy (pDispInfo->item.pszText, strName, strName.GetLength () + 1);
			}
			break;

		case COLUMN_ADDR:
			{
				CString strAddress = WordToOctString (pTapeUnit->GetAddress ());

				if (pTapeUnit->GetAddress () == -1)
					strAddress.LoadString (IDS_TAPEMNGR_UNKNOWN);

				memcpy (pDispInfo->item.pszText, strAddress, strAddress.GetLength () + 1);
			}
			break;

		case COLUMN_LEN:
			{
				CString strLength = WordToOctString (pTapeUnit->GetLength ());

				if (pTapeUnit->GetLength () == -1)
					strLength.LoadString (IDS_TAPEMNGR_UNKNOWN);

				memcpy (pDispInfo->item.pszText, strLength, strLength.GetLength () + 1);
			}
			break;

		case COLUMN_TIME:
			{
				CString strTime = MsTimeToTimeString (pTapeUnit->GetTime ());

				if (pTapeUnit->GetTime () == -1)
					strTime.LoadString (IDS_TAPEMNGR_UNKNOWN);

				memcpy (pDispInfo->item.pszText, strTime, strTime.GetLength () + 1);
			}
			break;

		case COLUMN_WAVELEN:
			{
				CString strLength = IntToFileLengthString (pTapeUnit->GetWaveLength ());

				if (pTapeUnit->GetWaveLength () == -1)
					strLength.LoadString (IDS_TAPEMNGR_UNKNOWN);

				memcpy (pDispInfo->item.pszText, strLength, strLength.GetLength () + 1);
			}
			break;

		case COLUMN_CRC:
			{
				BOOL bCRC = pTapeUnit->GetCRC ();

				CString strCRC;
				if (bCRC)
					strCRC.LoadString (IDS_TAPEMNGR_CRC_OK);
				else
					strCRC.LoadString (IDS_TAPEMNGR_CRC_FAIL);

				memcpy (pDispInfo->item.pszText, strCRC, strCRC.GetLength () + 1);
			}
			break;

		case COLUMN_PATH:
			{
				CString strPath = pTapeUnit->GetPath ();
				memcpy (pDispInfo->item.pszText, strPath, strPath.GetLength () + 1);
			}
			break;

		}
	}
	
	*pResult = 0;
}



/////////////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::SetFolder (CString strFolder)
{
	m_strCurrentFolder = strFolder;
	NormalizePath (m_strCurrentFolder);

	SetDlgItemText (IDC_TM_FOLDER, m_strCurrentFolder);
}



/////////////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::DeleteUnits (BOOL bSelected)
{
	CArray<int, int> items; 

	if (bSelected)
	{
		POSITION pos = m_list.GetFirstSelectedItemPosition ();

		while (pos)
			items.Add (m_list.GetNextSelectedItem (pos));
	}
	else
	{
		int nItems = m_list.GetItemCount ();
		for (int n = 0; n < nItems; n++)
			items.Add (n);
	}

	for (int n = items.GetSize () - 1; n >= 0; n--)
	{
		int nItem = items[n];
		delete [](CTapeUnit*)m_list.GetItemData (nItem);
		m_list.DeleteItem (nItem);
	}
}



/////////////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::SaveUnitsAs (int type)
{
	if (type == CTapeUnit::TYPE_NONE
		//|| type == CTapeUnit::TYPE_WAV
		|| type == CTapeUnit::TYPE_TMP)
		return;

	CString strSaveMessage;
	strSaveMessage.LoadString (IDS_TAPEMNGR_SAVE_MESSAGE);
	OutString (strSaveMessage);

	POSITION pos = m_list.GetFirstSelectedItemPosition ();

	while (pos)
	{
		int nItem = m_list.GetNextSelectedItem (pos);
		CTapeUnit* pTapeUnit = (CTapeUnit*)m_list.GetItemData (nItem);

		if (pTapeUnit->SaveAs (m_strCurrentFolder, type))
			strSaveMessage.Format (IDS_TAPEMNGR_SAVE_SUCCEED, pTapeUnit->GetName ());
		else
			strSaveMessage.Format (IDS_TAPEMNGR_SAVE_ERROR, pTapeUnit->GetName ());

		OutString (strSaveMessage);
	}
}



/////////////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::OnRclickTmList(NMHDR* pNMHDR, LRESULT* pResult) 
{
	CMenu menu;
	menu.LoadMenu (IDR_TAPEMANAGER_MENU);

	CBKMenu menuBK;
	menuBK.Customize (&menu);
	CMenu* pSubMenu = menuBK.GetSubMenu (0);
	
	CPoint point;
	GetCursorPos (&point);

	pSubMenu->TrackPopupMenu (TPM_RIGHTBUTTON, point.x, point.y, this);
	
	*pResult = 0;
}



/////////////////////////////////////////////////////////////////////////////
void CTapeManagerDlg::OnTmSelectall() 
{
	int nCount = m_list.GetItemCount ();

	for (int n = 0; n < nCount; n++)
		m_list.SetItemState (n, LVIS_SELECTED, LVIS_SELECTED);
	
}
