/*************************************************
*            BK Emulator for Window 3.0          *
*************************************************/

/*
File: Tape.cpp
Version: 1.0.0
Written by: Yuriy Kalmykov <kalmykov@stoik.com>
    Copyright (c) 2002-2004 Yuriy Kalmykov

    BK Emulator is a program emulated hardware environment for running
code for BK 0010(01) in different configurations. 
           
    This code may be used in compiled form in any way you desire.
This file or it's parts can't be redistributed without the authors
written consent, but can be modified for your private needs.
    Providing that this notice and the authors name and all copyright
notices remains intact.

    Please, an email me to know how you are using it and where. You can
ask me for any information about this below code or any attendant
knowledge.
    
    This file is provided "as is" with no expressed or implied warranty.
The author accepts no liability for any damage or loss of business that
this product may cause.
*/




// Tape.cpp: implementation of the CTape class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "resource.h"
#include "Globals.h"
#include "Tape.h"
#include "MSF.h"
#include "MSFManager.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CTape::CTape()
{
	m_bPlay = FALSE;
	m_pWave = NULL;
	m_pPackBits = NULL;
	m_pPackLZW = NULL;
	m_waveLength = 0;

	m_bRecord = FALSE;
	m_pRecord = NULL;
	m_bAutoBeginRecord = TRUE;
	m_bAutoEndRecord = TRUE;

	m_pBin = NULL;

	// Initialize table pointers
	for (int y = 0; y < MAX_TABLE_SIZE; y++)
		m_scan_table[y] = NULL;
}

CTape::~CTape()
{
	delete []m_pWave;
	delete []m_pPackBits;
	delete []m_pPackLZW;

	delete []m_pBin;

	ClearTables ();
}



//////////////////////////////////////////////////////////////////////
BOOL CTape::LoadWaveFile (CString strPath)
{
	CFile waveFile;
	if (!waveFile.Open (strPath, CFile::modeRead))
		return FALSE;

	struct WaveHeader
	{
		long				riffTag;
		long				size;
		long				waveTag;
		long				fmtTag;
		long				fmtSize;
	} waveHeader;
	
	if (waveFile.Read (&waveHeader, sizeof (WaveHeader)) != sizeof (WaveHeader))
		return FALSE;

	if (waveHeader.riffTag != 0x46464952 || waveHeader.fmtTag != 0x20746d66)
		return FALSE;

	WAVEFORMATEX wfx;
	if (waveFile.Read (&wfx, waveHeader.fmtSize) != waveHeader.fmtSize)
		return FALSE;

	wfx.cbSize = 0;

	if (wfx.wFormatTag != WAVE_FORMAT_PCM)
		return FALSE;

	struct DataHeader
	{
		long				dataTag;
		long				dataSize;
	} dataHeader;
	
	while (1)
	{
		if (waveFile.Read (&dataHeader, sizeof (DataHeader)) != sizeof (DataHeader))
			return FALSE;

		if (dataHeader.dataTag == 0x61746164)
			break;

		waveFile.Seek (dataHeader.dataSize, SEEK_CUR);
	}
	
	AllocWaveBuffer (dataHeader.dataSize);
	waveFile.Read (m_pWave, dataHeader.dataSize);
	waveFile.Close ();

	//m_waveLength = dataHeader.dataSize;

	// Calculate average
	m_average = 0;
	for (int s = 0; s < m_waveLength; s++)
	{
		m_average += m_pWave[s];
	}

	m_average /= m_waveLength;
	m_average = 128;
	//m_average -= 5;

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CTape::LoadTmpFile (CString strPath)
{
	CFileStatus fs;
	if (!CFile::GetStatus (strPath, fs))
		return FALSE;

	CFile tmpFile;
	if (!tmpFile.Open (strPath, CFile::modeRead))
		return FALSE;

	AllocWaveBuffer (fs.m_size);
	tmpFile.Read (m_pWave, fs.m_size);
	tmpFile.Close ();

	// Calculate average
	m_average = 0;
	for (int s = 0; s < m_waveLength; s++)
	{
		m_average += m_pWave[s];
	}

	m_average /= m_waveLength;
	m_average = 128;

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CTape::SaveWaveFile (CString strPath)
{
	CFile waveFile;
	if (!waveFile.Open (strPath, CFile::modeCreate|CFile::modeReadWrite))
		return FALSE;

	struct WaveHeader
	{
		long				riffTag;
		long				size;
		long				waveTag;
		long				fmtTag;
		long				fmtSize;
	} waveHeader;

	struct DataHeader
	{
		long				dataTag;
		long				dataSize;
	} dataHeader;

	waveHeader.riffTag = 0x46464952;
	waveHeader.size = sizeof (WaveHeader) + sizeof(WAVEFORMATEX) + sizeof (DataHeader) + m_waveLength;
	waveHeader.waveTag = 0x45564157;
	waveHeader.fmtTag = 0x20746d66;
	waveHeader.fmtSize = 18;
	
	waveFile.Write (&waveHeader, sizeof (WaveHeader));

	WAVEFORMATEX wfx;
	wfx.wFormatTag = WAVE_FORMAT_PCM;
	wfx.nSamplesPerSec = 44100;
	wfx.nChannels = 1;
	wfx.wBitsPerSample = 8;
	wfx.nBlockAlign = (wfx.wBitsPerSample >> 3) * wfx.nChannels;
	wfx.nAvgBytesPerSec = wfx.nSamplesPerSec * wfx.nBlockAlign;
	wfx.cbSize = 0;

	waveFile.Write (&wfx, waveHeader.fmtSize);

	dataHeader.dataTag = 0x61746164;
	dataHeader.dataSize = m_waveLength;

	waveFile.Write (&dataHeader, sizeof (DataHeader));

	waveFile.Write (m_pWave, m_waveLength);

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
DWORD CTape::PackBits ()
{
	int pos = 0; // Current byte position
	BYTE mask = 1; // Cyclic bit mask
	memset (m_pPackBits, 0, m_bitsLength);

	for (int i = 0; i < m_waveLength; i++)
	{
		// Get current byte
		if (m_pWave[i] > m_average) // If positive set current bit 1 else pass bit
			m_pPackBits[pos] = m_pPackBits[pos] | mask;

		mask <<= 1; //Scroll mask

		if (!mask) 
		{
			// If pack byte is done reset mask and go to next byte
			mask = 1;
			pos++;
		}
	}

	return m_bitsLength; // Return compressed length
}



//////////////////////////////////////////////////////////////////////
void CTape::UnpackBits ()
{
	// Allocate memory for unpacket data
	AllocWaveBuffer (m_bitsLength * 8);
	m_waveLength = 0;
	
	for (int i = 0; i < m_bitsLength; i++)
	{
		BYTE mask = 1;

		// Unpack every byte to eight bytes
		for (int b = 0; b < 8; b++)
		{
			if (m_pPackBits[i] & mask)
				m_pWave[m_waveLength++] = 255; // If bit 1 unpack as 255
			else
				m_pWave[m_waveLength++] = 0; // If bit 0 unpack as 0

			mask <<= 1; //Scroll mask
		}
	}
}



//////////////////////////////////////////////////////////////////////
DWORD CTape::PackLZW_Fast ()
{
	/*
	m_pPackBits[0] = 'A';
	m_pPackBits[1] = 'B';
	m_pPackBits[2] = 'C';
	m_pPackBits[3] = 'A';
	m_pPackBits[4] = 'B';
	m_pPackBits[5] = 'C';
	m_pPackBits[6] = 'A';
	m_pPackBits[7] = 'B';
	m_pPackBits[8] = 'C';
	m_pPackBits[9] = 'A';
	m_pPackBits[10] = 'B';
	m_pPackBits[11] = 'C';
	m_pPackBits[12] = 'A';
	m_pPackBits[13] = 'B';
	m_pPackBits[14] = 'C';
	m_pPackBits[15] = 'A';
	m_pPackBits[16] = 'B';
	m_pPackBits[17] = 'C';
	m_pPackBits[18] = 'A';
	m_pPackBits[19] = 'B';
	m_pPackBits[20] = 'C';
	m_pPackBits[21] = 'A';
	m_pPackBits[22] = 'B';
	m_pPackBits[23] = 'C';
	m_pPackBits[24] = 'A';
	m_pPackBits[25] = 'B';
	m_pPackBits[26] = 'C';
	m_pPackBits[27] = 'A';
	m_pPackBits[28] = 'B';
	m_pPackBits[29] = 'C';
	*/
	
	// Initialize table
	InitTables ();
	
	m_lzwLength = 0;
	WORD last_code = BAD_CODE;

	for (int read_pos = 0; read_pos < m_bitsLength; read_pos++)
	{
		int code = TableLookup_Fast (last_code, read_pos); // Get current sequence

		if (code == BAD_CODE || read_pos == (m_bitsLength - 1))
		{	
			// If not in table or if is the end
			m_pPackLZW[m_lzwLength++] = last_code;  // Pack is: sequence - 1 byte

			AddWord_Fast (last_code, read_pos); // Add: sequence to table

			last_code = m_pPackBits[read_pos];
		}
		else
			last_code = code;
	}

	m_pPackLZW[m_lzwLength++] = m_pPackBits[read_pos - 1]; // Pack end byte as is

	return m_lzwLength; // Return compressed length
}



//////////////////////////////////////////////////////////////////////
void CTape::UnpackLZW_Fast ()
{
	// Initialize table
	InitTables ();

	memset (m_pPackBits, 0, m_bitsLength);
	m_bitsLength = 0;
	int	unpack_pos = 0;
	
	/*
	m_pPackLZW[0] = 'A';
	m_pPackLZW[1] = 'B';
	m_pPackLZW[2] = 'C';
	m_pPackLZW[3] = 257;
	m_pPackLZW[4] = 259;
	m_pPackLZW[5] = 258;
	m_pPackLZW[6] = 260;
	m_pPackLZW[7] = 263;
	m_pPackLZW[8] = 262;
	m_pPackLZW[9] = 265;
	m_pPackLZW[10] = 261;
	m_pPackLZW[11] = 267;
	*/
	
	WORD last_code = BAD_CODE;

	for (int read_pos = 0; read_pos < m_lzwLength; read_pos++)
	{
		if (m_pPackLZW[read_pos] < 256) // If current code < 256 unpack it as is
			m_pPackBits[m_bitsLength++] = m_pPackLZW[read_pos];
		else
		{
			// Else create table from current position to this code position
			while (unpack_pos < m_bitsLength)
			{
				int code = TableLookup_Fast (last_code, unpack_pos);
				
				if (code == BAD_CODE)
				{	
					// If not current sequence in table add it
					AddWord_Fast (last_code, unpack_pos);
					last_code = m_pPackBits[unpack_pos];
				}
				else
					last_code = code;

				unpack_pos++;
			}

			if (!UnpackWord (m_pPackLZW[read_pos])) // Try to unpack current code
			{
				// If not in table (case XandX)
				int first_pos = m_bitsLength;
				UnpackWord (last_code); // Unpack last known code

				m_pPackBits[m_bitsLength++] = TableLookup_Fast (BAD_CODE, first_pos); // Unpack last known code first byte
				
				AddWord_Fast (last_code, unpack_pos); // Add: Xsequence + X to table
				last_code = BAD_CODE;
			}
		}
	}
}



//////////////////////////////////////////////////////////////////////
void CTape::InitTables ()
{
	for (int y = 0; y < MAX_TABLE_SIZE; y++)
	{
		if (!m_scan_table[y])
			m_scan_table[y] = new WORD[258];

		for (int x = 0; x < 258; x++)
			m_scan_table[y][x] = BAD_CODE;
	}

	for (y = 0; y < 256; y++)
		m_scan_table[y][CODE] = y;

	m_table_size = 257;
}



//////////////////////////////////////////////////////////////////////
void CTape::ClearTables ()
{
	for (int y = 0; y < MAX_TABLE_SIZE; y++)
	{
		delete []m_scan_table[y];
		m_scan_table[y] = NULL;
	}
}



//////////////////////////////////////////////////////////////////////
int	CTape::TableLookup_Fast (WORD prefix, int end_pos)
{
	if (prefix == BAD_CODE)
		return m_pPackBits[end_pos];

	return m_scan_table[prefix][m_pPackBits[end_pos]];
}



//////////////////////////////////////////////////////////////////////
void CTape::AddWord_Fast (WORD prefix, int end_pos)
{
	if (m_table_size >= MAX_TABLE_SIZE)
	{
		InitTables ();
		return;
	}

	m_scan_table[prefix][m_pPackBits[end_pos]] = m_table_size;
	m_scan_table[m_table_size][PREFIX] = prefix;
	m_scan_table[m_table_size][CODE] = m_pPackBits[end_pos];
	m_table_size++;
}



//////////////////////////////////////////////////////////////////////
BOOL CTape::UnpackWord (int code)
{
	if (code == BAD_CODE)
		return FALSE;

	if (code < 256)
	{
		m_pPackBits[m_bitsLength++] = code;
		return TRUE;
	}

	if (!UnpackWord (m_scan_table[code][PREFIX]))
		return FALSE;

	m_pPackBits[m_bitsLength++] = m_scan_table[code][CODE];

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CTape::GetWaveFile (TAPE_FILE_INFO* pTfi, BOOL bHeaderOnly)
{
	if (!m_pWave)
		return FALSE;

	m_pos = 0;
	m_last_len = 0;

	memset (pTfi, 255, sizeof (TAPE_FILE_INFO));

	int length = 0;
	if (FindTuning (1024, pTfi->start_tuning, length) == BAD_LENGTH)
		return FALSE;

	pTfi->synchro_start = m_pos;

	if (!FindSyncro ())
		return FALSE;

	if (FindTuning (7, pTfi->marker1, length) == BAD_LENGTH)
		return FALSE;

	pTfi->synchro_header = m_pos;

	if (!FindSyncro ())
		return FALSE;

	pTfi->header = m_pos;

	for (int i = 0; i < 20; i++)
	{
		if (!ReadByte (((BYTE*)&pTfi->address)[i]))
			return FALSE;
	}

	if (FindTuning (7, pTfi->marker2, length) == BAD_LENGTH)
		return FALSE;

	pTfi->synchro_data = m_pos;

	if (!FindSyncro ())
		return FALSE;

	if (bHeaderOnly)
		return TRUE;

	delete []m_pBin;
	m_pBin = new BYTE[pTfi->length];

	pTfi->data = m_pos;

	for (i = 0; i < pTfi->length; i++)
	{
		if (!ReadByte (m_pBin[i]))
			return FALSE;
	}

	if (!ReadByte (((BYTE*)&pTfi->crc)[0]))
		return FALSE;

	if (!ReadByte (((BYTE*)&pTfi->crc)[1]))
		return FALSE;

	pTfi->synchro6 = m_pos;

	if (!FindSyncro6 ())
		return FALSE;

	FindTuning (128, pTfi->end_tuning, length);

	pTfi->end = pTfi->end_tuning + length;
	
	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CTape::FindTuning (int length, DWORD& wave_pos, int& wave_length)
{	
	DWORD syncro_pos = BAD_LENGTH;

	DWORD last_pos = m_pos;
	
	int imp_num = 0;
	m_avg_length = 0;

	DWORD begin_pos = m_pos;
	BOOL bFound = FALSE;

	while (1)
	{
		last_pos = m_pos;

		int len = CalcImpLength (m_pWave, m_pos, m_waveLength);

		if (len == BAD_LENGTH)
		{
			m_pos = last_pos;
			return FALSE;
		}

		if ((m_last_len - len) * (m_last_len - len) <= 3 * 3)
		{
			m_avg_length += len;
			imp_num++;

			if (imp_num >= length && !bFound)
			{
				syncro_pos = begin_pos;
				bFound = TRUE;
			}
		}
		else if (!bFound)
		{
			m_avg_length = 0;
			imp_num = 0;
			begin_pos = m_pos;
		}
		else
			break;
		
		m_last_len = len;

		//TRACE ("\nWave pos %06i, Start = %i, Len = %i, Imp_Num = %i", m_pos, begin_pos, len, imp_num);
		//Sleep (1);
	}

	m_pos = last_pos;
	
	wave_pos = syncro_pos;
	wave_length = m_pos - syncro_pos;

	m_avg_length = (m_avg_length + imp_num / 2) / imp_num;

	/*
	for (int i = 0; i < 100; i++)
	{
		int len = CalcImpLength (m_pWave, m_pos, m_waveLength);
		TRACE ("\nAfter Wave pos %06i, Start = %i, Len = %i, Imp_Num = %i", m_pos, begin_pos, len, imp_num);
		Sleep (1);
	}
	/*/

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CTape::FindSyncro ()
{
	int size = 0;
	int length =  0;

	while (1)
	{
		length = CalcImpLength (m_pWave, m_pos, m_waveLength);
	
		if (length == BAD_LENGTH)
			return FALSE;
	
		size = DefineLength (length);

		if (size >= 3)
			break;
	}
	
	int nSyncLen = 2;
	while (nSyncLen > 0)
	{
		length = CalcImpLength (m_pWave, m_pos, m_waveLength);
	
		if (length == BAD_LENGTH)
			return FALSE;
	
		size = DefineLength (length);

		if (size == nSyncLen)
			nSyncLen--;
	}

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CTape::FindSyncro6 ()
{
	int length = CalcImpLength (m_pWave, m_pos, m_waveLength);
	
	if (length == BAD_LENGTH)
		return FALSE;
	
	int size = DefineLength (length);

	if (size < 6)
		return FALSE;

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CTape::ReadBit (BOOL& bBit)
{
	int length0 = CalcImpLength (m_pWave, m_pos, m_waveLength);
	
	if (length0 == BAD_LENGTH)
		return FALSE;
	
	int size0 = DefineLength (length0);

	int length1 = CalcImpLength (m_pWave, m_pos, m_waveLength);
	
	if (length1 == BAD_LENGTH)
		return FALSE;
	
	int size1 = DefineLength (length1);

	//if (size1 != 1)
	//	return FALSE;

	if (size0 == 1)
		bBit = FALSE;
	else if (size0 == 2)
		bBit = TRUE;
	else
		return FALSE;
	
	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CTape::ReadByte (BYTE& byte)
{
	BYTE mask = 1;
	byte = 0;

	for (int i = 0; i < 8; i++)
	{
		BOOL bBit;

		if (!ReadBit (bBit))
			return FALSE;

		if (bBit)
			byte |= mask;

		mask <<= 1;
	}

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
int CTape::CalcImpLength (BYTE* pWave, DWORD& pos, DWORD length)
{
	if (pos >= length)
		return BAD_LENGTH;

	// Pass inverse impulses
	if (pWave[pos] < m_average)
	{
		while (pWave[pos] < m_average)
		{
			if (pos >= length)
				return BAD_LENGTH;

			pos++;
		}
	}

	int len_1 = 0;

	// Calculate number of 1
	while (pWave[pos] >= m_average)
	{
		if (pos >= length)
			return BAD_LENGTH;

		len_1++;
		pos++;
	}

	int len_0 = 0;

	// Calculate number of 0
	while (pWave[pos] < m_average)
	{
		if (pos >= length)
			return BAD_LENGTH;

		len_0++;
		pos++;
	}

	return len_1 + len_0;
}



//////////////////////////////////////////////////////////////////////
int CTape::DefineLength (int length)
{
	return (length + m_avg_length / 2) / m_avg_length;
}



//////////////////////////////////////////////////////////////////////
BOOL CTape::CalcCRC (TAPE_FILE_INFO* pTfi, BOOL bCheck)
{
	if (!m_pBin)
		return FALSE;

	DWORD crc = 0;

	for (int i = 0; i < pTfi->length; i++)
	{
		crc += m_pBin[i];

		if (crc & 0xFFFF0000)
		{
			crc &= 0x0000FFFF;
			crc++;
		}
	}

	if (bCheck)
	{
		if (crc != pTfi->crc)
			return FALSE;
	}
	else
		pTfi->crc = crc;

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CTape::LoadBinFile (CString strPath, TAPE_FILE_INFO* pTfi)
{
	CFile binFile;

	if (!binFile.Open (strPath, CFile::modeRead))
		return FALSE;

	if (binFile.Read (&pTfi->address, sizeof (WORD)) != sizeof (WORD))
		return FALSE;

	//pTfi->address = 040000;

	if (binFile.Read (&pTfi->length, sizeof (WORD)) != sizeof (WORD))
		return FALSE;

	delete []m_pBin;
	m_pBin = new BYTE[pTfi->length];

	if (binFile.Read (m_pBin, pTfi->length) != pTfi->length)
		return FALSE;

	delete []m_pWave;
	m_pWave = NULL;
	m_pos = 0;

	CString strName = GetFileName (strPath);
	CString strExt = GetFileExt (strPath);
	CString strBinExt;
	strBinExt.LoadString (IDS_FILEEXT_BINARY);

	if (!strExt.CompareNoCase (strBinExt))
		strName = GetFileTitle (strPath);

	int len = strName.GetLength ();

	for (int i = 0; i < 16 - len; i++)
		strName += " ";

	memcpy (pTfi->name, strName, 16);

	if (!SetWaveFile (pTfi))
		return FALSE;

	m_waveLength = m_pos;
	AllocWaveBuffer (m_waveLength);
	m_pos = 0;

	if (!SetWaveFile (pTfi))
		return FALSE;

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CTape::SaveBinFile (CString strPath, TAPE_FILE_INFO* pTfi)
{
	if (!m_pBin)
		return FALSE;

	CFile binFile;

	if (!binFile.Open (strPath, CFile::modeCreate|CFile::modeWrite))
		return FALSE;

	binFile.Write (&pTfi->address, sizeof (WORD));
	binFile.Write (&pTfi->length, sizeof (WORD));
	binFile.Write (m_pBin, pTfi->length);

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CTape::LoadMSFFile (CString strPath)
{
	CMSFManager msf;
	if (!msf.OpenFile (strPath, TRUE))
		return FALSE;

	DWORD nPackBits;
	DWORD nPackLZW;

	if (!msf.GetBlockTape (NULL, &nPackBits, &nPackLZW))
		return FALSE;

	delete []m_pPackBits;
	m_pPackBits = new BYTE[nPackBits];

	m_bitsLength = nPackBits;

	delete []m_pPackLZW;
	m_pPackLZW = new WORD[nPackLZW];

	if (!msf.GetBlockTape ((BYTE*)m_pPackLZW, &nPackBits, &nPackLZW))
		return FALSE;

	m_lzwLength = nPackLZW / 2;

	UnpackLZW_Fast ();
	UnpackBits ();

	// Calculate average
	m_average = 0;
	for (int s = 0; s < m_waveLength; s++)
	{
		m_average += m_pWave[s];
	}

	m_average /= m_waveLength;
	m_average = 128;

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CTape::SaveMSFFile (CString strPath)
{
	m_bitsLength = (m_waveLength + 7) / 8;

	delete []m_pPackBits;
	m_pPackBits = new BYTE[m_bitsLength];

	delete []m_pPackLZW;
	m_pPackLZW = new WORD[m_bitsLength];

	CMSFManager msf;
	if (!msf.OpenFile (strPath, FALSE))
		return FALSE;

	DWORD nPackBits = PackBits ();
	DWORD nPackLZW = PackLZW_Fast () * 2;
	
	if (!msf.SetBlockTape ((BYTE*)m_pPackLZW, nPackBits, nPackLZW))
		return FALSE;

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CTape::SetWaveFile (TAPE_FILE_INFO* pTfi)
{
	m_pos = 0;
	m_avg_length = 17;
	ASSERT ((m_avg_length & 1));

	if (!SaveTuning (4096))
		return FALSE;

	if (!SaveSyncro ())
		return FALSE;

	if (!SaveTuning (8))
		return FALSE;

	if (!SaveSyncro ())
		return FALSE;

	for (int i = 0; i < 20; i++)
	{
		if (!SaveByte (((BYTE*)&pTfi->address)[i]))
			return FALSE;
	}

	if (!SaveTuning (8))
		return FALSE;

	if (!SaveSyncro ())
		return FALSE;

	for (i = 0; i < pTfi->length; i++)
	{
		if (!SaveByte (m_pBin[i]))
			return FALSE;
	}

	CalcCRC (pTfi, FALSE);

	if (!SaveByte (((BYTE*)&pTfi->crc)[0]))
		return FALSE;

	if (!SaveByte (((BYTE*)&pTfi->crc)[1]))
		return FALSE;

	if (!SaveSyncro6 ())
		return FALSE;

	if (!SaveTuning (256))
		return FALSE;

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CTape::SaveTuning (int length)
{
	for (int i = 0; i < length; i++)
	{
		if (!SaveImp (1))
			return FALSE;
	}

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CTape::SaveSyncro ()
{
	if (!SaveImp (3))
		return FALSE;

	if (!SaveImp (2))
		return FALSE;

	if (!SaveImp (1))
		return FALSE;

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CTape::SaveSyncro6 ()
{
	if (!SaveImp (12))
		return FALSE;

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CTape::SaveImp (int size)
{
	if (m_pWave)
	{
		for (int i = 0; i < size * (m_avg_length / 2 + 1); i++)
		{
			if (m_pos >= m_waveLength)
				return FALSE;

			m_pWave[m_pos++] = 255;
		}

		for (i = 0; i < size * (m_avg_length / 2); i++)
		{
			if (m_pos >= m_waveLength)
				return FALSE;

			m_pWave[m_pos++] = 0;
		}
	}
	else
		m_pos += size * m_avg_length;

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CTape::SaveBit (BOOL bBit)
{
	if (bBit)
	{
		if (!SaveImp (2))
			return FALSE;
	}
	else
	{
		if (!SaveImp (1))
			return FALSE;
	}

	if (!SaveImp (1))
		return FALSE;

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CTape::SaveByte (BYTE byte)
{
	BYTE mask = 1;
	
	for (int i = 0; i < 8; i++)
	{
		if (!SaveBit (byte & mask))
			return FALSE;

		mask <<= 1;
	}

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CTape::AllocWaveBuffer (DWORD nSize)
{
	if (m_pWave == NULL || m_waveLength < nSize)
	{
		delete []m_pWave;
		m_pWave = new BYTE[nSize];
		m_waveLength = 0;

		if (!m_pWave)
			return FALSE;

		m_waveLength = nSize;
	}

	memset (m_pWave, 128, m_waveLength);
	m_average = 128;

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CTape::LoadBuffer (BYTE* pBuff, DWORD nSize)
{
	if (!AllocWaveBuffer (nSize))
		return FALSE;

	memcpy (m_pWave, pBuff, nSize);
	m_average = 128;

	return TRUE;
}


//////////////////////////////////////////////////////////////////////
void CTape::SetPlayWavePos (DWORD play_pos)
{
	m_play_pos = play_pos;
}



//////////////////////////////////////////////////////////////////////
void CTape::PlayWaveGetBuffer (BYTE* pBuff, int buffLength)
{
	int length = 0;

	if (m_bPlay && m_pWave)
	{
		if (m_waveLength - m_play_pos < buffLength)
			length = m_waveLength - m_play_pos;
		else
			length = buffLength;

		if (length)
			memcpy (pBuff, m_pWave + m_play_pos, length);
	}
	
	if (length < buffLength)
		memset (pBuff + length, 0, buffLength - length);

	m_play_pos += length;

	if (m_play_pos >= m_waveLength)
		StopPlay ();
}



//////////////////////////////////////////////////////////////////////
DWORD CTape::GetWaveLength ()
{
	return m_waveLength;
}



//////////////////////////////////////////////////////////////////////
DWORD CTape::GetPlayWavePos ()
{
	return m_play_pos;
}



//////////////////////////////////////////////////////////////////////
void CTape::StartPlay ()
{
	m_pos = 0;
	m_bPlay = TRUE;
}



//////////////////////////////////////////////////////////////////////
void CTape::StopPlay ()
{
	m_bPlay = FALSE;
}



//////////////////////////////////////////////////////////////////////
void CTape::StartRecord (BOOL bAutoBeginRecord, BOOL bAutoEndRecord)
{
	m_bRecord = TRUE;

	m_bAutoBeginRecord = bAutoBeginRecord;
	m_bAutoEndRecord = bAutoEndRecord;

	delete []m_pRecord;
	m_pRecord = new BYTE[RECORD_BUFFER];
	m_record_pos = 0;
	m_recordLength = RECORD_BUFFER;
	m_average = 128;
}



//////////////////////////////////////////////////////////////////////
void CTape::RecordWaveGetBuffer (BYTE* pBuff, int buffLength)
{
	if (!m_pRecord)
		return;

	if (m_record_pos + buffLength >= m_recordLength)
	{
		while (m_record_pos + buffLength >= m_recordLength)
		{
			BYTE* pNewRecord = new BYTE[m_recordLength + RECORD_GROW];
			memcpy (pNewRecord, m_pRecord, m_record_pos);
			delete []m_pRecord;
			m_pRecord = pNewRecord;
			m_recordLength = m_recordLength + RECORD_GROW;
		}
	}

	memcpy (m_pRecord + m_record_pos, pBuff, buffLength);

	if (FindRecordEnd ())
		StopRecord ();

	if (FindRecordBegin (buffLength)) // If start record position find
		m_record_pos += buffLength;
}



//////////////////////////////////////////////////////////////////////
void CTape::StopRecord ()
{
	if (m_pRecord)
	{
		m_bRecord = FALSE;

		delete []m_pWave;
		m_pWave = m_pRecord;
		m_pRecord = NULL;
		m_waveLength = m_record_pos;
	}
}



//////////////////////////////////////////////////////////////////////
BOOL CTape::FindRecordBegin (DWORD buffLength)
{
	if (m_bAutoBeginRecord)
	{
		m_record_pos = 0;
		int imp_num = 0;
		
		while (1)
		{
			int len = CalcImpLength (m_pRecord, m_record_pos, buffLength);

			if (len == BAD_LENGTH)
			{
				m_record_pos = 0;
				return FALSE;
			}

			if ((m_last_len - len) * (m_last_len - len) <= 3 * 3)
			{
				imp_num++;

				if (imp_num >= 30)
					break;
			}
			else
				imp_num = 0;
			
			m_last_len = len;

			//TRACE ("\nWave pos %06i, Start = %i, Len = %i, Imp_Num = %i", m_pos, begin_pos, len, imp_num);
			//Sleep (1);
		}

		m_record_pos = 0;
		m_bAutoBeginRecord = FALSE;
	}

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CTape::FindRecordEnd ()
{
	if (m_bAutoBeginRecord)
		return FALSE;

	if (m_bAutoEndRecord)
	{
		int last_sample = 0;
		int last_length = 0;

		for (int i = 0; i < 1000; i++)
		{
			if (m_pRecord[m_record_pos + i] == last_sample)
				last_length++;
			else
			{
				last_sample = m_pRecord[m_record_pos + i];
				last_length = 0;
			}

			if (last_length == 500)
			{
				m_bAutoEndRecord = FALSE;
				return TRUE;
			}
		}
	}

	return FALSE;
}