/*************************************************
*            BK Emulator for Window 3.0          *
*************************************************/

/*
File: Speaker.cpp
Version: 1.0.0
Written by: Yuriy Kalmykov <kalmykov@stoik.com>
    Copyright (c) 2002-2004 Yuriy Kalmykov

    BK Emulator is a program emulated hardware environment for running
code for BK 0010(01) in different configurations. 
           
    This code may be used in compiled form in any way you desire.
This file or it's parts can't be redistributed without the authors
written consent, but can be modified for your private needs.
    Providing that this notice and the authors name and all copyright
notices remains intact.

    Please, an email me to know how you are using it and where. You can
ask me for any information about this below code or any attendant
knowledge.
    
    This file is provided "as is" with no expressed or implied warranty.
The author accepts no liability for any damage or loss of business that
this product may cause.
*/




// Speaker.cpp: implementation of the CSpeaker class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "Speaker.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CSpeaker::CSpeaker()
{
	m_bEnableSound = TRUE;
	m_pSamples = new BYTE[DEFAULT_SIZE];
	m_nSampleCount = 0;
	m_lastSample = MIN_SAMPLE;
	m_lastTick = 0;
	m_pRecieveSamples = NULL;
}

CSpeaker::~CSpeaker()
{
	delete []m_pSamples;
	delete []m_pRecieveSamples;
}



//////////////////////////////////////////////////////////////////////
void CSpeaker::OnReset ()
{
	m_nSampleCount = 0;
	m_lastTick = 0;
}



//////////////////////////////////////////////////////////////////////
void CSpeaker::GetByte (int addr, BYTE* pValue)
{
	ASSERT (FALSE);
	*pValue = 0;
}



//////////////////////////////////////////////////////////////////////
void CSpeaker::GetWord (int addr, WORD* pValue)
{
	/*
	if (m_pRecieveSamples[m_tickCount - m_lastTick] >= 128)
		return 040;
	else
		return 0;
	*/
	if (m_pRecieveSamples[m_tickCount - m_lastTick] >= 128)
		*pValue |= 040;
	else
		*pValue &= ~040;

}



//////////////////////////////////////////////////////////////////////
void CSpeaker::SetByte (int addr, BYTE value)
{
	ASSERT (FALSE);
}



//////////////////////////////////////////////////////////////////////
void CSpeaker::SetWord (int addr, WORD value)
{
	ASSERT (addr == 0177716);

	if (addr == 0177716)
	{
		// If register 0177716
		BYTE sample = MIN_SAMPLE;
	
		if (value & 0100)
			sample = MAX_SAMPLE;

		SetSample (sample);
	}
}



//////////////////////////////////////////////////////////////////////
void CSpeaker::SetSample (BYTE sample)
{
	// Fill dropped samples
	for (int n = 0; n < m_tickCount - m_lastTick - 1; n++)
	{
		// Add last sample to buffer
		m_pSamples[m_nSampleCount] = m_lastSample;
		m_nSampleCount++;
	}

	// Add new sample to buffer
	m_pSamples[m_nSampleCount] = sample;
	m_nSampleCount++;

	// Save last values
	m_lastSample = sample;
	m_lastTick = m_tickCount;
}



//////////////////////////////////////////////////////////////////////
int	CSpeaker::GetSampleCount ()
{
	return m_nSampleCount;
}



//////////////////////////////////////////////////////////////////////
void CSpeaker::GetAndRemoveSamples (BYTE* pBuff, int length)
{
	// Flush sound data of buffer end
	SetSample (m_lastSample);

	if (m_bEnableSound)
	{
		// Stretch buffer to specified length
		///*
		for (int n = 0; n < length; n++)
			pBuff[n] = m_pSamples[n * (m_nSampleCount - 1) / (length - 1)];
		//*/
	}
	else
		memset (pBuff, 0, length); // If no sound fill all buffer by zero
		
	ASSERT (m_lastTick == m_tickCount);

	// Reset buffer
	m_nSampleCount = 0;
}



//////////////////////////////////////////////////////////////////////
void CSpeaker::MixAndRemoveSamples (BYTE* pBuff, int length)
{
	// Flush sound data of buffer end
	SetSample (m_lastSample);

	if (m_bEnableSound)
	{
		// Stretch buffer to specified length
		for (int n = 0; n < length; n++)
		{
			int samples = (pBuff[n] + m_pSamples[n * (m_nSampleCount - 1) / (length - 1)] - 128 * 2) / 2;
			pBuff[n] = min (MAX_SAMPLE, max (MIN_SAMPLE, samples + 128));
		}
	}
		
	ASSERT (m_lastTick == m_tickCount);

	// Reset buffer
	m_nSampleCount = 0;
}



//////////////////////////////////////////////////////////////////////
void CSpeaker::SetIPS (int ips)
{
	delete []m_pRecieveSamples;
	m_pRecieveSamples = new BYTE[ips];

	m_ips = ips;
}



//////////////////////////////////////////////////////////////////////
void CSpeaker::ReceiveSamples (BYTE* pBuff, int length)
{
	ASSERT (m_pRecieveSamples);

	// Stretch buffer to specified length
	for (int n = 0; n < m_ips; n++)
	{
		m_pRecieveSamples[n] = pBuff[n * (length - 1) / (m_ips - 1)];
	}
}



//////////////////////////////////////////////////////////////////////
void CSpeaker::EnableSound (BOOL bEnable)
{
	m_bEnableSound = bEnable;
}



//////////////////////////////////////////////////////////////////////
BOOL CSpeaker::IsSoundEnabled ()
{
	return m_bEnableSound;
}
