/*************************************************
*            BK Emulator for Window 3.0          *
*************************************************/

/*
File: Screen.cpp
Version: 1.0.0
Written by: Yuriy Kalmykov <kalmykov@stoik.com>
    Copyright (c) 2002-2004 Yuriy Kalmykov

    BK Emulator is a program emulated hardware environment for running
code for BK 0010(01) in different configurations. 
           
    This code may be used in compiled form in any way you desire.
This file or it's parts can't be redistributed without the authors
written consent, but can be modified for your private needs.
    Providing that this notice and the authors name and all copyright
notices remains intact.

    Please, an email me to know how you are using it and where. You can
ask me for any information about this below code or any attendant
knowledge.
    
    This file is provided "as is" with no expressed or implied warranty.
The author accepts no liability for any damage or loss of business that
this product may cause.
*/




// CScreen.cpp : implementation of the CScreen class
//

#include "stdafx.h"
#include "Dst.h"
#include "BK.h"
#include "Screen.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CScreen

CScreen::CScreen ()
{
	m_nFrame = 0;
	m_curFPS = 0;
	m_msec = GetTickCount ();
	
	m_bFullScreen = FALSE;
	m_bSaveCurrentMode = FALSE;
	m_bColorMode = TRUE;
	m_bExtended = FALSE;
	m_bAdapt = FALSE;
	m_nOfs = 0330;

	m_colTable8 = NULL;
	m_monTable8 = NULL;
	m_colTable16 = NULL;
	m_monTable16 = NULL;
	m_colTable32 = NULL;
	m_monTable32 = NULL;

	m_pDD = NULL;
	m_pPrimarySurface = NULL;
	m_pBackSurface = NULL;
	m_pScreenBits = NULL;

	SetWindowMode ();

	InitColorTables ();
	InitDirectDraw ();
}

CScreen::~CScreen()
{
	SetWindowMode ();

	if (m_pPalette)
		m_pPalette.Release ();

	if (m_pPrimarySurface)
		m_pPrimarySurface.Release ();

	if (m_pBackSurface)
		m_pBackSurface.Release ();

	if (m_pDD)
		m_pDD.Release ();

	delete []m_pScreenBits;
	delete []m_colTable8;
	delete []m_monTable8;
	delete []m_colTable16;
	delete []m_monTable16;
	delete []m_colTable32;
	delete []m_monTable32;
}

BEGIN_MESSAGE_MAP(CScreen,CWnd )
	//{{AFX_MSG_MAP(CScreen)
	ON_WM_PAINT()
	ON_WM_ERASEBKGND()
	ON_WM_SETFOCUS()
	ON_WM_CREATE()
	ON_WM_LBUTTONDOWN()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CScreen message handlers

/////////////////////////////////////////////////////////////////////////////
void CScreen::OnReset ()
{
}



/////////////////////////////////////////////////////////////////////////////
void CScreen::InitColorTables ()
{
	// Create color tables of all display modes for faster drawing
	delete []m_colTable8;
	delete []m_monTable8;
	delete []m_colTable16;
	delete []m_monTable16;
	delete []m_colTable32;
	delete []m_monTable32;
	m_colTable8 = new BYTE[256 * 8 * 2];
	m_monTable8 = new BYTE[256 * 8 * 2];
	m_colTable16 = new WORD[256 * 8 * 2];
	m_monTable16 = new WORD[256 * 8 * 2];
	m_colTable32 = new DWORD[256 * 8 * 2];
	m_monTable32 = new DWORD[256 * 8 * 2];

	for (int i = 0; i < 256; i++)
	{
		BYTE* pColBuff8 = &m_colTable8[i * 8 * 2];
		BYTE* pMonBuff8 = &m_monTable8[i * 8 * 2];
		WORD* pColBuff16 = &m_colTable16[i * 8 * 2];
		WORD* pMonBuff16 = &m_monTable16[i * 8 * 2];
		DWORD* pColBuff32 = &m_colTable32[i * 8 * 2];
		DWORD* pMonBuff32 = &m_monTable32[i * 8 * 2];

		for (int n = 0; n < 4; n++)
		{
			switch ((i >> (n * 2)) & 3)
			{
			// Black
			case 0:
					pColBuff8[n * 4 + 0] = 0;
					pColBuff8[n * 4 + 1] = 0;
					pColBuff8[n * 4 + 2] = 0;
					pColBuff8[n * 4 + 3] = 0;

					pMonBuff8[n * 4 + 0] = 0;
					pMonBuff8[n * 4 + 1] = 0;
					pMonBuff8[n * 4 + 2] = 0;
					pMonBuff8[n * 4 + 3] = 0;
				
					pColBuff16[n * 4 + 0] = 0x0;
					pColBuff16[n * 4 + 1] = 0x0;
					pColBuff16[n * 4 + 2] = 0x0;
					pColBuff16[n * 4 + 3] = 0x0;

					pMonBuff16[n * 4 + 0] = 0x0;
					pMonBuff16[n * 4 + 1] = 0x0;
					pMonBuff16[n * 4 + 2] = 0x0;
					pMonBuff16[n * 4 + 3] = 0x0;
				
					pColBuff32[n * 4 + 0] = 0x0;
					pColBuff32[n * 4 + 1] = 0x0;
					pColBuff32[n * 4 + 2] = 0x0;
					pColBuff32[n * 4 + 3] = 0x0;

					pMonBuff32[n * 4 + 0] = 0x0;
					pMonBuff32[n * 4 + 1] = 0x0;
					pMonBuff32[n * 4 + 2] = 0x0;
					pMonBuff32[n * 4 + 3] = 0x0;
				
				break;
			// Blue and Gray
			case 1:
					pColBuff8[n * 4 + 0] = 252;
					pColBuff8[n * 4 + 1] = 252;
					pColBuff8[n * 4 + 2] = 252;
					pColBuff8[n * 4 + 3] = 252;

					pMonBuff8[n * 4 + 0] = 255;
					pMonBuff8[n * 4 + 1] = 255;
					pMonBuff8[n * 4 + 2] = 0;
					pMonBuff8[n * 4 + 3] = 0;

					pColBuff16[n * 4 + 0] = 0x1f;
					pColBuff16[n * 4 + 1] = 0x1f;
					pColBuff16[n * 4 + 2] = 0x1f;
					pColBuff16[n * 4 + 3] = 0x1f;

					pMonBuff16[n * 4 + 0] = 0xffff;
					pMonBuff16[n * 4 + 1] = 0xffff;
					pMonBuff16[n * 4 + 2] = 0x0;
					pMonBuff16[n * 4 + 3] = 0x0;

					pColBuff32[n * 4 + 0] = RGB (0xff, 0x0, 0x0);
					pColBuff32[n * 4 + 1] = RGB (0xff, 0x0, 0x0);
					pColBuff32[n * 4 + 2] = RGB (0xff, 0x0, 0x0);
					pColBuff32[n * 4 + 3] = RGB (0xff, 0x0, 0x0);

					pMonBuff32[n * 4 + 0] = RGB (0xff, 0xff, 0xff);
					pMonBuff32[n * 4 + 1] = RGB (0xff, 0xff, 0xff);
					pMonBuff32[n * 4 + 2] = 0x0;
					pMonBuff32[n * 4 + 3] = 0x0;
				
				break;
			// Green and Gray
			case 2:
					pColBuff8[n * 4 + 0] = 250;
					pColBuff8[n * 4 + 1] = 250;
					pColBuff8[n * 4 + 2] = 250;
					pColBuff8[n * 4 + 3] = 250;

					pMonBuff8[n * 4 + 0] = 0;
					pMonBuff8[n * 4 + 1] = 0;
					pMonBuff8[n * 4 + 2] = 255;
					pMonBuff8[n * 4 + 3] = 255;

					pColBuff16[n * 4 + 0] = 0x7e0;
					pColBuff16[n * 4 + 1] = 0x7e0;
					pColBuff16[n * 4 + 2] = 0x7e0;
					pColBuff16[n * 4 + 3] = 0x7e0;

					pMonBuff16[n * 4 + 0] = 0x0;
					pMonBuff16[n * 4 + 1] = 0x0;
					pMonBuff16[n * 4 + 2] = 0xffff;
					pMonBuff16[n * 4 + 3] = 0xffff;

					pColBuff32[n * 4 + 0] = RGB (0x0, 0xff, 0x0);
					pColBuff32[n * 4 + 1] = RGB (0x0, 0xff, 0x0);
					pColBuff32[n * 4 + 2] = RGB (0x0, 0xff, 0x0);
					pColBuff32[n * 4 + 3] = RGB (0x0, 0xff, 0x0);

					pMonBuff32[n * 4 + 0] = 0x0;
					pMonBuff32[n * 4 + 1] = 0x0;
					pMonBuff32[n * 4 + 2] = RGB (0xff, 0xff, 0xff);
					pMonBuff32[n * 4 + 3] = RGB (0xff, 0xff, 0xff);
					
				break;
			case 3:
					pColBuff8[n * 4 + 0] = 249;
					pColBuff8[n * 4 + 1] = 249;
					pColBuff8[n * 4 + 2] = 249;
					pColBuff8[n * 4 + 3] = 249;

					pMonBuff8[n * 4 + 0] = 255;
					pMonBuff8[n * 4 + 1] = 255;
					pMonBuff8[n * 4 + 2] = 255;
					pMonBuff8[n * 4 + 3] = 255;

					pColBuff16[n * 4 + 0] = 0xf800;
					pColBuff16[n * 4 + 1] = 0xf800;
					pColBuff16[n * 4 + 2] = 0xf800;
					pColBuff16[n * 4 + 3] = 0xf800;

					pMonBuff16[n * 4 + 0] = 0xffff;
					pMonBuff16[n * 4 + 1] = 0xffff;
					pMonBuff16[n * 4 + 2] = 0xffff;
					pMonBuff16[n * 4 + 3] = 0xffff;
					
					pColBuff32[n * 4 + 0] = RGB (0x0, 0x0, 0xff);
					pColBuff32[n * 4 + 1] = RGB (0x0, 0x0, 0xff);
					pColBuff32[n * 4 + 2] = RGB (0x0, 0x0, 0xff);
					pColBuff32[n * 4 + 3] = RGB (0x0, 0x0, 0xff);

					pMonBuff32[n * 4 + 0] = RGB (0xff, 0xff, 0xff);
					pMonBuff32[n * 4 + 1] = RGB (0xff, 0xff, 0xff);
					pMonBuff32[n * 4 + 2] = RGB (0xff, 0xff, 0xff);
					pMonBuff32[n * 4 + 3] = RGB (0xff, 0xff, 0xff);
					
				break;
			default:
				ASSERT (FALSE);
				break;
			}

			// If fullscreen and adapt mode
			// Swap 1 and 2 mask for original BK view in fullscreen
			if (m_bAdapt && m_columnspp == 2)
			{
				BYTE mask8 =  pMonBuff8[n * 4 + 1];
				pMonBuff8[n * 4 + 1] = pMonBuff8[n * 4 + 2];
				pMonBuff8[n * 4 + 2] = mask8;

				WORD mask16 =  pMonBuff16[n * 4 + 1];
				pMonBuff16[n * 4 + 1] = pMonBuff16[n * 4 + 2];
				pMonBuff16[n * 4 + 2] = mask16;

				DWORD mask32 =  pMonBuff32[n * 4 + 1];
				pMonBuff32[n * 4 + 1] = pMonBuff32[n * 4 + 2];
				pMonBuff32[n * 4 + 2] = mask32;
			}
		}
	}
}



/////////////////////////////////////////////////////////////////////////////
BOOL CScreen::PreCreateWindow(CREATESTRUCT& cs) 
{
	if (!CWnd::PreCreateWindow(cs))
		return FALSE;

	cs.lpszClass = AfxRegisterWndClass(CS_HREDRAW|CS_VREDRAW|CS_DBLCLKS, 
		::LoadCursor(NULL, IDC_ARROW), HBRUSH(COLOR_WINDOW+1), NULL);

	return TRUE;
}



/////////////////////////////////////////////////////////////////////////////
int CScreen::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
	if (CWnd ::OnCreate(lpCreateStruct) == -1)
		return -1;
	
	// CreateEx for creating popup window
	m_DDWnd.CreateEx (0, NULL, NULL, WS_POPUP|WS_VISIBLE, CRect (0, 0, 0, 0), NULL, 0);
	
	if (m_bFullScreen)
		SetFullScreenMode ();
	else
		SetWindowMode ();
	
	return 0;
}



/////////////////////////////////////////////////////////////////////////////
void CScreen::OnPaint() 
{
	CPaintDC dc(this); // device context for painting

	DrawScreen ();
}



/////////////////////////////////////////////////////////////////////////////
void CScreen::DrawScreen ()
{
	ASSERT (m_pDD);
	if (!m_pDD)
		return;

	ASSERT (m_columnspp == 1 || m_columnspp == 2);
	ASSERT (m_linespp == 2 || m_linespp == 3);

	// Calc screen roll offset rectangles
	int nOfs = 0;
	
	if (m_nOfs >= 0330)
		nOfs = (m_nOfs - 0330);
	else
		nOfs = (050 + m_nOfs);
	
	// DDraw drawing
	// Restore surfaces if necessary
	if (!m_pPrimarySurface)
		return;

	if (m_pPrimarySurface->IsLost () != S_OK)
		m_pPrimarySurface->Restore ();

	if (!m_pBackSurface)
		return;

	if (m_pBackSurface->IsLost () != S_OK)
		m_pBackSurface->Restore ();
	
	// Draw screen buffer in back surface
	int screen_bpr = SCREEN_WIDTH * m_columnspp * (m_bpp / 8);

	DDSURFACEDESC2 ddsd;
	ddsd.dwSize  = sizeof(ddsd);
	
	if (m_pBackSurface->Lock (NULL, &ddsd, DDLOCK_SURFACEMEMORYPTR|DDLOCK_WAIT, NULL) != S_OK)
		return; 

	BYTE* pBackRow = (BYTE*)ddsd.lpSurface;
	BYTE* pScreenRow = m_pScreenBits + nOfs * screen_bpr;

	// Draw top part of screen
	for (int y = 0; y < SCREEN_HEIGHT - nOfs; y++)
	{
		for (int n = 0; n < m_linespp; n++)
		{
			memcpy (pBackRow, pScreenRow, screen_bpr);
			pBackRow += ddsd.lPitch;
		}

		pScreenRow += screen_bpr;
	}

	// Draw bottom part of screen
	pScreenRow = m_pScreenBits;
	for (y = 0; y < nOfs; y++)
	{
		for (int n = 0; n < m_linespp; n++)
		{
			memcpy (pBackRow, pScreenRow, screen_bpr);
			pBackRow += ddsd.lPitch;
		}

		pScreenRow += screen_bpr;
	}

	// Draw extended mode black mask if necessary
	if (m_bExtended)
		memset ((BYTE*)ddsd.lpSurface + screen_bpr * 64 * m_linespp, 0, screen_bpr * 192 * m_linespp);

	
	if (m_pBackSurface->Unlock (NULL) != S_OK)
		return;
	
	// Draw back surface in primary surface
	CComPtr <IDirectDrawClipper> pClipper;

	CRect rcWindow = GetScreenRect ();
	
	CRect* pDstRect = NULL;
	if (!m_bFullScreen)
	{
		pDstRect = &rcWindow;

		// For window mode create Clipper
		m_pDD->CreateClipper (0, &pClipper, NULL);

		// Set clipper to screen window and attach it to primary surface
		pClipper->SetHWnd (0, m_hWnd);
		m_pPrimarySurface->SetClipper (pClipper);
		pClipper.Release (); // Release for decrement reference count from 2 to 1
	}

	m_pPrimarySurface->Blt (pDstRect, m_pBackSurface, NULL, DDBLT_ASYNC, NULL);

	// Remove and Release clipper
	m_pPrimarySurface->SetClipper (NULL); 
	pClipper.Release ();

	// Calculate FPS
	m_nFrame++;

	if (GetTickCount () - m_msec >= 1000)
	{
		m_msec = GetTickCount ();

		m_curFPS = m_nFrame;
		m_nFrame = 0;
	}
}



/////////////////////////////////////////////////////////////////////////////
CRect CScreen::GetScreenRect ()
{
	CRect rcScreen (0, 0, SCREEN_WIDTH * m_columnspp, SCREEN_HEIGHT * m_linespp);
	CRect rcNewScreen = rcScreen;
	
	CRect rcClient;
	GetClientRect (&rcClient);
	ClientToScreen (&rcClient);

	if (rcScreen.Width () * rcClient.Height () > rcClient.Width () * rcScreen.Height ())
	{
		rcNewScreen.right = rcClient.Width ();
		rcNewScreen.bottom = rcScreen.bottom * rcClient.Width () / rcScreen.right;
	}
	else
	{
		rcNewScreen.bottom = rcClient.Height ();
		rcNewScreen.right = rcScreen.right * rcClient.Height () / rcScreen.bottom;
	}

	if (rcNewScreen.Width () > SCREEN_WIDTH * m_columnspp)
		rcNewScreen.right = SCREEN_WIDTH * m_columnspp;

	if (rcNewScreen.Height () > SCREEN_HEIGHT * m_linespp)
		rcNewScreen.bottom = SCREEN_HEIGHT * m_linespp;
	
	int x_offs = (rcClient.Width () - rcNewScreen.Width ()) / 2;
	int y_offs = (rcClient.Height () - rcNewScreen.Height ()) / 2;

	rcClient.right = rcClient.left + rcNewScreen.Width ();
	rcClient.bottom = rcClient.top + rcNewScreen.Height ();
	rcClient.OffsetRect (x_offs, y_offs);

	return rcClient;
}



/////////////////////////////////////////////////////////////////////////////
BOOL CScreen::OnEraseBkgnd(CDC* pDC) 
{
	CRect rcScreen = GetScreenRect ();
	ScreenToClient (&rcScreen);

	CRect rcClient;
	GetClientRect (&rcClient);

	CRgn rgnRes;
	rgnRes.CreateRectRgn (0, 0, 0, 0);

	CRgn rgnScreen;
	rgnScreen.CreateRectRgnIndirect (&rcScreen);

	CRgn rgnClient;
	rgnClient.CreateRectRgnIndirect (&rcClient);

	rgnRes.CombineRgn (&rgnClient, &rgnScreen, RGN_DIFF);

	CBrush br;
	br.CreateSysColorBrush (COLOR_BTNFACE);

	pDC->SelectClipRgn (&rgnRes);
	CBrush* pOld = pDC->SelectObject (&br);
	pDC->PatBlt (0, 0, rcClient.Width (), rcClient.Height (), PATCOPY);
	pDC->SelectObject (pOld);

	return TRUE;
}



/////////////////////////////////////////////////////////////////////////////
void CScreen::SetByte (int addr, BYTE value)
{
	if (addr != 0177664)
		SetScreenByte (addr, value); // If screen memory
	else
	{
		// If register 0177664
		SetOffset (value);
		SetExtendedMode (!(value & 01000));
	}
}



/////////////////////////////////////////////////////////////////////////////
void CScreen::GetByte (int addr, BYTE* pValue)
{
	ASSERT (FALSE);
}



/////////////////////////////////////////////////////////////////////////////
void CScreen::SetWord (int addr, WORD value)
{
	ASSERT ((addr >= 040000 && addr < 0100000) || addr == 0177664);

	if (addr != 0177664)
		SetScreenWord (addr, value); // If screen memory
	else
	{
		// If register 0177664
		SetOffset (value);
		SetExtendedMode (!(value & 01000));
	}
}



/////////////////////////////////////////////////////////////////////////////
void CScreen::GetWord (int addr, WORD* pValue)
{
	ASSERT (FALSE);
}



/////////////////////////////////////////////////////////////////////////////
void CScreen::SetScreenByte (WORD addr, BYTE val)
{
	// Draw 4 pixels
	ASSERT (m_columnspp == 1 || m_columnspp == 2);
	ASSERT (m_linespp == 2 || m_linespp == 3);

	// Calculate x and y  
	DWORD x = addr & 63;
	DWORD y = (addr - 040000) >> 6;

	// Draw 8 bit pixels
	if (m_bpp == 8)
	{
		BYTE* pDst = (BYTE*)m_pScreenBits + x * 8 * m_columnspp + y * SCREEN_WIDTH * m_columnspp;
		BYTE* pSrc;
			
		if (m_bColorMode)
			pSrc = &m_colTable8[(val << 3) * 2];
		else
			pSrc = &m_monTable8[(val << 3) * 2];				

		if (m_columnspp == 1)
		{
			// In window mode
			*pDst++ = *pSrc++; pSrc++;
			*pDst++ = *pSrc++; pSrc++;
			*pDst++ = *pSrc++; pSrc++;
			*pDst++ = *pSrc++; pSrc++;
			*pDst++ = *pSrc++; pSrc++;
			*pDst++ = *pSrc++; pSrc++;
			*pDst++ = *pSrc++; pSrc++;
			*pDst++ = *pSrc++; pSrc++;
		}
		else
		{
			// In fullscreen mode
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
		}
	}
	// Draw 16 bit pixels
	else if (m_bpp == 16)
	{
		WORD* pDst = (WORD*)m_pScreenBits + x * 8 * m_columnspp + y * SCREEN_WIDTH * m_columnspp;
		WORD* pSrc;
			
		if (m_bColorMode)
			pSrc = &m_colTable16[(val << 3) * 2];
		else
			pSrc = &m_monTable16[(val << 3) * 2];				

		if (m_columnspp == 1)
		{
			// In window mode
			*pDst++ = *pSrc++; pSrc++;
			*pDst++ = *pSrc++; pSrc++;
			*pDst++ = *pSrc++; pSrc++;
			*pDst++ = *pSrc++; pSrc++;
			*pDst++ = *pSrc++; pSrc++;
			*pDst++ = *pSrc++; pSrc++;
			*pDst++ = *pSrc++; pSrc++;
			*pDst++ = *pSrc++; pSrc++;
		}
		else
		{
			// In fullscreen mode
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
		}
	}
	// Draw 24 bit pixels
	else if (m_bpp == 24)
	{
		BYTE* pDst = (BYTE*)m_pScreenBits + (x * 8 * m_columnspp + y * SCREEN_WIDTH * m_columnspp) * 3;
		DWORD* pSrc;
			
		if (m_bColorMode)
			pSrc = &m_colTable32[(val << 3) * 2];
		else
			pSrc = &m_monTable32[(val << 3) * 2];				

		if (m_columnspp == 1)
		{
			// In window mode
			for (int n = 0; n < 8; n++)
			{
				*pDst++ = GetRValue (*pSrc);
				*pDst++ = GetGValue (*pSrc);
				*pDst++ = GetBValue (*pSrc); 
				pSrc += 2;
			}
		}
		else
		{
			// In fullscreen mode
			for (int n = 0; n < 16; n++)
			{
				*pDst++ = GetRValue (*pSrc);
				*pDst++ = GetGValue (*pSrc);
				*pDst++ = GetBValue (*pSrc); 
				pSrc++;
			}
		}
	}
	// Draw 32 bit pixels
	else if (m_bpp == 32)
	{
		DWORD* pDst = (DWORD*)m_pScreenBits + x * 8 * m_columnspp + y * SCREEN_WIDTH * m_columnspp;
		DWORD* pSrc;
			
		if (m_bColorMode)
			pSrc = &m_colTable32[(val << 3) * 2];
		else
			pSrc = &m_monTable32[(val << 3) * 2];				

		if (m_columnspp == 1)
		{
			// In window mode
			*pDst++ = *pSrc++; pSrc++;
			*pDst++ = *pSrc++; pSrc++;
			*pDst++ = *pSrc++; pSrc++;
			*pDst++ = *pSrc++; pSrc++;
			*pDst++ = *pSrc++; pSrc++;
			*pDst++ = *pSrc++; pSrc++;
			*pDst++ = *pSrc++; pSrc++;
			*pDst++ = *pSrc++; pSrc++;
		}
		else
		{
			// In fullscreen mode
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
			*pDst++ = *pSrc++;
		}
	}
}



/////////////////////////////////////////////////////////////////////////////
void CScreen::SetScreenWord (WORD addr, WORD val)
{
	SetByte (addr & 0xfffe, val & 0xff);
	SetByte ((addr & 0xfffe) + 1, val >> 8);
}



/////////////////////////////////////////////////////////////////////////////
void CScreen::SetColorMode (BOOL bColorMode)
{
	m_bColorMode = bColorMode;
}



/////////////////////////////////////////////////////////////////////////////
void CScreen::SetExtendedMode (BOOL bFlag)
{
	m_bExtended = bFlag;
}



/////////////////////////////////////////////////////////////////////////////
void CScreen::SetAdaptMode (BOOL bFlag)
{
	m_bAdapt = bFlag;
}



/////////////////////////////////////////////////////////////////////////////
void CScreen::SetSaveCurrentMode (BOOL bFlag)
{
	m_bSaveCurrentMode = bFlag;
}



/////////////////////////////////////////////////////////////////////////////
void CScreen::SetOffset (BYTE nOfs)
{
	m_nOfs = nOfs;
}



//////////////////////////////////////////////////////////////////////
BYTE CScreen::GetOffset ()
{
	return m_nOfs;
}



//////////////////////////////////////////////////////////////////////
void CScreen::OnSetFocus (CWnd* pOldWnd) 
{	
	AfxGetMainWnd ()->SetFocus ();
}



//////////////////////////////////////////////////////////////////////
BOOL CScreen::SetFullScreenMode ()
{
	if (m_pDD)
	{
		try
		{
			CDSTHResult hr;

			// Set Normal (Window Mode)
			hr = m_pDD->RestoreDisplayMode ();
			
			// Release old surfaces
			m_pPrimarySurface.Release ();
			m_pBackSurface.Release ();

			// Set Exclusive (Fullscreen Mode)
			hr = m_pDD->SetCooperativeLevel (m_DDWnd.m_hWnd, DDSCL_EXCLUSIVE|DDSCL_FULLSCREEN);
		
			DDSURFACEDESC2 ddsd;
			memset(&ddsd, 0, sizeof(ddsd));
			ddsd.dwSize  = sizeof(ddsd);
	
			// Get current display mode
			hr = m_pDD->GetDisplayMode (&ddsd);
			ChangeDisplayMode ();
			
			// Create primary surface
			hr = m_pDD->GetDisplayMode (&ddsd);
			ddsd.dwSize  = sizeof(ddsd);
			ddsd.dwFlags = DDSD_CAPS;
			ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;
		
			hr = m_pDD->CreateSurface (&ddsd, &m_pPrimarySurface, NULL);

			// Init palette in 8 bit mode
			if (ddsd.ddpfPixelFormat.dwRGBBitCount == 8)
				CreatePalette ();
			
			// Show background window
			if (m_DDWnd.m_hWnd)
			{
				m_DDWnd.ShowWindow (SW_SHOW);
				m_DDWnd.SetWindowPos (&wndTopMost, 0, 0, 0, 0, SWP_NOSIZE|SWP_NOMOVE);
			}

			// Create Back surface with primary surface parameters
			m_columnspp = 2;
			m_linespp = 3;
			
			ddsd.dwFlags = DDSD_CAPS|DDSD_WIDTH|DDSD_HEIGHT|DDSD_PIXELFORMAT;
			ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
			ddsd.dwWidth = SCREEN_WIDTH * m_columnspp;
			ddsd.dwHeight = SCREEN_HEIGHT * m_linespp;
			hr = m_pDD->CreateSurface (&ddsd, &m_pBackSurface, NULL);
				
			// Create new screen buffer
			m_bpp = ddsd.ddpfPixelFormat.dwRGBBitCount;
			CreateScreenBuffer ();
		}
		catch (CDSTFailQueryInterfaceException)
		{
			// If failed restore window mode
			SetWindowMode ();

			return FALSE;
		}
	}

	InitColorTables ();

	m_bFullScreen = TRUE;

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CScreen::SetWindowMode ()
{
	// Hide Background window
	if (m_DDWnd.m_hWnd)
		m_DDWnd.ShowWindow (SW_HIDE);
	
	if (m_pDD)
	{
		try
		{
			CDSTHResult hr;

			// Set Normal (Window Mode)
			hr = m_pDD->RestoreDisplayMode ();
			hr = m_pDD->SetCooperativeLevel (NULL, DDSCL_NORMAL);
			
			// Release old surfaces
			m_pPrimarySurface.Release ();
			m_pBackSurface.Release ();

			// Create primary surface
			DDSURFACEDESC2 ddsd;
			memset(&ddsd, 0, sizeof(ddsd));
			ddsd.dwSize  = sizeof(ddsd);
			ddsd.dwFlags = DDSD_CAPS;
			ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;
		
			hr = m_pDD->CreateSurface (&ddsd, &m_pPrimarySurface, NULL);

			// Get current display mode
			hr = m_pDD->GetDisplayMode (&ddsd);

			// Create Back surface with primary surface parameters
			m_columnspp = 1;
			m_linespp = 2;
			
			ddsd.dwFlags = DDSD_CAPS|DDSD_WIDTH|DDSD_HEIGHT|DDSD_PIXELFORMAT;
			ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
			ddsd.dwWidth = SCREEN_WIDTH * m_columnspp;
			ddsd.dwHeight = SCREEN_HEIGHT * m_linespp;
			hr = m_pDD->CreateSurface (&ddsd, &m_pBackSurface, NULL);

			// Init palette in 8 bit mode
			if (ddsd.ddpfPixelFormat.dwRGBBitCount == 8)
				CreatePalette ();
			
			// Create new screen buffer
			m_columnspp = 1;
			m_linespp = 2;
			m_bpp = ddsd.ddpfPixelFormat.dwRGBBitCount;
			CreateScreenBuffer ();
		}
		catch (CDSTFailQueryInterfaceException)
		{
			return FALSE;
		}
	}
	
	InitColorTables ();

	m_bFullScreen = FALSE;

	return TRUE;
}



////////////////////////////////////////////////////////////////////////////
BOOL CScreen::InitDirectDraw ()
{
	CDSTHResult hr;

	try 
	{
		// Create Direct Draw object
		CComPtr <IDirectDraw> pDD;

		hr = DirectDrawCreate (NULL, &pDD, NULL);
		hr = pDD->QueryInterface (IID_IDirectDraw7, (void**)&m_pDD);
	}
	catch (CDSTFailQueryInterfaceException)
	{
		CBKMessageBox message;
		message.Show (IDS_DDRAW_NEEDED, MB_OK|MB_ICONSTOP);

		return FALSE;
	}
			
	return TRUE;
}



////////////////////////////////////////////////////////////////////////////
BOOL CScreen::CreateScreenBuffer ()
{
	ASSERT (m_columnspp == 1 || m_columnspp == 2);
	ASSERT (m_linespp == 2 || m_linespp == 3);

	// Delete old buffer
	delete []m_pScreenBits;
	m_pScreenBits = NULL;

	// Create new buffer according specified parameter
	if (m_bpp != 8 && m_bpp != 16 && m_bpp != 24 && m_bpp != 32)
		return FALSE;

	// Allocate screen buffer
	m_pScreenBits = new BYTE[SCREEN_WIDTH * m_columnspp * (m_bpp / 8) * SCREEN_HEIGHT];

	return TRUE;
}



void CScreen::CreatePalette ()
{
	CDSTHResult hr;

	// Create palette rfor 8-bit modes
	PALETTEENTRY palette[256];

	HDC hDC = ::GetDC (this->m_hWnd);
	::GetSystemPaletteEntries (hDC, 0, 256, palette);
	::ReleaseDC (this->m_hWnd, hDC);
	//memset (&palette, 0, sizeof (PALETTEENTRY) * 256);

	palette[0].peRed = 0;
	palette[0].peGreen = 0;
	palette[0].peBlue = 0;
	palette[0].peFlags = PC_NOCOLLAPSE;	
	palette[249].peRed = 255;
	palette[249].peGreen = 0;
	palette[249].peBlue = 0;
	palette[249].peFlags = PC_NOCOLLAPSE;
	palette[250].peRed = 0;
	palette[250].peGreen = 255;
	palette[250].peBlue = 0;
	palette[250].peFlags = PC_NOCOLLAPSE;
	palette[252].peRed = 0;
	palette[252].peGreen = 0;
	palette[252].peBlue = 255;
	palette[252].peFlags = PC_NOCOLLAPSE;
	palette[255].peRed = 255;
	palette[255].peGreen = 255;
	palette[255].peBlue = 255;
	palette[255].peFlags = PC_NOCOLLAPSE;
	
	m_pPalette.Release ();
	hr = m_pDD->CreatePalette (DDPCAPS_8BIT, palette, &m_pPalette, NULL);
	m_pPrimarySurface->SetPalette (m_pPalette);
}



////////////////////////////////////////////////////////////////////////////
void CScreen::ChangeDisplayMode ()
{
	CDSTHResult hr;

	if (m_pDD && !m_bSaveCurrentMode)
	{
		// Change display mode only if no m_bSaveCurrentMode flag
		// Try set 100 Hz first
		if (m_pDD->SetDisplayMode (1024, 768, 8, 100, 0) != S_OK)
		{
			ASSERT (FALSE);
			// If failed try 50Hz mode
			hr = m_pDD->SetDisplayMode (1024, 768, 8, 50, 0);
		}
	}
}



////////////////////////////////////////////////////////////////////////////
void CScreen::OnLButtonDown(UINT nFlags, CPoint point) 
{
	AfxGetMainWnd ()->SetFocus ();
	
	CWnd ::OnLButtonDown(nFlags, point);
}



/////////////////////////////////////////////////////////////////////////////
HBITMAP CScreen::GetScreenshot ()
{
	ASSERT (m_columnspp == 1 || m_columnspp == 2);
	ASSERT (m_linespp == 2 || m_linespp == 3);

	// Calc screen roll offset rectangles
	int nOfs = 0;
	
	if (m_nOfs >= 0330)
		nOfs = (m_nOfs - 0330);
	else
		nOfs = (050 + m_nOfs);
	
	// DDraw drawing
	// Restore surfaces if necessary
	if (!m_pBackSurface)
		return NULL;

	if (m_pBackSurface->IsLost () != S_OK)
		m_pBackSurface->Restore ();
	
	// Draw screen buffer in back surface
	int screen_bpr = SCREEN_WIDTH * m_columnspp * (m_bpp / 8);

	DDSURFACEDESC2 ddsd;
	ddsd.dwSize  = sizeof(ddsd);
	
	if (m_pBackSurface->Lock (NULL, &ddsd, DDLOCK_SURFACEMEMORYPTR|DDLOCK_WAIT, NULL) != S_OK)
		return NULL; 

	BYTE* pBackRow = (BYTE*)ddsd.lpSurface;
	BYTE* pScreenRow = m_pScreenBits + nOfs * screen_bpr;

	// Draw top part of screen
	for (int y = 0; y < SCREEN_HEIGHT - nOfs; y++)
	{
		for (int n = 0; n < m_linespp; n++)
		{
			memcpy (pBackRow, pScreenRow, screen_bpr);
			pBackRow += ddsd.lPitch;
		}

		pScreenRow += screen_bpr;
	}

	// Draw bottom part of screen
	pScreenRow = m_pScreenBits;
	for (y = 0; y < nOfs; y++)
	{
		for (int n = 0; n < m_linespp; n++)
		{
			memcpy (pBackRow, pScreenRow, screen_bpr);
			pBackRow += ddsd.lPitch;
		}

		pScreenRow += screen_bpr;
	}

	// Draw extended mode black mask if necessary
	if (m_bExtended)
		memset ((BYTE*)ddsd.lpSurface + screen_bpr * 64 * m_linespp, 0, screen_bpr * 192 * m_linespp);

	
	if (m_pBackSurface->Unlock (NULL) != S_OK)
		return NULL;
	
	// Draw back surface in DIB surface
	
	// Create DIB for screenshot
	BITMAPINFOHEADER bmih;
	ZeroMemory (&bmih, sizeof (BITMAPINFOHEADER));
	bmih.biSize = sizeof (BITMAPINFOHEADER);
	bmih.biWidth = SCREEN_WIDTH * m_columnspp;
	bmih.biHeight = SCREEN_HEIGHT * m_linespp;
	bmih.biCompression = BI_RGB;
	bmih.biPlanes = 1;
	bmih.biBitCount = 32;

	// Prepare DC's for coping
	HDC hDC = NULL;
	if (m_pBackSurface->GetDC (&hDC) != S_OK)
		return NULL;

	CDC* pDC = CDC::FromHandle (hDC);

	BYTE* pBits = NULL;
	HBITMAP hBm = ::CreateDIBSection (NULL, (BITMAPINFO*)&bmih, DIB_RGB_COLORS, (void**)&pBits, NULL, 0);

	CDC memDC;
	memDC.CreateCompatibleDC (pDC);

	HBITMAP hOld = (HBITMAP)memDC.SelectObject (hBm);
	memDC.BitBlt (0, 0, bmih.biWidth, bmih.biHeight, pDC, 0, 0, SRCCOPY);
	memDC.SelectObject (hOld);

	m_pBackSurface->ReleaseDC (hDC);

	return hBm;
}