/*************************************************
*            BK Emulator for Window 3.0          *
*************************************************/

/*
File: Panel.cpp
Version: 1.0.0
Written by: Yuriy Kalmykov <kalmykov@stoik.com>
    Copyright (c) 2002-2004 Yuriy Kalmykov

    BK Emulator is a program emulated hardware environment for running
code for BK 0010(01) in different configurations. 
           
    This code may be used in compiled form in any way you desire.
This file or it's parts can't be redistributed without the authors
written consent, but can be modified for your private needs.
    Providing that this notice and the authors name and all copyright
notices remains intact.

    Please, an email me to know how you are using it and where. You can
ask me for any information about this below code or any attendant
knowledge.
    
    This file is provided "as is" with no expressed or implied warranty.
The author accepts no liability for any damage or loss of business that
this product may cause.
*/




// Panel.cpp: implementation of the CPanel class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "Panel.h"
#include "PanelSplitter.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Class factory
//////////////////////////////////////////////////////////////////////

CPanel* CPanel::CreatePanelInstance (CWnd* pBaseWnd)
{
	return new CPanel (pBaseWnd);
}

CPanel* CPanel::CreatePanelInstance (const TCHAR* pName, double size)
{
	return new CPanel (pName, size);
}

CPanel* CPanel::CreatePanelInstance (const TCHAR* pName, int size)
{
	return new CPanel (pName, size);
}

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////
CPanel::CPanel (CWnd* pBaseWnd)
{
	m_windowAlign = 0;
	m_bSizeRelativeToWindow = TRUE;
	m_bHide = FALSE;
	m_bDrawDebugInfo = TRUE;
	m_bEnableSplitter = FALSE;
	m_bMouseTransparentBkg = TRUE;
	m_hBaseWnd = pBaseWnd->m_hWnd;
	m_hWnd = NULL;
	m_pParent = NULL;
	m_pSplitter = NULL;
	m_rcControlsBound.SetRectEmpty ();
	SetBorder (DEFAULT_BORDER_SIZE);

	SetName (NULL);

	m_align = ALIGN_NOWHERE;
	m_bFixedWidth = FALSE;
	m_bFixedHeight = FALSE;
	m_bParentWidth = FALSE;
	m_bParentHeight = FALSE;
	m_original_xpos = 0.0;
	m_original_ypos = 0.0;
	m_original_cx = 1.0;
	m_original_cy = 1.0;
	m_fixedWidth = 0;
	m_fixedHeight = 0;
	m_fixed_min_width = 0;
	m_fixed_min_height = 0;
	m_fixed_max_width = DEFAULT_MAX_SIZE;
	m_fixed_max_height = DEFAULT_MAX_SIZE;

	CreateBackground ();
	
	m_clrBackground = ::GetSysColor (COLOR_BTNFACE);
	m_clrSplitter = ::GetSysColor (COLOR_BTNFACE);
}

CPanel::CPanel (const TCHAR* pName, double size)
{
	m_windowAlign = 0;
	m_bSizeRelativeToWindow = TRUE;
	m_bHide = FALSE;
	m_bDrawDebugInfo = TRUE;
	m_bEnableSplitter = FALSE;
	m_bMouseTransparentBkg = TRUE;
	m_hBaseWnd = NULL;
	m_hWnd = NULL;
	m_pParent = NULL;
	m_pSplitter = NULL;
	m_rcControlsBound.SetRectEmpty ();

	SetBorder (DEFAULT_BORDER_SIZE);
	SetName (pName);

	m_align = ALIGN_NOWHERE;
	m_bFixedWidth = FALSE;
	m_bFixedHeight = FALSE;
	m_bParentWidth = FALSE;
	m_bParentHeight = FALSE;
	
	m_original_xpos = 0.0;
	m_original_ypos = 0.0;
	m_original_cx = size;
	m_original_cy = size;
	m_fixedWidth = 0;
	m_fixedHeight = 0;
	m_fixed_min_width = 0;
	m_fixed_min_height = 0;
	m_fixed_max_width = DEFAULT_MAX_SIZE;
	m_fixed_max_height = DEFAULT_MAX_SIZE;

	m_clrBackground = ::GetSysColor (COLOR_BTNFACE);
	m_clrSplitter = ::GetSysColor (COLOR_BTNFACE);
}

CPanel::CPanel (const TCHAR* pName, int size)
{
	m_windowAlign = 0;
	m_bSizeRelativeToWindow = TRUE;
	m_bHide = FALSE;
	m_bDrawDebugInfo = TRUE;
	m_bEnableSplitter = FALSE;
	m_bMouseTransparentBkg = TRUE;
	m_hBaseWnd = NULL;
	m_hWnd = NULL;
	m_pParent = NULL;
	m_pSplitter = NULL;
	m_rcControlsBound.SetRectEmpty ();
	SetBorder (DEFAULT_BORDER_SIZE);
	
	SetName (pName);

	m_align = ALIGN_NOWHERE;
	m_bFixedWidth = FALSE;
	m_bFixedHeight = FALSE;
	m_bParentWidth = FALSE;
	m_bParentHeight = FALSE;
	
	m_original_xpos = 0.0;
	m_original_ypos = 0.0;
	m_original_cx = 0.0;
	m_original_cy = 0.0;
	m_fixedWidth = size;
	m_fixedHeight = size;
	m_fixed_min_width = 0;
	m_fixed_min_height = 0;
	m_fixed_max_width = DEFAULT_MAX_SIZE;
	m_fixed_max_height = DEFAULT_MAX_SIZE;

	m_clrBackground = ::GetSysColor (COLOR_BTNFACE);
	m_clrSplitter = ::GetSysColor (COLOR_BTNFACE);
}

CPanel::~CPanel ()
{
	while (!m_childList.IsEmpty ())
		delete m_childList.RemoveTail ();
}



//////////////////////////////////////////////////////////////////////
void CPanel::OnSetAlign (int align)
{
	// Do nothing in base panel
}



//////////////////////////////////////////////////////////////////////
void CPanel::OnSetParent ()
{
	// Do nothing in base panel
}



//////////////////////////////////////////////////////////////////////
void CPanel::OnDraw (CDC* pDC, CRect& rcBound)
{
	// Draw edged rectangle
	CBrush br;
	br.CreateSolidBrush (m_clrBackground);

	CBrush* pOld = pDC->SelectObject (&br);
	pDC->PatBlt (rcBound.left, rcBound.top, rcBound.Width (), rcBound.Height (), PATCOPY);
	pDC->SelectObject (pOld);

	// Draw eged frame
	/*
	CRect rcNorm = rcBound;
	rcNorm.NormalizeRect ();
	
	COLORREF clrLight = RGB (min (GetRValue (m_clrBackground) + 200, 255)
		, min (GetGValue (m_clrBackground) + 200, 255)
		, min (GetBValue (m_clrBackground) + 200, 255));
	COLORREF clrShadow = RGB (max (GetRValue (m_clrBackground) - 200, 0)
		, max (GetGValue (m_clrBackground) - 200, 0)
		, max (GetBValue (m_clrBackground) - 200, 0));

	pDC->Draw3dRect (rcNorm, clrLight, clrShadow);
	*/

	// Check is need to draw debug info
	// If panel has at least one window - return
	if (!m_bDrawDebugInfo)
		return;

	pDC->MoveTo (rcBound.left, rcBound.top);
	pDC->LineTo (rcBound.right, rcBound.bottom);
	pDC->MoveTo (rcBound.left, rcBound.bottom);
	pDC->LineTo (rcBound.right, rcBound.top);

	// Draw panel name or child level
	int nChild = 0;
	
	CPanel* pPanel;
	pPanel = m_pParent;

	while (pPanel)
	{
		nChild++;
		pPanel = pPanel->m_pParent;
	}

	CString str ("Parent");

	if (nChild)
	{
		if (!m_name[0])
			str.Format ("Child %i", nChild);
		else
			str = m_name;
	}

	pDC->SetTextColor (RGB (0, 0, 0));
	pDC->SetBkColor (RGB (255, 255, 255));
	pDC->DrawText (str, &rcBound, DT_CENTER|DT_VCENTER|DT_SINGLELINE);
	
}



//////////////////////////////////////////////////////////////////////
void CPanel::OnDrawSplitter (CDC* pDC, CRect& rcBound)
{
	// Draw panel pplitter
	CBrush br;
	br.CreateSolidBrush (m_clrSplitter);

	CBrush* pOld = pDC->SelectObject (&br);
	pDC->PatBlt (rcBound.left, rcBound.top, rcBound.Width (), rcBound.Height (), PATCOPY);
	pDC->SelectObject (pOld);

	//pDC->DrawEdge (&rcBound, EDGE_RAISED, BF_RECT);
	COLORREF clrLight = RGB (min (GetRValue (m_clrBackground) + 200, 255)
		, min (GetGValue (m_clrBackground) + 200, 255)
		, min (GetBValue (m_clrBackground) + 200, 255));
	COLORREF clrShadow = RGB (max (GetRValue (m_clrBackground) - 200, 0)
		, max (GetGValue (m_clrBackground) - 200, 0)
		, max (GetBValue (m_clrBackground) - 200, 0));

	pDC->Draw3dRect (rcBound, clrLight, clrShadow);
}



//////////////////////////////////////////////////////////////////////
void CPanel::OnMeasureRect (CRect* pRect)
{
	// Don't change default draw rectangle
}



//////////////////////////////////////////////////////////////////////
void CPanel::OnRecalcWindowsZorder ()
{
	// Reposition all panel windows by z-order

	if (!m_bFromDialog)
		// If only one window
		::BringWindowToTop (m_hWnd);
	else
	{
		// If dialog controls
		POSITION pos = m_ctrlsInfoList.GetHeadPosition ();

		while (pos)
		{
			// Get next control info from ealier saved list
			CTRLINFO ctrlInfo = m_ctrlsInfoList.GetNext (pos);
			::BringWindowToTop (ctrlInfo.hCtrl);
		}
	}

	// In replace methods you must call base member 
	// CPanel::OnRecalcWindowsZorder ()
}



//////////////////////////////////////////////////////////////////////
void CPanel::OnResizeWindows ()
{
	// Do nothing by default
}



//////////////////////////////////////////////////////////////////////
void CPanel::OnShow (BOOL bShow)
{
	// Do nothing by default
}



//////////////////////////////////////////////////////////////////////
void CPanel::Add (CPanel* pPanel, int align, BOOL bAtHead)
{
	// Add new panel

	ASSERT (CanInsertPanel (pPanel));
		
	// Set new panel as child of this panel
	pPanel->SetParent (this);

	// If flag set recalc new panel sizes relative to this
	if (!pPanel->m_bSizeRelativeToWindow)
		pPanel->SetSizeRelativeToParentPanel ();

	ApplyAlignment (align, pPanel);

	// Restrict size of this panel and add new panel to child list
	pPanel->SetAlign (align);

	if (bAtHead)
		m_childList.AddHead (pPanel);
	else
		m_childList.AddTail (pPanel);

	// Add splitter if neccesary
	pPanel->AddSplitter ();
}



//////////////////////////////////////////////////////////////////////
void CPanel::AddToRight (CPanel* pPanel, BOOL bAtHead)
{
	// Add new panel to the right of this panel
	Add (pPanel, ALIGN_RIGHT, bAtHead);
}



//////////////////////////////////////////////////////////////////////
void CPanel::AddToLeft (CPanel* pPanel, BOOL bAtHead)
{
	// Add new panel to the left of this panel
	Add (pPanel, ALIGN_LEFT, bAtHead);
}



//////////////////////////////////////////////////////////////////////
void CPanel::AddToBottom (CPanel* pPanel, BOOL bAtHead)
{
	// Add new panel to the bottom of this panel
	Add (pPanel, ALIGN_BOTTOM, bAtHead);
}



//////////////////////////////////////////////////////////////////////
void CPanel::AddToTop (CPanel* pPanel, BOOL bAtHead)
{
	// Add new panel to the bottom of this panel
	Add (pPanel, ALIGN_TOP, bAtHead);
}



//////////////////////////////////////////////////////////////////////
void CPanel::AddBefor (CPanel* pPanel)
{
	// Add new panel befor this panel
	ASSERT (CanInsertPanel (pPanel));
	ASSERT (GetParent ());

	// Set new panel as child of parent of this panel
	pPanel->SetParent (GetParent ());

	// If flag set recalc new panel sizes relative to this
	if (!pPanel->m_bSizeRelativeToWindow)
		pPanel->SetSizeRelativeToParentPanel ();

	ApplyAlignment (GetAlign (), pPanel);

	// Restrict size of this panel and add new panel to child list
	pPanel->SetAlign (GetAlign ());

	GetParent ()->m_childList.InsertBefore (GetParent ()->m_childList.Find (this), pPanel);
	
	// Add splitter if neccesary
	pPanel->AddSplitter ();
}



//////////////////////////////////////////////////////////////////////
void CPanel::AddAfter (CPanel* pPanel)
{
	// Add new panel after this panel
	ASSERT (CanInsertPanel (pPanel));
	ASSERT (GetParent ());

	// Set new panel as child of parent of this panel
	pPanel->SetParent (GetParent ());

	// If flag set recalc new panel sizes relative to this
	if (!pPanel->m_bSizeRelativeToWindow)
		pPanel->SetSizeRelativeToParentPanel ();

	ApplyAlignment (GetAlign (), pPanel);

	// Restrict size of this panel and add new panel to child list
	pPanel->SetAlign (GetAlign ());

	GetParent ()->m_childList.InsertAfter (GetParent ()->m_childList.Find (this), pPanel);
	
	// Add splitter if neccesary
	pPanel->AddSplitter ();
}

void CPanel::ApplyAlignment (int align, CPanel* pPanel)
{
	// Initialize dimensions
	ASSERT (ALIGN_NOWHERE);

	if (align == ALIGN_LEFT || align == ALIGN_RIGHT)
	{
		if (pPanel->m_original_cx == 0.0)
		{
			pPanel->m_fixed_min_width = pPanel->m_fixedWidth;
			pPanel->m_bFixedWidth = TRUE;
		}
		else
		{
			pPanel->m_bFixedWidth = FALSE;
		}

		pPanel->m_bFixedHeight = FALSE;
		pPanel->m_bParentHeight = TRUE;
	}
	else
	{
		if (pPanel->m_original_cy == 0.0)
		{
			pPanel->m_fixed_min_height = pPanel->m_fixedHeight;
			pPanel->m_bFixedHeight = TRUE;
		}
		else
		{
			pPanel->m_bFixedHeight = FALSE;
		}

		pPanel->m_bFixedWidth = FALSE;
		pPanel->m_bParentWidth = TRUE;
	}
}


//////////////////////////////////////////////////////////////////////
void CPanel::SetSizeRelativeToParentPanel ()
{
	m_original_cx = fabs (m_pParent->m_original_cx) * m_original_cx;
	m_original_cy = fabs (m_pParent->m_original_cy) * m_original_cy;
}



//////////////////////////////////////////////////////////////////////
void CPanel::SetAlign (int align)
{
	OnSetAlign (align);

	m_align = align;

	m_xpos = m_original_xpos;
	m_ypos = m_original_ypos;
	m_cx = m_original_cx;
	m_cy = m_original_cy;

	// if one dimension is zero we assume parent dimension
	if (m_bParentWidth)
		m_cx = m_pParent->m_cx;

	if (m_bParentHeight)
		m_cy = m_pParent->m_cy;

	// No parent - no align
	if (!m_pParent)
	{
		NormalizePanel (m_xpos, m_ypos, m_cx, m_cy);
		return;
	}
	
	// Align panel
	switch (m_align)
	{
	case ALIGN_RIGHT:
		//m_xpos = m_pParent->m_xpos + m_pParent->m_cx; // for '-' support
		m_xpos = m_pParent->m_xpos + m_pParent->m_cx - m_cx;
		m_ypos = m_pParent->m_ypos;
		break;
	case ALIGN_LEFT:
		//m_xpos = m_pParent->m_xpos - m_cx; // for '-' support
		m_xpos = m_pParent->m_xpos;
		m_ypos = m_pParent->m_ypos;
		break;
	case ALIGN_BOTTOM:
		m_xpos = m_pParent->m_xpos;
		//m_ypos = m_pParent->m_ypos + m_pParent->m_cy; // for '-' support
		m_ypos = m_pParent->m_ypos + m_pParent->m_cy - m_cy;
		break;
	case ALIGN_TOP:
		m_xpos = m_pParent->m_xpos;
		//m_ypos = m_pParent->m_ypos - m_cy; // for '-' support
		m_ypos = m_pParent->m_ypos; 
		break;
	default:
		ASSERT (FALSE);
		break;
	}

	NormalizePanel (m_xpos, m_ypos, m_cx, m_cy);

	// Clip parent panel if necessary
	switch (m_align)
	{
	case ALIGN_RIGHT:
		if (m_xpos < m_pParent->m_xpos + m_pParent->m_cx)
			m_pParent->m_cx = m_pParent->m_cx - m_cx;
		break;
	case ALIGN_LEFT:
		if (m_xpos + m_cx > m_pParent->m_xpos)
		{
			m_pParent->m_xpos = m_xpos + m_cx;
			m_pParent->m_cx = m_pParent->m_cx - m_cx;
		}
		break;
	case ALIGN_BOTTOM:
		if (m_ypos < m_pParent->m_ypos + m_pParent->m_cy)
			m_pParent->m_cy = m_pParent->m_cy - m_cy;
		break;
	case ALIGN_TOP:
		if (m_ypos + m_cy > m_pParent->m_ypos)
		{
			m_pParent->m_ypos = m_ypos + m_cy;
			m_pParent->m_cy = m_pParent->m_cy - m_cy;
		}
		break;
	default:
		ASSERT (FALSE);
		break;
	}
}	



//////////////////////////////////////////////////////////////////////
void CPanel::SetParent (CPanel* pParent)
{
	ASSERT (pParent);

	m_hBaseWnd = pParent->m_hBaseWnd;
	m_pParent = pParent;
	m_clrBackground = pParent->m_clrBackground;
	m_clrSplitter = pParent->m_clrSplitter;
	
	// If has childs panels correct their z-order
	POSITION pos = m_childList.GetHeadPosition ();
	
	while (pos)
	{
		CPanel* pChild = m_childList.GetNext (pos);	
		pChild->SetParent (this);
	}

	CreateBackground ();

	OnSetParent ();
}



//////////////////////////////////////////////////////////////////////
void CPanel::DrawToWindow (BOOL bInvalidateAll)
{
	// If no base window - return
	if (!m_hBaseWnd)
		return;

	if (m_bHide)
		return;
	
	POSITION pos = m_childList.GetHeadPosition ();
	
	// If we has at least one child panel, draw their first..
	while (pos)
	{
		CPanel* pChild = m_childList.GetNext (pos);	
		pChild->DrawToWindow (bInvalidateAll);
	}

	// Redraw own panel and controls
	DWORD swpFlags = RDW_FRAME|RDW_UPDATENOW|RDW_ERASENOW;

	if (bInvalidateAll)
		swpFlags |= RDW_INVALIDATE|RDW_ERASE;

	// Draw panel debug information
	m_backgroundWnd.RedrawWindow (NULL, NULL, swpFlags);
	
	// redraw build-in windows
	// If dialog controls
	if (m_bFromDialog)
	{
		POSITION pos = m_ctrlsInfoList.GetHeadPosition ();

		while (pos)
		{
			CTRLINFO& ctrlInfo = m_ctrlsInfoList.GetNext (pos);
			::RedrawWindow (ctrlInfo.hCtrl, NULL, NULL, swpFlags);
		}
	}
	// If one window
	else
		::RedrawWindow (m_hWnd, NULL, NULL, swpFlags);
}



//////////////////////////////////////////////////////////////////////
void CPanel::SetFixedWidth (int width)
{
	ASSERT (m_bFixedWidth);
	m_fixedWidth = width;

	if (m_fixedWidth < m_fixed_min_width)
		m_fixedWidth = m_fixed_min_width;
	if (m_fixedWidth > m_fixed_max_width)
		m_fixedWidth = m_fixed_max_width;
}



//////////////////////////////////////////////////////////////////////
void CPanel::SetFixedHeight (int height)
{
	ASSERT (m_bFixedHeight);
	m_fixedHeight = height;

	if (m_fixedHeight < m_fixed_min_height)
		m_fixedHeight = m_fixed_min_height;
	if (m_fixedHeight > m_fixed_max_height)
		m_fixedHeight = m_fixed_max_height;
}



//////////////////////////////////////////////////////////////////////
int	CPanel::GetFixedWidth ()
{
	ASSERT (m_bFixedWidth);
	return m_fixedWidth;
}



//////////////////////////////////////////////////////////////////////
int CPanel::GetFixedHeight ()
{
	ASSERT (m_bFixedHeight);
	return m_fixedHeight;
}




//////////////////////////////////////////////////////////////////////
void CPanel::ResizeToWindow ()
{
	// If no base window - return
	if (!m_hBaseWnd)
		return;

	// Resize layout relative to specified window 
	CRect rcClient;
	::GetClientRect (m_hBaseWnd, &rcClient);

	if (m_bHide)
		return;

	// Correct panel size if fixed at least one side
	if (m_bFixedWidth)
	{
		m_original_cx = 0.0;

		if (rcClient.Width ())
			m_original_cx = (double)(m_fixedWidth) / rcClient.Width ();
	}
	
	if (m_bFixedHeight)
	{
		m_original_cy = 0.0;

		if (rcClient.Height ())
			m_original_cy = (double)(m_fixedHeight) / rcClient.Height ();
	}

	// Set align it self first
	SetAlign (GetAlign ());

	// If at least one child exists then realign it
	POSITION pos = m_childList.GetHeadPosition ();
	while (pos)
	{
		CPanel* pChild = m_childList.GetNext (pos);
		pChild->ResizeToWindow ();
	}

	// Resize child window if exist
	SetWindowPos ();

	// Resize background window
	SetBackgroundPos ();

	// Call virtual event function
	OnResizeWindows ();
}



//////////////////////////////////////////////////////////////////////
void CPanel::RecalcWindowsZorder ()
{
	// If no base window - return
	if (!m_hBaseWnd)
		return;

	if (m_bHide)
		return;

	// recalculate background window z-order
	m_backgroundWnd.BringWindowToTop ();

	// If any window inserted
	if (m_hWnd)
		OnRecalcWindowsZorder ();

	// Recalc windows z-orders in back order
	POSITION pos = m_childList.GetTailPosition ();
	while (pos)
	{
		CPanel* pChild = m_childList.GetPrev (pos);
		pChild->RecalcWindowsZorder ();
	}
}



//////////////////////////////////////////////////////////////////////
BOOL CPanel::IsUnique (CPanel* pPanel)
{
	// Find is pPanel unique
	BOOL bUnique = TRUE;

	if (pPanel == NULL || pPanel == this)
		return FALSE;

	POSITION pos = m_childList.GetHeadPosition ();
	
	while (pos)
	{
		CPanel* pChild = m_childList.GetNext (pos);
		
		if (!pChild->IsUnique (pPanel))
		{
			bUnique = FALSE;
			break;
		}
	}

	return bUnique;
}



//////////////////////////////////////////////////////////////////////
BOOL CPanel::IsUniqueName (CPanel* pPanel)
{
	// Find is pPanel name unique
	BOOL bUniqueName = TRUE;

	if (pPanel->GetPanel (GetName ()))
		return FALSE;

	POSITION pos = m_childList.GetHeadPosition ();
	
	while (pos)
	{
		CPanel* pChild = m_childList.GetNext (pos);
		
		if (!pChild->IsUniqueName (pPanel))
		{
			bUniqueName = FALSE;
			break;
		}
	}

	return bUniqueName;
}



//////////////////////////////////////////////////////////////////////
CPanel*	CPanel::GetTopLevelParent ()
{
	CPanel* pParent = m_pParent;

	if (!pParent)
		return this;

	while (pParent->m_pParent)
		pParent = pParent->m_pParent;
	
	return pParent;
}



//////////////////////////////////////////////////////////////////////
BOOL CPanel::CanInsertPanel (CPanel* pPanel)
{
	BOOL bIsUnique = GetTopLevelParent ()->IsUnique (pPanel);
	ASSERT (bIsUnique);
	BOOL bIsUniqueName = IsUniqueName (pPanel);
	ASSERT (bIsUniqueName);

	return bIsUnique && bIsUniqueName;
}



//////////////////////////////////////////////////////////////////////
void CPanel::InsertWindow (CWnd* pWnd, int windowAlign)
{
	m_hWnd = NULL;
	if (pWnd)
		m_hWnd = pWnd->m_hWnd;

	m_bFromDialog = FALSE;
	m_firstID = -1;
	m_lastID = -1;
	m_windowAlign = windowAlign;

	::SetWindowLong (m_hWnd, GWL_STYLE, ::GetWindowLong (m_hWnd, GWL_STYLE) | WS_CLIPSIBLINGS);

	ClearDialogCtrlsInfo ();
}



//////////////////////////////////////////////////////////////////////
void CPanel::InsertWindow (CWnd* pWnd, int ctrlID, int windowAlign)
{
	ASSERT (pWnd);

	m_hWnd = pWnd->GetDlgItem (ctrlID)->m_hWnd;
	m_bFromDialog = FALSE;
	m_firstID = -1;
	m_lastID = -1;
	m_windowAlign = windowAlign;

	ClearDialogCtrlsInfo ();
}

//////////////////////////////////////////////////////////////////////
void CPanel::InsertWindow (CWnd* pWnd, int firstID, int lastID, int windowAlign)
{
	ASSERT (pWnd);

	m_hWnd = pWnd->m_hWnd;
	m_bFromDialog = TRUE;
	m_firstID = firstID;
	m_lastID = lastID;
	m_windowAlign = windowAlign;

	ClearDialogCtrlsInfo ();
	CreateDialogCtrlsInfo ();
}



//////////////////////////////////////////////////////////////////////
void CPanel::AddBorder (CRect* pRcPanel)
{
	pRcPanel->left += m_rcBorder.left;
	pRcPanel->top += m_rcBorder.top;
	pRcPanel->right -= m_rcBorder.right;
	pRcPanel->bottom -= m_rcBorder.bottom;
}



//////////////////////////////////////////////////////////////////////
void CPanel::SetWindowPos ()
{
	if (m_hWnd)
	{
		if (!::IsWindow (m_hWnd))
		{
			ASSERT (FALSE);
			return;
		}

		if (!m_bFromDialog)
			SetOneWindowPos ();
		else
			SetDialogWindowsPos ();
	}
}



//////////////////////////////////////////////////////////////////////
void CPanel::NormalizePanel (double& m_xpos, double& m_ypos, double& m_cx, double& m_cy)
{
	double xtail = m_xpos + m_cx;
	double ytail = m_ypos + m_cy;

	if (m_xpos > xtail)
	{
		double x = m_xpos;
		m_xpos = xtail;
		xtail = x;
	}

	if (m_ypos > ytail)
	{
		double y = m_ypos;
		m_ypos = ytail;
		ytail = y;
	}

	m_cx = xtail - m_xpos;
	m_cy = ytail - m_ypos;
}



//////////////////////////////////////////////////////////////////////
void CPanel::SetOneWindowPos ()
{
	CRect rcWindow;
	::GetWindowRect (m_hWnd, &rcWindow);

	// Get panel rect in pexels
	CRect rcPanel = GetRect ();
	// Add border
	AddBorder (&rcPanel);
	
	// Initialize window position relative panel
	int windowX = 0;
	int windowY = 0;
	int windowWidth = rcWindow.Width ();
	int windowHeight = rcWindow.Height ();
	
	// Culc new window position
	switch (m_windowAlign & WINDOW_HSTRETCHED)
	{
	case WINDOW_HCENTERED:
		windowX = rcPanel.left + (rcPanel.Width () - windowWidth) / 2;
		break;
	case WINDOW_RIGHT:
		windowX = rcPanel.left + rcPanel.Width () - windowWidth;
		break;
	case WINDOW_LEFT:
		windowX = rcPanel.left;
		break;
	case WINDOW_HSTRETCHED:
		windowX = rcPanel.left;
		windowWidth = rcPanel.Width ();
		break;
	default:
		ASSERT (FALSE);
		break;
	}

	switch (m_windowAlign & WINDOW_VSTRETCHED)
	{
	case WINDOW_VCENTERED:
		windowY = rcPanel.top + (rcPanel.Height () - windowHeight) / 2;
		break;
	case WINDOW_BOTTOM:
		windowY = rcPanel.top + rcPanel.Height () - windowHeight;
		break;
	case WINDOW_TOP:
		windowY = rcPanel.top;
		break;
	case WINDOW_VSTRETCHED:
		windowY = rcPanel.top;
		windowHeight = rcPanel.Height ();
		break;
	default:
		ASSERT (FALSE);
		break;
	}

	::SetWindowPos (m_hWnd, NULL, windowX, windowY, windowWidth, windowHeight, SWP_NOZORDER|SWP_NOREDRAW);
	//::RedrawWindow (m_hWnd, NULL, NULL, RDW_FRAME|RDW_INVALIDATE|RDW_ERASE/*|RDW_UPDATENOW|RDW_ERASENOW*/);
}



//////////////////////////////////////////////////////////////////////
void CPanel::SetDialogWindowsPos ()
{
	// Align bounding rectangle
	int xoffs = 0;
	int yoffs = 0;

	// Get panel rect in pexels
	CRect rcPanel = GetRect ();
	// Add border
	AddBorder (&rcPanel);
	
	switch (m_windowAlign & WINDOW_HSTRETCHED)
	{
	case WINDOW_HCENTERED:
		xoffs = (rcPanel.Width () - m_rcControlsBound.Width ()) / 2;
		break;
	case WINDOW_RIGHT:
		xoffs = rcPanel.Width () - m_rcControlsBound.Width ();
		break;
	case WINDOW_LEFT:
	case WINDOW_HSTRETCHED:
		break;
	default:
		ASSERT (FALSE);
		break;
	}

	switch (m_windowAlign & WINDOW_VSTRETCHED)
	{
	case WINDOW_VCENTERED:
		yoffs = (rcPanel.Height () - m_rcControlsBound.Height ()) / 2;
		break;
	case WINDOW_BOTTOM:
		yoffs = rcPanel.Height () - m_rcControlsBound.Height ();
		break;
	case WINDOW_TOP:
	case WINDOW_VSTRETCHED:
		break;
	default:
	ASSERT (FALSE);
		break;
	}
	
	// Culc new controls positions	
	POSITION pos = m_ctrlsInfoList.GetHeadPosition ();

	while (pos)
	{
		// Get next control info from ealier saved list
		CTRLINFO ctrlInfo = m_ctrlsInfoList.GetNext (pos);

		int ctrlXPos = ctrlInfo.ctrlRect.left - m_rcControlsBound.left;
		int ctrlYPos = ctrlInfo.ctrlRect.top - m_rcControlsBound.top;
		int ctrlWidth = ctrlInfo.ctrlRect.Width ();
		int ctrlHeight = ctrlInfo.ctrlRect.Height ();
		
		if (m_windowAlign & WINDOW_HSTRETCHED == WINDOW_HSTRETCHED)
		{
			ctrlXPos = ctrlXPos * rcPanel.Width () / m_rcControlsBound.Width ();
			ctrlWidth = ctrlWidth * rcPanel.Width () / m_rcControlsBound.Width ();
		}

		if (m_windowAlign & WINDOW_VSTRETCHED == WINDOW_VSTRETCHED)
		{
			ctrlYPos = ctrlYPos * rcPanel.Height () / m_rcControlsBound.Height ();
			ctrlHeight = ctrlHeight * rcPanel.Height () / m_rcControlsBound.Height ();
		}

		::SetWindowPos (ctrlInfo.hCtrl, NULL, rcPanel.left + xoffs + ctrlXPos, rcPanel.top + yoffs + ctrlYPos, ctrlWidth, ctrlHeight, SWP_NOZORDER|SWP_NOREDRAW);
		//::RedrawWindow (ctrlInfo.hCtrl, NULL, NULL, RDW_FRAME|RDW_INVALIDATE|RDW_ERASE/*|RDW_UPDATENOW|RDW_ERASENOW*/);
	}
}



//////////////////////////////////////////////////////////////////////
void CPanel::SetBackgroundPos ()
{
	// Get panel rect in pexels
	CRect rcPanel = GetRect ();
	
	m_backgroundWnd.SetWindowPos (NULL, rcPanel.left, rcPanel.top, rcPanel.Width (), rcPanel.Height (), SWP_NOZORDER|SWP_NOREDRAW);
	m_backgroundWnd.ModifyStyle (0, WS_VISIBLE, SWP_NOZORDER|SWP_NOREDRAW);
}



//////////////////////////////////////////////////////////////////////
void CPanel::SetBorder (int all)
{
	m_rcBorder.SetRect (all, all, all, all);
}



//////////////////////////////////////////////////////////////////////
void CPanel::SetBorder (int hor, int ver)
{
	m_rcBorder.SetRect (hor, ver, hor, ver);
}



//////////////////////////////////////////////////////////////////////
void CPanel::SetBorder (int left, int top, int right, int bottom)
{
	m_rcBorder.SetRect (left, top, right, bottom);
}



//////////////////////////////////////////////////////////////////////
void CPanel::SetBorder (CRect& rcBorder)
{
	m_rcBorder.CopyRect (rcBorder);
}



//////////////////////////////////////////////////////////////////////
void CPanel::CreateDialogCtrlsInfo ()
{
	// Culculate bounding rectangle of selected dialog controls 
	m_rcControlsBound.left = INT_MAX;
	m_rcControlsBound.right = INT_MIN;
	m_rcControlsBound.top = INT_MAX;
	m_rcControlsBound.bottom = INT_MIN;

	HWND hCtrl = ::GetDlgItem (m_hWnd, m_firstID);
	HWND hLastCtrl = ::GetDlgItem (m_hWnd, m_lastID);

	while (1)
	{
		if (!hCtrl)
		{
			ASSERT (FALSE);
			break;
		}

		CRect rcCtrl;
		::GetWindowRect (hCtrl, &rcCtrl);

		if (rcCtrl.left < m_rcControlsBound.left)
			 m_rcControlsBound.left = rcCtrl.left;

		if (rcCtrl.right > m_rcControlsBound.right)
			 m_rcControlsBound.right = rcCtrl.right;

		if (rcCtrl.top < m_rcControlsBound.top)
			 m_rcControlsBound.top = rcCtrl.top;

		if (rcCtrl.bottom > m_rcControlsBound.bottom)
			 m_rcControlsBound.bottom = rcCtrl.bottom;

		::SetWindowLong (hCtrl, GWL_STYLE, ::GetWindowLong (hCtrl, GWL_STYLE) | WS_CLIPSIBLINGS);

		// Add current control to list for future resizing
		CTRLINFO ci;
		ci.hCtrl = hCtrl;
		ci.ctrlRect = rcCtrl;
		m_ctrlsInfoList.AddTail (ci);

		if (hCtrl == hLastCtrl)
			break;

		hCtrl = ::GetNextWindow (hCtrl, GW_HWNDNEXT);
	}
}



//////////////////////////////////////////////////////////////////////
void CPanel::ClearDialogCtrlsInfo ()
{
	// Remove old control list information
	while (!m_ctrlsInfoList.IsEmpty ())
		m_ctrlsInfoList.RemoveTail ();
}



//////////////////////////////////////////////////////////////////////
void CPanel::CreateBackground ()
{
	ASSERT (m_hBaseWnd);

	CWnd* pParent = CWnd::FromHandle (m_hBaseWnd);

	if (!m_backgroundWnd.m_hWnd)
		m_backgroundWnd.Create (NULL, NULL, WS_CHILD|WS_CLIPSIBLINGS, CRect (0, 0, 0, 0), pParent, 0);
	else
		m_backgroundWnd.SetParent (pParent);
	
	m_backgroundWnd.PutOnPanel (this);
	m_backgroundWnd.SetMouseTransparence (m_bMouseTransparentBkg);
}
	


//////////////////////////////////////////////////////////////////////
void CPanel::SetName (const TCHAR* pName)
{
	int name_len = sizeof (m_name) / sizeof (m_name[0]);
	memset (m_name, 0, name_len * sizeof (TCHAR));

	if (pName)
		_tcsncpy (m_name, pName, name_len);
}



//////////////////////////////////////////////////////////////////////
const TCHAR* CPanel::GetName ()
{
	return (const TCHAR*)m_name;
}



//////////////////////////////////////////////////////////////////////
void CPanel::SetBase (CWnd* pBaseWnd)
{
	if (!pBaseWnd)
	{
		m_hBaseWnd = NULL;
		m_backgroundWnd.SetParent (NULL);
		return;
	}

	m_hBaseWnd = pBaseWnd->m_hWnd;
	m_backgroundWnd.SetParent (pBaseWnd);
}



//////////////////////////////////////////////////////////////////////
CWnd* CPanel::GetBase ()
{
	return CWnd::FromHandle (m_hBaseWnd);
}



//////////////////////////////////////////////////////////////////////
CRect CPanel::GetInitialRect ()
{
	ASSERT (m_hBaseWnd);

	CRect rcParent;
	::GetClientRect (m_hBaseWnd, &rcParent);

	// Recalc window is pWnd coordinates
	CRect rcPanel (m_xpos * rcParent.Width () + 0.5, m_ypos * rcParent.Height () + 0.5, 
		(m_xpos + m_cx) * rcParent.Width () + 0.5, (m_ypos + m_cy) * rcParent.Height () + 0.5);

	rcPanel.NormalizeRect ();

	return rcPanel;
}



//////////////////////////////////////////////////////////////////////
CRect CPanel::GetRect ()
{
	CRect rcPanel;
	rcPanel = GetInitialRect ();

	OnMeasureRect (&rcPanel);

	return rcPanel;
}



//////////////////////////////////////////////////////////////////////
CRect CPanel::GetScreenRect ()
{
	CRect rcPanel = GetRect ();
	
	::ClientToScreen (m_hBaseWnd, &rcPanel.TopLeft ());
	::ClientToScreen (m_hBaseWnd, &rcPanel.BottomRight ());

	return rcPanel;
}



//////////////////////////////////////////////////////////////////////
CPanel* CPanel::GetPanel (const TCHAR* pName)
{
	ASSERT (pName);

	// Is noname return NULL
	if (!pName[0])
		return NULL;

	// Maybe searching panel is this panel
	if (!strcmp (pName, GetName ()))
		return this;
	
	// If no, first try to find panel in child panels
	POSITION pos = m_childList.GetHeadPosition ();

	while (pos)
	{
		CPanel* pChild = m_childList.GetNext (pos);
		
		if (!strcmp (pName, pChild->GetName ()))
			return pChild;
	}

	// If panel not found go to chid find procedure
	pos = m_childList.GetHeadPosition ();

	while (pos)
	{
		CPanel* pChild = m_childList.GetNext (pos);
		CPanel* pPanelByName = pChild->GetPanel (pName);

		if (pPanelByName)
			return pPanelByName;
	}

	// We didn't find specified panel
	return NULL;
}



//////////////////////////////////////////////////////////////////////
CPanel* CPanel::GetPanel (CPoint ptPos)
{
	ASSERT (m_hBaseWnd);

	CRect rcPanel = GetRect ();
	::ClientToScreen (m_hBaseWnd, &rcPanel.TopLeft ());
	::ClientToScreen (m_hBaseWnd, &rcPanel.BottomRight ());
	
	// Maybe point in this panel?
	if (rcPanel.PtInRect (ptPos))
		// If yes return this panel
		return this;

	// If not recursive searching in child panels
	CPanel* pPanel = NULL;
	POSITION pos = m_childList.GetHeadPosition ();
	
	while (pos)
	{
		CPanel* pChild = m_childList.GetNext (pos);
		pPanel = pChild->GetPanel (ptPos);

		// If panel found return
		if (pPanel)
			break;
	}

	return pPanel;
}



//////////////////////////////////////////////////////////////////////
void CPanel::DrawDebugInfo (BOOL bEnabled)
{
	POSITION pos = m_childList.GetHeadPosition ();
	
	while (pos)
	{
		CPanel* pChild = m_childList.GetNext (pos);
		pChild->DrawDebugInfo (bEnabled);
	}

	m_bDrawDebugInfo = bEnabled;
}



//////////////////////////////////////////////////////////////////////
void CPanel::EnableSplitter ()
{
	m_bEnableSplitter = TRUE;
}



//////////////////////////////////////////////////////////////////////
void CPanel::DisableSplitter ()
{
	// Delete splitter
	delete m_pSplitter;
	m_pSplitter = NULL;

	m_bEnableSplitter = FALSE;
}



//////////////////////////////////////////////////////////////////////
void CPanel::AddSplitter ()
{
	if (m_bEnableSplitter)
	{
		m_pSplitter = new CPanelSplitter (CWnd::FromHandle (m_hBaseWnd));

		int reverse_align = m_align;

		switch (m_align)
		{
		case ALIGN_RIGHT: reverse_align = ALIGN_LEFT; break;
		case ALIGN_LEFT: reverse_align = ALIGN_RIGHT; break;
		case ALIGN_BOTTOM: reverse_align = ALIGN_TOP; break;
		case ALIGN_TOP: reverse_align = ALIGN_BOTTOM; break;
		default: ASSERT (FALSE); break;
		}

		Add (m_pSplitter, reverse_align, TRUE);
	}
}



//////////////////////////////////////////////////////////////////////
CPanel*	CPanel::RemovePanel (CPanel* pPanel)
{
	// Try to find in list
	POSITION pos = m_childList.Find (pPanel);
	CPanel* pFoundPanel = NULL;

	// Recurently find and remove specidfied panel
	if (!pos)
	{
		pos = m_childList.GetHeadPosition ();
		
		while (pos)
		{
			pFoundPanel = m_childList.GetNext (pos)->RemovePanel (pPanel);

			if (pFoundPanel)
				break;
		}
	}
	else
	{
		m_childList.RemoveAt (pos);
		pFoundPanel = pPanel;
	}

	// If not found return NULL
	if (!pFoundPanel)
		return NULL;

	pPanel = pFoundPanel;
	
	// Hide all panel windows
	pPanel->Show (FALSE);

	// Delete splitter
	if (pPanel->m_pSplitter)
	{
		delete pPanel->RemovePanel (pPanel->m_pSplitter);
		pPanel->m_pSplitter = NULL;
	}

	return pPanel;
}



//////////////////////////////////////////////////////////////////////
CPanel*	CPanel::RemovePanel (TCHAR* pName)
{
	return RemovePanel (GetPanel (pName));
}



//////////////////////////////////////////////////////////////////////
void CPanel::Show (BOOL bShow)
{
	// Show or hide all layout windows
	POSITION pos = m_childList.GetHeadPosition ();
	while (pos)
	{
		CPanel* pChild = m_childList.GetNext (pos);
		pChild->Show (bShow);
	}

	m_bHide = !bShow;

	int bFlag = bShow ? SW_SHOW : SW_HIDE;

	if (m_hWnd)
	{
		if (!m_bFromDialog)
			// If only one window
			::ShowWindow (m_hWnd, bFlag);
		else
		{
			// If dialog controls
			POSITION pos = m_ctrlsInfoList.GetHeadPosition ();

			while (pos)
			{
				// Get next control info from ealier saved list
				CTRLINFO ctrlInfo = m_ctrlsInfoList.GetNext (pos);
				::ShowWindow (ctrlInfo.hCtrl, bFlag);
			}
		}
	}

	// Show/Hide background window
	if (m_backgroundWnd.m_hWnd)
		m_backgroundWnd.ShowWindow (bFlag);

	// Call event virtual function
	OnShow (bShow);
}



//////////////////////////////////////////////////////////////////////
void CPanel::Redraw (BOOL bInvalidateAll)
{
	ResizeToWindow ();
	RecalcWindowsZorder ();
	DrawToWindow (bInvalidateAll);
}