/*************************************************
*            BK Emulator for Window 3.0          *
*************************************************/

/*
File: MainFrm.cpp
Version: 1.0.0
Written by: Yuriy Kalmykov <kalmykov@stoik.com>
    Copyright (c) 2002-2004 Yuriy Kalmykov

    BK Emulator is a program emulated hardware environment for running
code for BK 0010(01) in different configurations. 
           
    This code may be used in compiled form in any way you desire.
This file or it's parts can't be redistributed without the authors
written consent, but can be modified for your private needs.
    Providing that this notice and the authors name and all copyright
notices remains intact.

    Please, an email me to know how you are using it and where. You can
ask me for any information about this below code or any attendant
knowledge.
    
    This file is provided "as is" with no expressed or implied warranty.
The author accepts no liability for any damage or loss of business that
this product may cause.
*/




// MainFrm.cpp : implementation of the CMainFrame class
//

#include "stdafx.h"
#include "BK.h"

#include "MainFrm.h"
#include "WindowMode.h"
#include "BkSound.h"
#include "Board.h"
#include "Board_MSTD.h"
#include "Board_EXT32.h"
#include "Board_FDD.h"
#include "MSFManager.h"
#include "LoadMemoryDlg.h"
#include "LoadTapeDlg.h"
#include "PrintDlg.h"
#include "ImageManagerDlg.h"
#include "TapeManagerDlg.h"
#include "DebugAddressDlg.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CMainFrame

IMPLEMENT_DYNAMIC(CMainFrame, CFrameWnd)

BEGIN_MESSAGE_MAP(CMainFrame, CFrameWnd)
	//{{AFX_MSG_MAP(CMainFrame)
	ON_WM_CREATE()
	ON_WM_PAINT()
	ON_WM_SIZE()
	ON_WM_ACTIVATE()
	ON_UPDATE_COMMAND_UI(ID_VIEW_TOOLBAR, OnUpdateViewToolbar)
	ON_COMMAND(ID_VIEW_TOOLBAR, OnViewToolbar)
	ON_UPDATE_COMMAND_UI(ID_VIEW_STATUS_BAR, OnUpdateViewStatusBar)
	ON_COMMAND(ID_VIEW_STATUS_BAR, OnViewStatusBar)
	ON_MESSAGE (WM_ENTERMENULOOP, OnEnterMenuLoop)
	ON_MESSAGE (WM_EXITMENULOOP, OnExitMenuLoop)
	ON_MESSAGE (WM_ENTERSIZEMOVE, OnEnterSizeMove)
	ON_MESSAGE (WM_EXITSIZEMOVE, OnExitSizeMove)
	ON_MESSAGE (WM_DISPLAYCHANGE, OnDisplayChange)
	ON_WM_KEYDOWN()
	ON_WM_KEYUP()
	ON_MESSAGE (WM_TAPE_FILE_LOAD, OnEmulateTape)
	ON_MESSAGE (WM_CPU_DEBUGBREAK, OnCpuBreak)
	ON_MESSAGE (WM_DEBUG_MODIFY, OnDebugModify)
	ON_COMMAND(ID_CPU_RUNCPU, OnCpuRuncpu)
	ON_COMMAND(ID_VIEW_FULLSCREENMODE, OnViewFullscreenmode)
	ON_UPDATE_COMMAND_UI(ID_VIEW_COLORMODE, OnUpdateViewColormode)
	ON_COMMAND(ID_VIEW_COLORMODE, OnViewColormode)
	ON_UPDATE_COMMAND_UI(ID_VIEW_ADAPTIVEBWMODE, OnUpdateViewAdaptivebwmode)
	ON_COMMAND(ID_VIEW_ADAPTIVEBWMODE, OnViewAdaptivebwmode)
	ON_UPDATE_COMMAND_UI(ID_OPTIONS_ENABLE_SPEAKER, OnUpdateOptionsEnableSpeaker)
	ON_COMMAND(ID_OPTIONS_ENABLE_SPEAKER, OnOptionsEnableSpeaker)
	ON_UPDATE_COMMAND_UI(ID_OPTIONS_ENABLE_COVOX, OnUpdateOptionsEnableCovox)
	ON_COMMAND(ID_OPTIONS_ENABLE_COVOX, OnOptionsEnableCovox)
	ON_UPDATE_COMMAND_UI(ID_CPU_ACCELERATE, OnUpdateCpuAccelerate)
	ON_COMMAND(ID_CPU_ACCELERATE, OnCpuAccelerate)
	ON_UPDATE_COMMAND_UI(ID_CPU_SLOWDOWN, OnUpdateCpuSlowdown)
	ON_COMMAND(ID_CPU_SLOWDOWN, OnCpuSlowdown)
	ON_UPDATE_COMMAND_UI(ID_CPU_RUNBK001001, OnUpdateCpuRunbk001001)
	ON_COMMAND(ID_CPU_RUNBK001001, OnCpuRunbk001001)
	ON_UPDATE_COMMAND_UI(ID_CPU_RUNBK001001_FOCAL, OnUpdateCpuRunbk001001Focal)
	ON_COMMAND(ID_CPU_RUNBK001001_FOCAL, OnCpuRunbk001001Focal)
	ON_UPDATE_COMMAND_UI(ID_CPU_RUNBK001001_32K, OnUpdateCpuRunbk00100132k)
	ON_COMMAND(ID_CPU_RUNBK001001_32K, OnCpuRunbk00100132k)
	ON_UPDATE_COMMAND_UI(ID_OPTIONS_EMULATE_BKKEYBOARD, OnUpdateOptionsEmulateBkkeyboard)
	ON_COMMAND(ID_OPTIONS_EMULATE_BKKEYBOARD, OnOptionsEmulateBkkeyboard)
	ON_UPDATE_COMMAND_UI(ID_OPTIONS_ENABLE_JOYSTICK, OnUpdateOptionsEnableJoystick)
	ON_COMMAND(ID_OPTIONS_ENABLE_JOYSTICK, OnOptionsEnableJoystick)
	ON_UPDATE_COMMAND_UI(ID_OPTIONS_USE_SAVESDIRECTORY, OnUpdateOptionsUseSavesdirectory)
	ON_COMMAND(ID_OPTIONS_USE_SAVESDIRECTORY, OnOptionsUseSavesdirectory)
	ON_UPDATE_COMMAND_UI(ID_OPTIONS_SHOWPERFOMANCEONSTATUSBAR, OnUpdateOptionsShowperfomanceonstatusbar)
	ON_COMMAND(ID_OPTIONS_SHOWPERFOMANCEONSTATUSBAR, OnOptionsShowperfomanceonstatusbar)
	ON_COMMAND(ID_FILE_LOADSTATE, OnFileLoadstate)
	ON_COMMAND(ID_FILE_SAVESTATE, OnFileSavestate)
	ON_COMMAND(ID_FILE_LOADTAPE, OnFileLoadtape)
	ON_COMMAND(ID_FILE_PRINT, OnFilePrint)
	ON_WM_TIMER()
	ON_UPDATE_COMMAND_UI(ID_DEBUG_BREAK, OnUpdateDebugBreak)
	ON_COMMAND(ID_DEBUG_BREAK, OnDebugBreak)
	ON_COMMAND(ID_CPU_NORMALSPEED, OnCpuNormalspeed)
	ON_UPDATE_COMMAND_UI(ID_CPU_RUNBK001001_FDD16K, OnUpdateCpuRunbk001001Fdd16k)
	ON_COMMAND(ID_CPU_RUNBK001001_FDD16K, OnCpuRunbk001001Fdd16k)
	ON_COMMAND(ID_OPTIONS_DISKIMAGEMANAGER, OnOptionsDiskimagemanager)
	ON_COMMAND(ID_DEBUG_STEPINTO, OnDebugStepinto)
	ON_COMMAND(ID_DEBUG_STEPOVER, OnDebugStepover)
	ON_COMMAND(ID_DEBUG_STEPOUT, OnDebugStepout)
	ON_COMMAND(ID_DEBUG_RUNTOCURSOR, OnDebugRuntocursor)
	ON_UPDATE_COMMAND_UI(ID_DEBUG_STEPINTO, OnUpdateDebugStepinto)
	ON_UPDATE_COMMAND_UI(ID_DEBUG_STEPOVER, OnUpdateDebugStepover)
	ON_UPDATE_COMMAND_UI(ID_DEBUG_STEPOUT, OnUpdateDebugStepout)
	ON_UPDATE_COMMAND_UI(ID_DEBUG_RUNTOCURSOR, OnUpdateDebugRuntocursor)
	ON_UPDATE_COMMAND_UI(ID_VIEW_PRESERVE_CURRENT_MODE, OnUpdateViewPreserveCurrentMode)
	ON_COMMAND(ID_VIEW_PRESERVE_CURRENT_MODE, OnViewPreserveCurrentMode)
	ON_WM_PALETTECHANGED()
	ON_UPDATE_COMMAND_UI(ID_OPTIONS_EMULATE_TAPE_LOADING, OnUpdateOptionsEmulateTapeLoading)
	ON_COMMAND(ID_OPTIONS_EMULATE_TAPE_LOADING, OnOptionsEmulateTapeLoading)
	ON_UPDATE_COMMAND_UI(ID_OPTIONS_EMULATE_TAPE_SAVING, OnUpdateOptionsEmulateTapeSaving)
	ON_COMMAND(ID_OPTIONS_EMULATE_TAPE_SAVING, OnOptionsEmulateTapeSaving)
	ON_UPDATE_COMMAND_UI(ID_FILE_LOADTAPE, OnUpdateFileLoadtape)
	ON_UPDATE_COMMAND_UI(ID_VIEW_TAPE_CONTROL, OnUpdateViewTapeControl)
	ON_COMMAND(ID_VIEW_TAPE_CONTROL, OnViewTapeControl)
	ON_UPDATE_COMMAND_UI(ID_VIEW_DEBUGWINDOW, OnUpdateViewDebugwindow)
	ON_COMMAND(ID_VIEW_DEBUGWINDOW, OnViewDebugwindow)
	ON_UPDATE_COMMAND_UI(ID_VIEW_MEMORYDUMP, OnUpdateViewMemorydump)
	ON_COMMAND(ID_VIEW_MEMORYDUMP, OnViewMemorydump)
	ON_UPDATE_COMMAND_UI(ID_DEBUG_SETADDRESS, OnUpdateDebugSetaddress)
	ON_COMMAND(ID_DEBUG_SETADDRESS, OnDebugSetaddress)
	ON_COMMAND(ID_EDIT_TAPE_RECORD, OnEditTapeRecord)
	ON_WM_MEASUREITEM()
	ON_COMMAND(ID_OPTIONS_TAPEMANAGER, OnOptionsTapemanager)
	ON_WM_COPYDATA()
	ON_MESSAGE (WM_MAINLOOP, OnMainLoop)
	ON_COMMAND(ID_DEBUG_BREAKPOINT, OnDebugBreakpoint)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

static UINT indicators[] =
{
	ID_SEPARATOR,           // status line indicator
	ID_INDICATOR_CAPS,
	ID_INDICATOR_NUM,
	ID_INDICATOR_SCRL,
};

/////////////////////////////////////////////////////////////////////////////
// CMainFrame construction/destruction

CMainFrame::CMainFrame()
: m_mtInstance (FALSE, _T("BK Emulator Instance")), m_dropTarget (this)
{
	m_pLayout = NULL;
	m_bFloatersInitialized = FALSE;
	m_bMenuLoop = FALSE;

	m_pChip = NULL;
	m_pSound = NULL;
	m_pSampleBuffer = new BYTE[10000];
	memset (m_pSampleBuffer, 0, 1000);
	
	m_bPauseCPUAfterStart = FALSE;
	m_CPURunAddr = 0;
	m_cpu_ips = CPU_TICKS_PER_SET;
	m_cpu_sps = CPU_SETS_PER_SECOND;
	m_initialCPUSpeed = 1.0;
	m_bIsAr2Press = FALSE;
	m_bIsShiftPress = FALSE;
	m_bIsKeyPress = FALSE;
	m_bBKKeyboard = TRUE;
	m_bJoystick = TRUE;
	m_bSavesDefault = FALSE;
	m_bShowPerfomance = TRUE;
	m_bEnableSaveCurrentMode = FALSE;
	m_bEmulateLoadTape = FALSE;
	m_bEmulateSaveTape = FALSE;
	m_bTapeAutoBeginDetection = TRUE;
	m_bTapeAutoEnsDetection = TRUE;
	m_bAskForBreak = TRUE;
	m_bShowDumpWindow = FALSE;
	m_bShowDebugWindow = FALSE;

	m_strBinPath = GetCurrentPath () + _T("bin\\");
	m_strToolsPath = GetCurrentPath () + _T("tools\\");
	m_strMemPath = GetCurrentPath () + _T("memory\\");
	m_strSavesPath = GetCurrentPath () + _T("usersaves\\");
	m_strTapePath = GetCurrentPath () + _T("tapes\\");
	m_strScriptsPath = GetCurrentPath () + _T("scripts\\");

	CString strIniFileName;
	strIniFileName.LoadString (IDS_INI_FILENAME);
	m_strIniFilePath = GetCurrentPath () + strIniFileName;

	m_bPrintScreen = TRUE;
	m_bPrintInverse = TRUE;
	m_strPrintTitle.LoadString (IDS_PRINT_TITLE);
}

CMainFrame::~CMainFrame()
{	
	delete m_pChip;

	delete m_pSampleBuffer;
	m_pSampleBuffer = NULL;

	delete m_pLayout;
	delete m_pSound;

	m_mtInstance.Unlock ();
}



/////////////////////////////////////////////////////////////////////////////
int CMainFrame::OnCreate(LPCREATESTRUCT lpCreateStruct)
{
	if (CFrameWnd::OnCreate(lpCreateStruct) == -1)
		return -1;

	// Initialize main frame 
	SetIcon (::LoadIcon (AfxGetInstanceHandle (), MAKEINTRESOURCE (IDR_MAINFRAME)), FALSE);
	SetIcon (::LoadIcon (AfxGetInstanceHandle (), MAKEINTRESOURCE (IDR_MAINFRAME)), TRUE);
	
	//m_mainMenu.Create (IDR_MAINFRAME, this);

	if (!m_wndToolBar.CreateEx (this, TBSTYLE_WRAPABLE|TBSTYLE_FLAT) ||
		!m_wndToolBar.LoadToolBar(IDR_MAINFRAME))
	{
		TRACE0("Failed to create toolbar\n");
		return -1;      // fail to create
	}
	
	if (!m_wndStatusBar.Create(this) ||
		!m_wndStatusBar.SetIndicators(indicators,
		  sizeof(indicators)/sizeof(UINT)))
	{
		TRACE0("Failed to create status bar\n");
		return -1;      // fail to create
	}
	CBKDialog::ChangeCommonDialogFont (&m_wndStatusBar);
	
	// TODO: Remove this if you don't want tool tips
	m_wndToolBar.SetBarStyle (m_wndToolBar.GetBarStyle() |
		CBRS_TOOLTIPS|CBRS_FLYBY|CBRS_SIZE_DYNAMIC);
	m_wndToolBar.SetHeight (15);
	
	// Create Screen
	m_screen.Create (NULL, NULL, WS_VISIBLE|WS_CHILD, CRect (0, 0, 512, 512), this, 0);
	m_screen.ModifyStyleEx (0, WS_EX_CLIENTEDGE);

	// Create Debug Window
	m_debugCtrl.Create (WS_VISIBLE|WS_CHILD|WS_BORDER, CRect (0, 0, 0, 0), this, 0);
	m_debugCtrl.AttachDebugger (&m_debugger);
	m_debugger.AttachWnd (&m_debugCtrl);
	CBKDialog::ChangeCommonDialogFont (&m_debugCtrl);
	
	// Create Memory Dump Window
	m_memoryCtrl.Create (IDD_MEMORY_DUMP, this);
	m_dropTarget.Register (this);

	// Initialize emulator
	InitEmulator ();
	InitRegistry ();
	
	CString strLine = GetCommandLine ();
	if (!ParseCommandLineParameters (strLine))
		return -1;

	m_script.SetScriptPath (m_strScriptsPath + m_strScriptFileName);

	// Create Tape Control Dialog
	m_tapeCtrl.m_strTapePath = m_strTapePath;
	m_tapeCtrl.m_bAutoBeginDetection = m_bTapeAutoBeginDetection;
	m_tapeCtrl.m_bAutoEnddetection = m_bTapeAutoEnsDetection;
	m_tapeCtrl.Create (IDD_TAPE_CONTROL, this);
	m_tapeCtrl.SetTape (&m_tape);

	// Create Main Frame layout amd add child windows to it
	InitLayout ();
	InitMenu ();

	// Init BK Chip
	if (!m_strMemFileName.IsEmpty ())
	{
		if (!LoadMemoryState (m_strMemPath + m_strMemFileName))
			// If cant load msf file
			ConfigurationConstructor (MSF_CONF_BK1001); // Create default configuration
	}
	else
		// If no msf file specified
		ConfigurationConstructor (MSF_CONF_BK1001); // Create default configuration

	m_tapeCtrl.SetScreen (&m_screen);
	
	// Init BK Sound
	m_pSound = new CBkSound (this, m_cpu_sps);


	if (!m_strTapeFileName.IsEmpty ())
	{
		StartPlayTape (m_strTapePath + m_strTapeFileName);
	}
	
	m_tape.SetPlayWavePos (0);
	TAPE_FILE_INFO tfi;

	//m_tape.LoadWaveFile (GetCurrentPath () + "..\\capture\\pereval.wav");
	//m_tape.SaveMSFFile (GetCurrentPath () + "..\\capture\\pereval.tap");

	
	//m_tape.GetWaveFile (&tfi);
	//BOOL bCRC = m_tape.CalcCRC (&tfi, TRUE);

	//m_tape.SaveBinFile (GetCurrentPath () + "bin\\sheriff.bin", &tfi);
	//m_tape.SaveWaveToFile (GetCurrentPath () + "tapes\\sherif_new.wav");

	return 0;
}

BOOL CMainFrame::PreCreateWindow(CREATESTRUCT& cs)
{
	if( !CFrameWnd::PreCreateWindow(cs) )
		return FALSE;
	
	//	Modify the Window class or styles
	cs.dwExStyle &= ~WS_EX_CLIENTEDGE;
	cs.lpszClass = _T("BK Emulator Main Window");
	UINT nClassStyle = CS_HREDRAW|CS_VREDRAW;

	WNDCLASS wndcls;
	if (::GetClassInfo (AfxGetInstanceHandle (), cs.lpszClass, &wndcls))
	{
		// already registered, assert everything is good
		ASSERT (wndcls.style == nClassStyle);

		if (!m_mtInstance.Lock (0))
			cs.lpszClass = _T("Other BK Emulator Main Window");
	}
	
	// otherwise we need to register a new class
	wndcls.style = CS_HREDRAW|CS_VREDRAW;
	wndcls.lpfnWndProc = ::DefWindowProc;
	wndcls.cbClsExtra = wndcls.cbWndExtra = 0;
	wndcls.hInstance = AfxGetInstanceHandle ();
	wndcls.hIcon = NULL;
	wndcls.hCursor = NULL;
	wndcls.hbrBackground = NULL;
	wndcls.lpszMenuName = NULL;
	wndcls.lpszClassName = cs.lpszClass;

	if (!AfxRegisterClass (&wndcls))
		return FALSE;

	return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
// CMainFrame diagnostics

#ifdef _DEBUG
void CMainFrame::AssertValid() const
{
	CFrameWnd::AssertValid();
}

void CMainFrame::Dump(CDumpContext& dc) const
{
	CFrameWnd::Dump(dc);
}

#endif //_DEBUG


/////////////////////////////////////////////////////////////////////////////
void CMainFrame::InitLayout ()
{
	// Create main layout
	delete m_pLayout;
	m_pLayout = new CPanel (this);

	// Add Screen to layout
	m_pLayout->InsertWindow (&m_screen, WINDOW_STRETCHED);

	// Create Toolbar floater
	m_toolBarFloater.CreateFloater (this, m_pLayout, GetToolBarHeight ());
	m_toolBarFloater.DrawDebugInfo (FALSE);
	m_toolBarFloater.InsertWindow (&m_wndToolBar, WINDOW_STRETCHED);
	
	// Create Debug window floater
	m_debugWndFloater.CreateFloater (this, m_pLayout, 300);
	m_debugWndFloater.DrawDebugInfo (FALSE);
	m_debugWndFloater.InsertWindow (&m_debugCtrl, WINDOW_STRETCHED);
	m_debugWndFloater.EnableSplitter ();

	// Add Main menu to layout
	/*
	CPanel* pMainMenu = new CPanel (_T("Main menu"), 30);
	pMainMenu->InsertWindow (&m_mainMenu, WINDOW_STRETCHED);
	pMainMenu->DrawDebugInfo (FALSE);
	m_pLayout->AddToTop(pMainMenu);
	*/
	
	// Create Memory Dump window floater
	m_memoryWndFloater.CreateFloater (this, m_pLayout, 300);
	m_memoryWndFloater.DrawDebugInfo (FALSE);
	m_memoryWndFloater.InsertWindow (&m_memoryCtrl, WINDOW_STRETCHED);
	m_memoryWndFloater.EnableSplitter ();
	
	// Add Statusbar to layout
	CPanel* pStatus = new CPanel (_T("Status bar"), STATUSBAR_HEIGHT);
	pStatus->InsertWindow (&m_wndStatusBar, WINDOW_STRETCHED);
	pStatus->DrawDebugInfo (FALSE);
	m_pLayout->AddToBottom (pStatus);

	// Add Tape control to layout
	CPanel* pTape = new CPanel (_T("Tape control"), 50);
	pTape->InsertWindow (&m_tapeCtrl, WINDOW_STRETCHED);
	pTape->DrawDebugInfo (FALSE);
	pTape->Show (FALSE);
	m_pLayout->AddToBottom (pTape);

	m_pLayout->DrawDebugInfo (FALSE);
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::InitFloaters ()
{
	if (!m_bFloatersInitialized)
	{
		m_debugWndFloater.DockFloater (m_pLayout, CPanel::ALIGN_RIGHT, m_bShowDebugWindow);
		m_memoryWndFloater.DockFloater (m_pLayout, CPanel::ALIGN_LEFT, m_bShowDumpWindow);
		m_toolBarFloater.DockFloater (m_pLayout, CPanel::ALIGN_TOP, TRUE);

		SetFocus (); // SetFocus to Main window

		m_bFloatersInitialized = TRUE;
	}
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::InitMenu ()
{
	m_mainMenu.CustomizeMain (GetMenu ());
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::InitEmulator ()
{
	// Init directories
	m_strBinPath = NormalizePath (GetCurrentPath () + GetIniFileString (IDS_INI_DIRECTORIES, IDS_INI_PROGRAM_DIRECTORY, m_strIniFilePath));
	m_strToolsPath = NormalizePath (GetCurrentPath () + GetIniFileString (IDS_INI_DIRECTORIES, IDS_INI_TOOLS_DIRECTORY, m_strIniFilePath));
	m_strMemPath = NormalizePath (GetCurrentPath () + GetIniFileString (IDS_INI_DIRECTORIES, IDS_INI_MEM_DIRECTORY, m_strIniFilePath));
	m_strSavesPath = NormalizePath (GetCurrentPath () + GetIniFileString (IDS_INI_DIRECTORIES, IDS_INI_SAVES_DIRECTORY, m_strIniFilePath));
	m_strTapePath = NormalizePath (GetCurrentPath () + GetIniFileString (IDS_INI_DIRECTORIES, IDS_INI_TAPES_DIRECTORY, m_strIniFilePath));
	m_strScriptsPath = NormalizePath (GetCurrentPath () + GetIniFileString (IDS_INI_DIRECTORIES, IDS_INI_SCRIPTS_DIRECTORY, m_strIniFilePath));

	m_bSavesDefault = GetIniFileBool (IDS_INI_DIRECTORIES, IDS_INI_SAVES_DEFAULT, m_strIniFilePath);

	// Init options
	m_speaker.EnableSound (GetIniFileBool (IDS_INI_OPTIONS, IDS_INI_SPEAKER, m_strIniFilePath));
	m_covox.EnableSound (GetIniFileBool (IDS_INI_OPTIONS, IDS_INI_COVOX, m_strIniFilePath));
	m_bBKKeyboard = GetIniFileBool (IDS_INI_OPTIONS, IDS_INI_BKKEYBOARD, m_strIniFilePath);
	m_bJoystick = GetIniFileBool (IDS_INI_OPTIONS, IDS_INI_JOYSTICK, m_strIniFilePath);

	m_screen.SetColorMode (GetIniFileBool (IDS_INI_OPTIONS, IDS_INI_COLOR_MODE, m_strIniFilePath));
	m_screen.SetAdaptMode (GetIniFileBool (IDS_INI_OPTIONS, IDS_INI_BLACK_WHITE, m_strIniFilePath));
	m_screen.SetSaveCurrentMode (GetIniFileBool (IDS_INI_OPTIONS, IDS_INI_SAVE_CURMODE, m_strIniFilePath));
	
	if (GetIniFileBool (IDS_INI_OPTIONS, IDS_INI_FULL_SCREEN, m_strIniFilePath))
		m_screen.SetFullScreenMode ();

	m_bEnableSaveCurrentMode = GetIniFileBool (IDS_INI_OPTIONS, IDS_INI_ENABLE_SAVE_CURMODE, m_strIniFilePath);

	m_bEmulateLoadTape = GetIniFileBool (IDS_INI_OPTIONS, IDS_INI_EMUL_LOAD_TAPE, m_strIniFilePath);
	m_bEmulateSaveTape = GetIniFileBool (IDS_INI_OPTIONS, IDS_INI_EMUL_SAVE_TAPE, m_strIniFilePath);
	m_bTapeAutoBeginDetection = GetIniFileBool (IDS_INI_OPTIONS, IDS_INI_AUTO_BEG_TAPE, m_strIniFilePath);
	m_bTapeAutoEnsDetection = GetIniFileBool (IDS_INI_OPTIONS, IDS_INI_AUTO_END_TAPE, m_strIniFilePath);

	m_bPauseCPUAfterStart = GetIniFileBool (IDS_INI_OPTIONS, IDS_INI_PAUSE_CPU, m_strIniFilePath);
	m_CPURunAddr = GetIniFileOct (IDS_INI_OPTIONS, IDS_INI_CPU_RUN_ADDR, m_strIniFilePath);
	m_initialCPUSpeed = GetIniFileFloat (IDS_INI_OPTIONS, IDS_INI_CPU_SPEED, m_strIniFilePath);
	m_cpu_sps = GetIniFileInt (IDS_INI_OPTIONS, IDS_INI_CPU_SPS, m_strIniFilePath);

	m_bShowPerfomance = GetIniFileBool (IDS_INI_OPTIONS, IDS_INI_SHOW_PERFOMANCE_STRING, m_strIniFilePath);
	m_bAskForBreak = GetIniFileBool (IDS_INI_OPTIONS, IDS_INI_ASK_FOR_BREAK, m_strIniFilePath);
	m_bShowDumpWindow = GetIniFileBool (IDS_INI_OPTIONS, IDS_INI_MEM_DUMP, m_strIniFilePath);
	m_bShowDebugWindow = GetIniFileBool (IDS_INI_OPTIONS, IDS_INI_DEBUG_WINDOW, m_strIniFilePath);
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::InitRegistry ()
{
	CString strTitle;
	strTitle.LoadString (IDS_EMUL_TITLE);

	CString strBinExt;
	strBinExt.LoadString (IDS_FILEEXT_BINARY);
	CString strMsfExt;
	strMsfExt.LoadString (IDS_FILEEXT_MEMORYSTATE);
	CString strTapeExt;
	strTapeExt.LoadString (IDS_FILEEXT_TAPE);
	CString strScriptExt;
	strScriptExt.LoadString (IDS_FILEEXT_SCRIPT);
	CString strROMExt;
	strROMExt.LoadString (IDS_FILEEXT_ROM);

	// Register BinFile
	RegisterDefaultIcon (strBinExt, _T("BKEmulator.BinFile"), _T("binary file"), IDR_MAINFRAME);
	RegisterShellCommand (_T("BKEmulator.BinFile"), _T("/b %1"));

	// Register MemFile
	RegisterDefaultIcon (strMsfExt, _T("BKEmulator.MemFile"), _T("memory state"), IDI_MEMORY);
	RegisterShellCommand (_T("BKEmulator.MemFile"), _T("/m %1"));

	// Register TapeFle
	RegisterDefaultIcon (strTapeExt, _T("BKEmulator.TapeFile"), _T("tape file"), IDI_TAPE);
	RegisterShellCommand (_T("BKEmulator.TapeFile"), _T("/t %1"));

	// Register ScriptFile
	RegisterDefaultIcon (strScriptExt, _T("BKEmulator.ScriptFile"), _T("script file"), IDI_SCRIPT);
	RegisterShellCommand (_T("BKEmulator.ScriptFile"), _T("/s %1"));

	// Register ROMFile
	RegisterDefaultIcon (strROMExt, _T("BKEmulator.RomFile"), _T("ROM file"), IDI_ROM);
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::RegisterDefaultIcon (CString strExt, CString strSection, CString strDescription, DWORD iconID)
{
	CString strTitle;
	strTitle.LoadString (IDS_EMUL_TITLE);

	CString strIcon;
	strIcon = IntToString (iconID * -1, 10);

	HKEY hKey;
	::RegCreateKey (HKEY_CLASSES_ROOT, strExt, &hKey);
	::RegSetValue (hKey, NULL, REG_SZ, strSection, 0);
	::RegCloseKey (hKey);

	::RegCreateKey (HKEY_CLASSES_ROOT, strSection, &hKey);
	::RegSetValue (hKey, NULL, REG_SZ, strTitle + _T(" ") + strDescription, 0);
	::RegCloseKey (hKey);

	::RegCreateKey (HKEY_CLASSES_ROOT, strSection + _T("\\DefaultIcon"), &hKey);
	::RegSetValue (hKey, NULL, REG_SZ, GetCurrentPath () + _T("bk.exe,") + strIcon, 0);
	::RegCloseKey (hKey);
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::RegisterShellCommand (CString strSection, CString strArguments)
{
	HKEY hKey;
	::RegCreateKey (HKEY_CLASSES_ROOT, strSection + _T("\\Shell"), &hKey);
	::RegCloseKey (hKey);

	::RegCreateKey (HKEY_CLASSES_ROOT, strSection + _T("\\Shell\\Open"), &hKey);
	::RegCloseKey (hKey);

	::RegCreateKey (HKEY_CLASSES_ROOT, strSection + _T("\\Shell\\Open\\Command"), &hKey);
	::RegSetValue (hKey, NULL, REG_SZ, GetCurrentPath () + _T("bk.exe ") +  strArguments, 0);
	::RegCloseKey (hKey);
}



/////////////////////////////////////////////////////////////////////////////
BOOL CMainFrame::ParseCommandLineParameters (CString strCommands)
{
	ClearProcessingFiles ();

	strCommands.MakeUpper ();

	if (!m_mtInstance.Lock (0))
	{
		// If other instance of Emulator exists
		// Pass command line to it
		TCHAR pInstanceCommandPrompt[1024];
		memcpy (pInstanceCommandPrompt, strCommands.GetBuffer (sizeof (pInstanceCommandPrompt)), sizeof (pInstanceCommandPrompt));

		COPYDATASTRUCT cd;
		cd.dwData = (UINT)2;
		cd.cbData = sizeof (pInstanceCommandPrompt);
		cd.lpData = (void*)pInstanceCommandPrompt;

		// Find other instance window
		CWnd* pOtherInstanceWnd = FindWindow (_T("BK Emulator Main Window"), NULL);

		if (!pOtherInstanceWnd)
			return FALSE; // If not found - mean error occure

		pOtherInstanceWnd->SendMessage (WM_COPYDATA, (WPARAM)this->m_hWnd, (LPARAM)&cd);
		strCommands.ReleaseBuffer ();

		return FALSE; // Close aplication
	}

	if (strCommands.Find ('/') == -1)
		return TRUE;

	strCommands = strCommands.Right (strCommands.GetLength () - max (0, strCommands.Find ('/')));

	while (strCommands.GetLength ())
	{
		int findPos = strCommands.Find ('/', 1);
		if (findPos == -1)
			findPos = strCommands.GetLength ();

		CString strParam = strCommands.Left (findPos);
		int len = strCommands.Find ('/', 1);
		len = len != -1 ? len : strCommands.GetLength ();
		strCommands = strCommands.Right (strCommands.GetLength () - len);

		strParam.TrimLeft (" ");
		strParam.TrimRight (" ");

		strParam.TrimLeft ("/");
		
		TCHAR command = strParam[0];
		strParam = strParam.Right (strParam.GetLength () - 1);
		strParam.TrimLeft (" ");

		switch (command)
		{
		case _T('B'): // Binary file
			{
				m_strBinFileName = GetFileName (strParam);

				CString strFilePath = GetFilePath (strParam);
				if (!strFilePath.IsEmpty ())
					m_strBinPath = strFilePath;

				m_strScriptFileName = _T("autorun\\monitor_load.bkscript");
				m_bEmulateLoadTape = TRUE;
			}
			break;
		case _T('M'): // Memory state file
			{
				m_strMemFileName = GetFileName (strParam);

				CString strFilePath = GetFilePath (strParam);
				if (!strFilePath.IsEmpty ())
					m_strMemPath = strFilePath;
			}
			break;
		case _T('T'): // BK tape file
			{
				m_strTapeFileName = GetFileName (strParam);

				CString strFilePath = GetFilePath (strParam);
				if (!strFilePath.IsEmpty ())
					m_strTapePath = strFilePath;
			}
			break;
		case _T('S'): // Initial Script file
			{
				m_strScriptFileName = GetFileName (strParam);
				
				CString strFilePath = GetFilePath (strParam);
				if (!strFilePath.IsEmpty ())
					m_strScriptsPath = strFilePath;
			}
			break;
		case _T('?'):
		case _T('H'):
			{
				CString strTitle;
				strTitle.LoadString (IDS_EMUL_TITLE);

				CString strPrompt;
				strPrompt.LoadString (IDS_COMMAND_PROMPT_INFO);

				CBKMessageBox (strTitle + strPrompt);
			return FALSE;
			}
		case _T('W'):
			break;
		}
	}

	return TRUE;
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::StartPlayTape (CString strPath)
{
	CString strWaveExt;
	strWaveExt.LoadString (IDS_FILEEXT_WAVE);
	CString strTapeExt;
	strTapeExt.LoadString (IDS_FILEEXT_TAPE);

	CString strTapePath = strPath;
	CString strExt = GetFileExt (strTapePath);

	if (!strExt.CompareNoCase (strWaveExt))
		m_tape.LoadWaveFile (strTapePath);
	else
		m_tape.LoadMSFFile (strTapePath);

	TAPE_FILE_INFO tfi;
	m_tape.GetWaveFile (&tfi);
	m_tape.SetPlayWavePos (0);

	m_tape.StartPlay ();
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnPaint() 
{
	CPaintDC dc(this); // device context for painting
	
	// Draw layout with child windows
	if (m_pLayout)
		m_pLayout->DrawToWindow ();
	
	// Do not call CFrameWnd::OnPaint() for painting messages
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnSize(UINT nType, int cx, int cy) 
{
	//CFrameWnd::OnSize(nType, cx, cy);
	
	// Resize layout to current window size
	if (m_pLayout)
	{
		m_pLayout->ResizeToWindow ();
		m_pLayout->RecalcWindowsZorder ();
	}

	InitFloaters (); // Init floaters one time at start
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnActivate(UINT nState, CWnd* pWndOther, BOOL bMinimized) 
{
	CFrameWnd::OnActivate(nState, pWndOther, bMinimized);
	
	if (m_pLayout)
		m_pLayout->Redraw (TRUE);
}



/////////////////////////////////////////////////////////////////////////////
int	CMainFrame::GetToolBarHeight ()
{
	// Retrieve toolbar height
	return TOOLBAR_HEIGHT;
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::UpdateControls ()
{
	m_wndToolBar.GetToolBarCtrl ().EnableButton (ID_FILE_LOADTAPE, !m_bEmulateLoadTape);
	m_tapeCtrl.GetDlgItem (IDC_TC_RECORD)->EnableWindow (!m_bEmulateSaveTape);
}



/////////////////////////////////////////////////////////////////////////////
HRESULT CMainFrame::OnMainLoop (WPARAM wParam, LPARAM lParam)
{
	static int second = 0;
	static int tick = 0;

	// Reset sound buffers
	m_speaker.Reset ();
	m_covox.Reset ();

	// Draw next frame
	m_screen.DrawScreen ();

	// Execute instruction set
	if (m_pChip->IsCPURun ())
		m_pChip->ExecuteInstructionSet (m_cpu_ips * 50 / m_cpu_sps);

	// Parse script if present
	m_script.RunScript ();

	// Get stored of 1/50 second sound data 
	int buffer_len = m_pSound->GetSampleCount ();

	// Get current tape wave data
	m_tape.PlayWaveGetBuffer (m_pSampleBuffer, buffer_len);

	// Send current tape buffer to 177716
	m_speaker.ReceiveSamples (m_pSampleBuffer, buffer_len);

	// Get sound data from BK speaker
	m_speaker.MixAndRemoveSamples (m_pSampleBuffer, buffer_len);

	m_tape.RecordWaveGetBuffer (m_pSampleBuffer, buffer_len);

	// Get sound data from covox
	m_covox.MixAndRemoveSamples (m_pSampleBuffer, buffer_len);
	
	// Wait for Sound Card syncronization and send buffer
	m_pSound->WaitAndSendBuffer (m_pSampleBuffer);
	
	if (tick >= 50)
	{
		second++;

		/*
		CString str;
		str.Format ("second = %i", second);
		SetWindowText (str);
		*/

		tick = 0;
	}

	if (m_bShowPerfomance)
	{
		// Show perfomance on status bar
		CString strPerfomance;
		strPerfomance.Format ("MClock %i sec., CPU speed %9.6f, FPS %i", second, m_pChip->GetCPUSpeed (), m_screen.GetFPS ());

		m_wndStatusBar.SetWindowText (strPerfomance);
	}

	tick++;

	UpdateControls ();

	return TRUE;
}



////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnMeasureItem(int nIDCtl, LPMEASUREITEMSTRUCT lpMeasureItemStruct) 
{
	lpMeasureItemStruct->itemWidth = m_mainMenu.GetItemWidth (lpMeasureItemStruct->itemID, MF_BYCOMMAND);
	
	CFrameWnd::OnMeasureItem(nIDCtl, lpMeasureItemStruct);
}



////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnTimer(UINT nIDEvent) 
{
	if (nIDEvent == TIMER_CPU)
		OnMainLoop (0, 0);
	
	CFrameWnd::OnTimer(nIDEvent);
}



////////////////////////////////////////////////////////////////////////////
BOOL CMainFrame::OnCopyData(CWnd* pWnd, COPYDATASTRUCT* pCopyDataStruct) 
{
	if (pCopyDataStruct->dwData == 2)
	{
		CString strLine ((LPCSTR)pCopyDataStruct->lpData, pCopyDataStruct->cbData);
		
		if (ParseCommandLineParameters (strLine))
			ProcessFile ();

		SetForegroundWindow ();
	}

	return CFrameWnd::OnCopyData(pWnd, pCopyDataStruct);
}



////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnDropFile (CString strPath)
{
	ClearProcessingFiles ();

	CString strFilePath = GetFilePath (strPath);
	CString strFileExt = GetFileExt (strPath);

	CString strBinExt (MAKEINTRESOURCE (IDS_FILEEXT_BINARY));
	CString strMemExt (MAKEINTRESOURCE (IDS_FILEEXT_MEMORYSTATE));
	CString strTapExt (MAKEINTRESOURCE (IDS_FILEEXT_TAPE));
	CString strWavExt (MAKEINTRESOURCE (IDS_FILEEXT_WAVE));
	CString strScptExt (MAKEINTRESOURCE (IDS_FILEEXT_SCRIPT));

	if (!strFileExt.CompareNoCase (strBinExt)) // Binary file
	{
		m_strBinFileName = GetFileName (strPath);

		if (!strFilePath.IsEmpty ())
			m_strBinPath = strFilePath;

		m_strScriptFileName = _T("autorun\\monitor_load.bkscript");
		m_bEmulateLoadTape = TRUE;
	}
	else if (!strFileExt.CompareNoCase (strMemExt)) // Memory state file
	{
		m_strMemFileName = GetFileName (strPath);

		CString strFilePath = GetFilePath (strPath);
		if (!strFilePath.IsEmpty ())
			m_strMemPath = strFilePath;	
	}
	else if (strFileExt.CompareNoCase (strTapExt) == 0
		|| strFileExt.CompareNoCase (strWavExt) == 0) // BK tape file
	{
		m_strTapeFileName = GetFileName (strPath);

		CString strFilePath = GetFilePath (strPath);
		if (!strFilePath.IsEmpty ())
			m_strTapePath = strFilePath;
	}
	else if (!strFileExt.CompareNoCase (strScptExt)) // Initial Script file
	{
		m_strScriptFileName = GetFileName (strPath);
		
		CString strFilePath = GetFilePath (strPath);
		if (!strFilePath.IsEmpty ())
			m_strScriptsPath = strFilePath;
	}
	else
	{
		ASSERT (FALSE);
		return;
	}

	ProcessFile ();
}



////////////////////////////////////////////////////////////////////////////
void CMainFrame::ClearProcessingFiles ()
{
	m_strBinFileName.Empty ();
	m_strMemFileName.Empty ();
	m_strTapeFileName.Empty ();
	m_strScriptFileName.Empty ();
}



////////////////////////////////////////////////////////////////////////////
void CMainFrame::ProcessFile ()
{
	m_script.SetScriptPath (m_strScriptsPath + m_strScriptFileName);

	// Init BK Chip
	if (!m_strMemFileName.IsEmpty ())
	{
		if (!LoadMemoryState (m_strMemPath + m_strMemFileName))
			// If cant load msf file
			ConfigurationConstructor (MSF_CONF_BK1001); // Create default configuration
	}
	else if (!m_strBinFileName.IsEmpty ())
		// If bin file specified
		ConfigurationConstructor (MSF_CONF_BK1001); // Create default configuration
	
	if (!m_strTapeFileName.IsEmpty ())
		// If tape file specified
		StartPlayTape (m_strTapePath + m_strTapeFileName);
}



////////////////////////////////////////////////////////////////////////////
HRESULT CMainFrame::OnEnterMenuLoop (WPARAM wParam, LPARAM lParam)
{
	SetTimer (TIMER_CPU, 1, NULL);
	m_bMenuLoop = TRUE;

	return TRUE;
}




////////////////////////////////////////////////////////////////////////////
HRESULT CMainFrame::OnExitMenuLoop (WPARAM wParam, LPARAM lParam)
{
	KillTimer (TIMER_CPU);
	m_bMenuLoop = FALSE;

	return TRUE;
}



////////////////////////////////////////////////////////////////////////////
HRESULT CMainFrame::OnEnterSizeMove (WPARAM wParam, LPARAM lParam)
{
	SetTimer (TIMER_CPU, 1, NULL);

	return TRUE;
}



////////////////////////////////////////////////////////////////////////////
HRESULT CMainFrame::OnExitSizeMove (WPARAM wParam, LPARAM lParam)
{
	KillTimer (TIMER_CPU);

	return TRUE;
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
	// Is Stop pressed
	if (nChar == VK_PAUSE)
	{
		m_pChip->StopInterrupt ();
		return;
	}

	if (EmulateJoystickDown (nChar, nRepCnt, nFlags))
		return;

	// Is Ar2 pressed
	if (nChar == VK_CONTROL)
	{
		m_bIsAr2Press = TRUE;
		return;
	}

	// Is Shift pressed
	if (nChar == VK_SHIFT)
	{
		m_bIsShiftPress = TRUE;
		return;
	}

	if (m_bIsKeyPress)
		return;

	if (m_bBKKeyboard)
		m_bIsKeyPress = TRUE;

	// Set keyboard state bit in 177660
	WORD reg177660 = m_pChip->GetWordIndirect (0177660);

	//if (!(reg177660 & 0200))
	{
		m_pChip->SetWordIndirect (0177660, reg177660 | 0200);

		// Set key code in 177662
		WORD reg177662 = TranslateKey (nChar, m_bIsShiftPress, nFlags & KF_EXTENDED);
		
		m_pChip->SetWordIndirect (0177662, reg177662 & 0177);
	}
		
	// Set keypressed flag in 177716
	WORD reg177716 = m_pChip->GetWordIndirect (0177716);
	m_pChip->SetWordIndirect (0177716, reg177716 & ~0100);
	
	if (!m_bIsAr2Press)
		m_pChip->KeyboardInterrupt (INTERRUPT_60);
	else
		m_pChip->KeyboardInterrupt (INTERRUPT_274);
	
	CFrameWnd::OnKeyDown(nChar, nRepCnt, nFlags);
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnKeyUp(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
	if (nChar == VK_PAUSE)
		return;
	
	if (EmulateJoystickUp (nChar, nRepCnt, nFlags))
		return;

	if (nChar == VK_SHIFT)
		m_bIsShiftPress = FALSE;	
	else if (nChar == VK_CONTROL)
		m_bIsAr2Press = FALSE;
	else
	{
		m_bIsKeyPress = FALSE;

		// Set keypressed flag in 177716
		WORD reg177716 = m_pChip->GetWordIndirect (0177716);
		m_pChip->SetWordIndirect (0177716, reg177716 | 0100);
	}
	
	CFrameWnd::OnKeyUp(nChar, nRepCnt, nFlags);
}



////////////////////////////////////////////////////////////////////////////
WORD CMainFrame::TranslateKey (int key, BOOL bShift, BOOL bExtended)
{
	if (bExtended)
	{
		switch (key)
		{
		case VK_LEFT:
			return BKKEY_L_ARROW;
		case VK_DELETE: // Clear
			return BKKEY_SBR;
		case VK_HOME:	// Lat
			return BKKEY_LAT;
		case VK_END:	// Rus
			return BKKEY_RUS;
		case VK_INSERT: // BC
			return BKKEY_VS;
		case VK_RIGHT:
			return BKKEY_R_ARROW;
		case VK_UP:
			return BKKEY_U_ARROW;
		case VK_DOWN:
			return BKKEY_D_ARROW;
		}
	}
	
	switch (key)
	{
	case VK_ESCAPE:	//KT
		return BKKEY_KT;
	case VK_ENTER:
		return BKKEY_ENTER;
	case VK_BACK_SPACE:
		return BKKEY_ZAB;
	case VK_INSERT: // |-->
		return BKKEY_RAZDVIG;
	case VK_DELETE: // |<--
		return BKKEY_SDVIG;
	case VK_SPACE:
		key = BKKEY_PROBEL;
		break;

	// [+,/]
	case VK_OEM_PLUS:
		key = 053;
		break;
	case VK_OEM_COMMA:
		key = 054;
		break;
	case VK_OEM_MINUS:
		key = 055;
		break;
	case VK_OEM_PERIOD:
		key = 056;
		break;
	case VK_OEM_2:
		key = 057;
		break;
	case VK_OEM_1:
		key = 073;
		break;

	// [[,^]
	case VK_OEM_4:
		key = 0133;
		break;
	case VK_OEM_5:
		key = 0134;
		break;
	case VK_OEM_6:
		key = 0135;
		break;
	case VK_OEM_7:
		key = 0136;
		break;
	}

	if (key == 053 && bShift)
		return 052;

	if (key == 073 && bShift)
		return 072;

	if (key == 0273 && bShift)
		return 056;

	if (key == 0275 && bShift)
		return 054;

	if (key >= 0200)
	{
		if (bShift)
			key -= 0140;
		else
			key -= 0200;

		if (key >= 0300)
		{
			if (key <= 337)
				key += 040;
			else
				key -= 040;
		}

		return key;
	}

	if (bShift)
	{
		if (key >= 054 && key <= 057)
			return key += 020;
		else
		if (key >= 060 && key <= 071)
			return key -= 020;
		else
			return key += 040;
	}		

	return key;
}



/////////////////////////////////////////////////////////////////////////////
BOOL CMainFrame::ConfigurationConstructor (int configuration)
{
	if (m_pChip)
	{
		// If already exist destroy old configuration
		m_pChip->StopCPU ();
		delete m_pChip;
		m_pChip= NULL;
	}

	// Make new configuration
	switch (configuration)
	{
	case MSF_CONF_BK1001:
		m_pChip = new CMotherBoard;
		break;

	case MSF_CONF_BK1001_MSTD:
		m_pChip = new CMotherBoard_MSTD;
		break;

	case MSF_CONF_BK1001_EXT32:
		m_pChip = new CMotherBoard_EXT32;
		break;

	case MSF_CONF_BK1001_FDD:
		m_pChip = new CMotherBoard_FDD;
		break;

	default:
		ASSERT (FALSE);
		return FALSE;
	}

	//Attach and init new instance
	m_pChip->AttachWindow (this);
	m_pChip->AttachScreen (&m_screen);
	m_pChip->AttachSpeaker (&m_speaker);
	m_pChip->AttachCovox (&m_covox);
	m_pChip->ResetHot (0100000);
	m_pChip->SetCPUSpeed (m_initialCPUSpeed);
	m_speaker.SetIPS (m_cpu_ips * 50 / m_cpu_sps);
	m_covox.SetIPS (m_cpu_ips * 50 / m_cpu_sps);

	// Run CPU
	if (m_bPauseCPUAfterStart)
		m_pChip->BreakCPU ();
	else
		m_pChip->RunCPU ();

	m_pChip->EnableAskForBreak (m_bAskForBreak);

	// Attach new chip to debugger
	m_debugger.AttachChip (m_pChip);

	// Attach script engine
	m_script.AttachChip (m_pChip);

	// Attach new chip to tape control
	m_tapeCtrl.SetMotheboard (m_pChip);

	return TRUE;
}



/////////////////////////////////////////////////////////////////////////////
HRESULT CMainFrame::OnDisplayChange (WPARAM wParam, LPARAM lParam)
{
	// Catch this message for restoring window mode
	if (m_screen.IsFullScreenMode ())
	{
		// Not fullscreen mode means that user return to window mode
		m_screen.SetWindowMode ();

		// Refreash screen
		m_pChip->RefreshScreen ();
	}

	return TRUE;
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnPaletteChanged(CWnd* pFocusWnd) 
{
	CFrameWnd::OnPaletteChanged(pFocusWnd);
	
	// Catch this message for restoring window mode
	if (m_screen.IsFullScreenMode ())
	{
		// Not fullscreen mode means that user return to window mode
		m_screen.SetWindowMode ();

		// Refreash screen
		m_pChip->RefreshScreen ();
	}
}



/////////////////////////////////////////////////////////////////////////////
HRESULT CMainFrame::OnEmulateTape (WPARAM wParam, LPARAM lParam)
{
	EmulateTapeOperation ();

	// Remove key press bit 100
	m_pChip->SetWordIndirect (0177716, m_pChip->GetWordIndirect (0177716) | 0100);

	return TRUE;
}



////////////////////////////////////////////////////////////////////////////
void CMainFrame::EmulateTapeOperation ()
{	
	// No error yet
	BOOL bError = FALSE;
	
	// Get Params block address from R1 (BK emt 36) 
	WORD pParam = m_pChip->GetWordIndirect (R1);

	// First byte it param is tape command Load, Save, etc.
	WORD command = m_pChip->GetByteIndirect (pParam);
	
	// Set error in second byte (will changed later)
	m_pChip->SetByteIndirect (pParam + 1, 2);
	
	WORD fileAddr = m_pChip->GetWordIndirect (pParam + 2); // Second word is file address
	WORD fileSize = m_pChip->GetWordIndirect (pParam + 4); // Fird word is file length

	CString strFileName;
	// Internal BK loading
		
	strFileName.GetBufferSetLength (16); // BK can't load file with name length more than 16 bytes

	// Takes next 16 bytes in params block as filename
	for (int c = 0; c < strFileName.GetLength (); c++)
	{
		BYTE byte = m_pChip->GetByteIndirect (pParam + 6 + c);
		strFileName.SetAt (c, byte);
	}

	strFileName.ReleaseBuffer ();

	// Remove all "return" and "space" sybbols from name string
	strFileName.Remove ('\012');
	strFileName.Remove (' ');
	
	////////////////////////////////////////////////////////////////////////////
	// File operation
	////////////////////////////////////////////////////////////////////////////

	// If current command is "Load"
	if (command == 3)
	{
		////////////////////////////////////////////////////////////////////////////
		// Loading file
		////////////////////////////////////////////////////////////////////////////

		//m_pChip->SetWordIndirect (0322, 040000);

		if (!m_bEmulateLoadTape)
			return;

		if (!m_strBinFileName.IsEmpty ())
		{
			strFileName = m_strBinFileName;
			m_strBinFileName = _T("");
		}

		if (strFileName.SpanExcluding (" ") == "")
		{
			CWindowMode mode (&m_screen, m_pChip);

			// Is name is empty Open File dialog
		
			// Save current dirrectory
			CString strCurDir;
			::GetCurrentDirectory (1024, strCurDir.GetBufferSetLength (1024));
			strCurDir.ReleaseBuffer ();

			::SetCurrentDirectory (m_strBinPath);

			CString strFilterBin;
			strFilterBin.LoadString (IDS_FILEFILTER_BIN);

			CFileDialog dlg (TRUE, NULL, NULL, 
				OFN_HIDEREADONLY|OFN_OVERWRITEPROMPT,
				strFilterBin, m_screen.GetBackgroundWindow ());

			// Set initial directory is Binary file directory
			dlg.m_ofn.lpstrInitialDir = m_strBinPath;

			if (dlg.DoModal () == IDCANCEL)
			{
				// If user press Cancel set error in second byte of params block
				m_pChip->SetByteIndirect (pParam + 1, 4);
				bError = TRUE; // error ocure
			}

			m_strBinPath = GetFilePath (dlg.GetPathName ());
			strFileName = dlg.GetPathName ();

			// Restore current directory
			::SetCurrentDirectory (strCurDir);
		}
		else
		{
			// If name is not empty

			// If internal file loading
			if (m_bSavesDefault)
				// If Saves Default flag is set loading from User directory
				strFileName = m_strSavesPath + strFileName;
			else
				// If not Load from Binary files directory 
				strFileName = m_strBinPath + strFileName;
		}
		
		// Loading file if no error
		CFile file;
		if ((file.Open (strFileName, CFile::modeRead)) && (!bError))
		{
			WORD readAddr; 
			file.Read (&readAddr, sizeof (readAddr)); // First word in file is address

			if (fileAddr) // If not zero read address = loading adddress
				readAddr = fileAddr; // else read address the same as in address block

			fileAddr = readAddr;

			WORD readSize;
			file.Read (&readSize, sizeof (readSize)); // Second word in file is it length

			// Load array to readAddr
			for (int i = 0; i < readSize; i++)
			{
				BYTE val;
				file.Read (&val, sizeof (val));

				m_pChip->SetByteIndirect (readAddr++, val);
			}

			file.Close ();

			// Set no error in params block
			m_pChip->SetByteIndirect (pParam + 1, 0);
			
			// Set system cells like after normal emt 36 loading
			m_pChip->SetWordIndirect (0264, fileAddr);
			m_pChip->SetWordIndirect (0266, readSize);
			m_pChip->SetWordIndirect (0346, fileAddr);
			m_pChip->SetWordIndirect (0350, readSize);
		}
		else
		{
			CWindowMode mode (&m_screen, m_pChip);

			// If error display message
			CString strError;
			strError.Format (IDS_CANT_OPEN_FILE_S, strFileName);

			CBKMessageBox message;
			if (message.Show (strError, MB_ICONWARNING|MB_YESNO|MB_DEFBUTTON2) == IDYES)
				m_pChip->StopCPU ();
		}

		// Move to R5 last reading address like in normal emt 36
		m_pChip->SetWordIndirect (R5, m_pChip->GetWordIndirect (0264) + m_pChip->GetWordIndirect (0266));
		
		// Make RTS PC
		m_pChip->SetWordIndirect (PC, m_pChip->GetWordIndirect (m_pChip->GetWordIndirect (SP)));
		m_pChip->SetWordIndirect (SP, m_pChip->GetWordIndirect (SP) + 2);
	}
	else
	if (command == 2)
	{
		////////////////////////////////////////////////////////////////////////////
		// Saving file
		////////////////////////////////////////////////////////////////////////////

		if (!m_bEmulateSaveTape)
			return;

		// Check file name for empty
		if (strFileName.SpanExcluding (" ") == "")
		{
			// If empty display save dialog
			CFileDialog dlg (FALSE, NULL, NULL, 
				OFN_HIDEREADONLY|OFN_OVERWRITEPROMPT,
				NULL, this);

			dlg.m_ofn.lpstrInitialDir = m_strBinPath;

			if (dlg.DoModal () != IDOK)
			{
				// Set error byte if CANCEL pressed
				m_pChip->SetByteIndirect (pParam + 1, 4);
				bError = TRUE;
			}

			m_strBinPath = GetFilePath (dlg.GetPathName ());

			// Get File name
			strFileName = dlg.GetPathName ();
		}

		CFile file;
		
		// Save file array if no errors
		if ((file.Open (m_strSavesPath + strFileName, CFile::modeCreate|CFile::modeWrite)) && (!bError))
		{
			// Write file address and length
			file.Write (&fileAddr, sizeof (fileAddr));
			file.Write (&fileSize, sizeof (fileSize));

			for (int i = 0; i < fileSize; i++)
			{
				BYTE val = m_pChip->GetByteIndirect (fileAddr++);
				file.Write (&val, sizeof (val));
			}

			file.Close ();

			// Set no error in parameter block byte
			m_pChip->SetByteIndirect (pParam + 1, 0);
			
			// Set last reading address like in normal emt 36
			m_pChip->SetWordIndirect (0264, fileAddr);
			m_pChip->SetWordIndirect (0266, fileSize);
			m_pChip->SetWordIndirect (0346, fileAddr); 
			m_pChip->SetWordIndirect (0350, fileSize);
		}
		
		// Make RTS PC
		m_pChip->SetWordIndirect (PC, m_pChip->GetWordIndirect (m_pChip->GetWordIndirect (SP)));
		m_pChip->SetWordIndirect (SP, m_pChip->GetWordIndirect (SP) + 2);
	}

	// Refresh screen and run CPU
	m_pChip->RefreshScreen ();

	// Refresh keyboard
	m_bIsKeyPress = FALSE;
	m_bIsAr2Press = FALSE;
	m_bIsShiftPress = FALSE;
}



////////////////////////////////////////////////////////////////////////////
BOOL CMainFrame::EmulateJoystickDown (UINT nChar, UINT nRepCnt, UINT nFlags)
{
	if (!m_bJoystick)
		return FALSE;

	if (!(nFlags & KF_EXTENDED))
	{
		WORD joystick = m_pChip->GetWordIndirect (0177714);

		switch (nChar)
		{
			case VK_HOME:
				joystick |= 1; // Button 1
				break;
			case VK_END:
				joystick |= 2; // Button 2
				break;
			case VK_PRIOR:
				joystick |= 4; // Button 3
				break;
			case VK_NEXT:
				joystick |= 8; // Button 4
				break;
			case VK_RIGHT:
				joystick |= 16; // Joystick right
				break;
			case VK_LEFT:
				joystick |= 512; // Joystick left
				break;
			case VK_CLEAR:
				joystick |= 32; // Joystick down
				break;
			case VK_UP:
				joystick |= 1024; // Joystick up
				break;
			default:
				return FALSE;
		}

		m_pChip->SetWordIndirect (0177714, joystick);	
		return TRUE;
	}

	return FALSE;
}



////////////////////////////////////////////////////////////////////////////
BOOL CMainFrame::EmulateJoystickUp (UINT nChar, UINT nRepCnt, UINT nFlags)
{
	if (!m_bJoystick)
		return FALSE;

	if (!(nFlags & KF_EXTENDED))
	{
		WORD joystick = m_pChip->GetWordIndirect (0177714);

		switch (nChar)
		{
			case VK_HOME:
				joystick &= ~1; // Button 1
				break;
			case VK_END:
				joystick &= ~2; // Button 2
				break;
			case VK_PRIOR:
				joystick &= ~4; // Button 3
				break;
			case VK_NEXT:
				joystick &= ~8; // Button 4
				break;
			case VK_RIGHT:
				joystick &= ~16; // Joystick right
				break;
			case VK_LEFT:
				joystick &= ~512; // Joystick left
				break;
			case VK_CLEAR:
				joystick &= ~32; // Joystick down
				break;
			case VK_UP:
				joystick &= ~1024; // Joystick up
				break;
			default:
				return FALSE;
		}

		m_pChip->SetWordIndirect (0177714, joystick);
		return TRUE;
	}

	return FALSE;
}



////////////////////////////////////////////////////////////////////////////
BOOL CMainFrame::LoadMemoryState (CString strPath)
{
	CMSFManager msf;
	if (!msf.OpenFile (strPath, TRUE))
		return FALSE;

	if (msf.GetType () == MSF_STATE_ID && msf.GetVersion () >= 10)
	{
		// Save old configuration
		CMotherBoard* pOldChip = m_pChip;
		m_pChip = NULL;

		if (!ConfigurationConstructor (msf.GetConfiguration ()))
		{
			// Unsuported configuration
			m_pChip = pOldChip;
			return FALSE;
		}

		delete pOldChip;
		
		m_pChip->RestoreState (msf, NULL);
		m_pChip->RefreshScreen ();
		return TRUE;
	}

	return FALSE;
}



////////////////////////////////////////////////////////////////////////////
BOOL CMainFrame::SaveMemoryState (CString strPath)
{
	if (m_pChip)
	{
		CMSFManager msf;
		if (!msf.OpenFile (strPath, FALSE))
			return FALSE;

		msf.SetConfiguration (m_pChip->GetConfiguration ());
		m_pChip->RestoreState (msf, m_screen.GetScreenshot ());
	}

	return TRUE;
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnFileLoadstate() 
{
	CWindowMode mode (&m_screen, m_pChip);

	CString strFilterMSF;
	strFilterMSF.LoadString (IDS_FILEFILTER_MSF);

	CLoadMemoryDlg dlg (TRUE, NULL, NULL, 
		OFN_HIDEREADONLY|OFN_OVERWRITEPROMPT|OFN_NOCHANGEDIR,
		strFilterMSF, this);

	dlg.m_ofn.lpstrInitialDir = m_strMemPath;

	if (dlg.DoModal () == IDOK)
	{
		if (LoadMemoryState (dlg.GetPathName ()))
			mode.SetMotheboard (m_pChip);
	}
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnFileSavestate() 
{
	CWindowMode mode (&m_screen, m_pChip);

	CString strFilterMSF;
	strFilterMSF.LoadString (IDS_FILEFILTER_MSF);

	CBKFileDialog dlg (FALSE, _T("msf"), NULL, 
		OFN_HIDEREADONLY|OFN_OVERWRITEPROMPT,
		strFilterMSF, this);

	dlg.m_ofn.lpstrInitialDir = m_strMemPath;

	if (dlg.DoModal () == IDOK)
	{
		if (SaveMemoryState (dlg.GetPathName ()))
		{
		}
	}	

	m_strMemPath = GetFilePath (dlg.GetPathName ());
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateFileLoadtape(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable (!m_bEmulateLoadTape);
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnFileLoadtape() 
{
	CWindowMode mode (&m_screen, m_pChip);

	CString strFilterTape;
	strFilterTape.LoadString (IDS_FILEFILTER_TAPE);
	CString strTapeExt;
	strTapeExt.LoadString (IDS_FILEEXT_TAPE);

	CLoadTapeDlg dlg (TRUE, strTapeExt, NULL, 
		OFN_HIDEREADONLY|OFN_OVERWRITEPROMPT|OFN_NOCHANGEDIR,
		strFilterTape, this);

	dlg.m_ofn.lpstrInitialDir = m_strTapePath;

	if (dlg.DoModal () == IDOK)
		StartPlayTape (dlg.GetPathName ());
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnFilePrint() 
{
	CPrintDlg dlg (FALSE);
	
	dlg.m_pDebugger = &m_debugger;
	dlg.m_strTitle = m_strPrintTitle;
	dlg.m_bPrintScreen = m_bPrintScreen;
	dlg.m_bInverse = m_bPrintInverse;
	dlg.m_startAddr = m_debugger.GetCurrentAddress ();
	dlg.m_endAddr = m_debugger.GetBottomAddress ();

	if (dlg.DoModal () != IDOK)
		return;

	m_strPrintTitle = dlg.m_strTitle;
	m_bPrintScreen = dlg.m_bPrintScreen;
	m_bPrintInverse = dlg.m_bInverse;
	
	Print (&dlg, dlg.m_startAddr, dlg.m_endAddr);
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnCpuRuncpu() 
{
	m_pChip->StopCPU ();
	m_pChip->ResetCold (m_CPURunAddr);
	m_pChip->RunCPU ();
	
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateCpuRunbk001001(CCmdUI* pCmdUI) 
{
	pCmdUI->SetRadio (m_pChip->GetConfiguration () == MSF_CONF_BK1001);	
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnCpuRunbk001001() 
{
	ConfigurationConstructor (MSF_CONF_BK1001);
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateCpuRunbk001001Focal(CCmdUI* pCmdUI) 
{
	pCmdUI->SetRadio (m_pChip->GetConfiguration () == MSF_CONF_BK1001_MSTD);
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnCpuRunbk001001Focal() 
{
	ConfigurationConstructor (MSF_CONF_BK1001_MSTD);
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateCpuRunbk00100132k(CCmdUI* pCmdUI) 
{
	pCmdUI->SetRadio (m_pChip->GetConfiguration () == MSF_CONF_BK1001_EXT32);
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnCpuRunbk00100132k() 
{
	ConfigurationConstructor (MSF_CONF_BK1001_EXT32);
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateCpuRunbk001001Fdd16k(CCmdUI* pCmdUI) 
{
	pCmdUI->SetRadio (m_pChip->GetConfiguration () == MSF_CONF_BK1001_FDD);
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnCpuRunbk001001Fdd16k() 
{
	ConfigurationConstructor (MSF_CONF_BK1001_FDD);
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateCpuAccelerate(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable (m_pChip->CanAccelerate ());
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnCpuAccelerate() 
{
	m_pChip->AccelerateCPU ();	
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateCpuSlowdown(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable (m_pChip->CanSlowDown ());
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnCpuSlowdown() 
{
	m_pChip->SlowdownCPU ();
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnCpuNormalspeed() 
{
	m_pChip->NormalizeCPU ();	
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateOptionsEnableSpeaker(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck (m_speaker.IsSoundEnabled ());	
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnOptionsEnableSpeaker() 
{
	m_speaker.EnableSound (!m_speaker.IsSoundEnabled ());
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateOptionsEnableCovox(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck (m_covox.IsSoundEnabled ());
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnOptionsEnableCovox() 
{
	m_covox.EnableSound (!m_covox.IsSoundEnabled ());
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateOptionsEmulateBkkeyboard(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck (m_bBKKeyboard);
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnOptionsEmulateBkkeyboard() 
{
	m_bBKKeyboard = !m_bBKKeyboard;		
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateOptionsEnableJoystick(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck (m_bJoystick);
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnOptionsEnableJoystick() 
{
	m_bJoystick = !m_bJoystick;
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateOptionsUseSavesdirectory(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck (m_bSavesDefault);	
	pCmdUI->Enable (m_bEmulateSaveTape);
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnOptionsUseSavesdirectory() 
{
	m_bSavesDefault = !m_bSavesDefault;
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateOptionsEmulateTapeLoading(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck (m_bEmulateLoadTape);
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnOptionsEmulateTapeLoading() 
{
	m_bEmulateLoadTape = !m_bEmulateLoadTape;
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateOptionsEmulateTapeSaving(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck (m_bEmulateSaveTape);
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnOptionsEmulateTapeSaving() 
{
	m_bEmulateSaveTape = !m_bEmulateSaveTape;
/*
	if (m_bEmulateSaveTape)
	{
		CPanel* pTape = m_pLayout->GetPanel ("Tape control");
		pTape->Show (FALSE);
	}
*/
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnOptionsDiskimagemanager() 
{
	CWindowMode mode (&m_screen, m_pChip);

	CImageManagerDlg dlg;
	dlg.DoModal ();
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnOptionsTapemanager() 
{
	CWindowMode mode (&m_screen, m_pChip);
	
	CTapeManagerDlg dlg;
	dlg.DoModal ();
	
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateOptionsShowperfomanceonstatusbar(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck (m_bShowPerfomance);
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnOptionsShowperfomanceonstatusbar() 
{
	m_bShowPerfomance = !m_bShowPerfomance;	
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateDebugBreak(CCmdUI* pCmdUI) 
{
	CString strMenu;

	if (m_pChip->IsCPURun ())
		strMenu.LoadString (IDS_MENU_DEBUG_BREAK);
	else
		strMenu.LoadString (IDS_MENU_DEBUG_CONTINUE);

	pCmdUI->SetText (strMenu);
	InitMenu ();
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnDebugBreak() 
{
	if (!m_pChip->IsCPURun ())
	{
		m_pChip->RunCPU ();
		SetFocusToBK ();
	}
	else
	{
		m_pChip->BreakCPU ();
		SetFocusToDebug ();
	}
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateDebugStepinto(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable (!m_pChip->IsCPURun ());
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnDebugStepinto() 
{
	m_pChip->RunInto ();
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateDebugStepover(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable (!m_pChip->IsCPURun ());
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnDebugStepover() 
{
	m_pChip->RunOver ();
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateDebugStepout(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable (!m_pChip->IsCPURun ());
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnDebugStepout() 
{
	m_pChip->RunOut ();
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateDebugRuntocursor(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable (!m_pChip->IsCPURun ());
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnDebugRuntocursor() 
{
	m_pChip->RunToAddr (m_debugger.GetCursorAddress ());
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnDebugBreakpoint() 
{
	if (!m_debugger.SetSimpleBreakpoint ())
		m_debugger.RemoveBreakpoint ();

	m_debugCtrl.Invalidate (FALSE);
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateViewToolbar(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck (m_toolBarFloater.IsWindowVisible ());
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateDebugSetaddress(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here	
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnDebugSetaddress() 
{
	CDebugAddressDlg dlg;

	dlg.m_debug_address = IntToString (m_debugger.GetCurrentAddress (), 8);
	dlg.m_dump_address = IntToString (m_memoryCtrl.GetDumpAddress (), 8);
	
	if (dlg.DoModal () == IDOK)
	{
		WORD debug_address = OctStringToWord (dlg.m_debug_address);
		WORD dump_address = OctStringToWord (dlg.m_dump_address);
		
		SetDebugCtrlsState (debug_address, dump_address);
	}
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnViewToolbar() 
{
	BOOL bFlag = m_toolBarFloater.IsWindowVisible () ? SW_HIDE : SW_SHOW;
	m_toolBarFloater.ShowWindow (bFlag);
	m_pLayout->Redraw (TRUE);
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateViewStatusBar(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck (m_pLayout->GetPanel (_T("Status bar"))->IsShown ());
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnViewStatusBar() 
{
	BOOL bShow = m_pLayout->GetPanel (_T("Status bar"))->IsShown ();
	m_pLayout->GetPanel (_T("Status bar"))->Show (!bShow);
	m_pLayout->Redraw (TRUE);
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateViewDebugwindow(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck (m_debugWndFloater.IsWindowVisible ());
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnViewDebugwindow() 
{
	int nShow = m_debugWndFloater.IsWindowVisible () ? SW_HIDE : SW_SHOW;
	m_debugWndFloater.ShowWindow (nShow);
	m_pLayout->Redraw (TRUE);
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateViewMemorydump(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck (m_memoryWndFloater.IsWindowVisible ());
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnViewMemorydump() 
{
	int nShow = m_memoryWndFloater.IsWindowVisible () ? SW_HIDE : SW_SHOW;
	m_memoryWndFloater.ShowWindow (nShow);
	m_pLayout->Redraw (TRUE);
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateViewColormode(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck (m_screen.IsColorMode ());	
	
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnViewColormode() 
{
	m_screen.SetColorMode (!m_screen.IsColorMode ());

	m_pChip->RefreshScreen ();
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnViewFullscreenmode() 
{
	if (m_screen.IsFullScreenMode ())
		m_screen.SetWindowMode ();
	else
		m_screen.SetFullScreenMode ();

	m_pChip->RefreshScreen ();
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateViewAdaptivebwmode(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck (m_screen.IsAdaptMode ());
	
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnViewAdaptivebwmode() 
{
	m_screen.SetAdaptMode (!m_screen.IsAdaptMode ());
	m_screen.InitColorTables ();

	m_pChip->RefreshScreen ();
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateViewPreserveCurrentMode(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable (m_bEnableSaveCurrentMode);
	pCmdUI->SetCheck (m_screen.IsSaveCurrentMode ());
	
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnViewPreserveCurrentMode() 
{
	m_screen.SetSaveCurrentMode (!m_screen.IsSaveCurrentMode ());	
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnUpdateViewTapeControl(CCmdUI* pCmdUI) 
{
	CPanel* pTape = m_pLayout->GetPanel ("Tape control");
	pCmdUI->SetCheck (pTape->IsShown ());
	//pCmdUI->Enable (!m_bEmulateSaveTape);
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnViewTapeControl() 
{
	CPanel* pTape = m_pLayout->GetPanel ("Tape control");
	pTape->Show (!pTape->IsShown ());
	m_pLayout->Redraw (TRUE);
}



/////////////////////////////////////////////////////////////////////////////
HRESULT CMainFrame::OnCpuBreak (WPARAM wParam, LPARAM lParam)
{
	WORD pc = m_pChip->GetWordIndirect (PC);
	SetDebugCtrlsState (pc, m_memoryCtrl.GetDumpAddress ());

	return TRUE;
}



/////////////////////////////////////////////////////////////////////////////
HRESULT CMainFrame::OnDebugModify (WPARAM wParam, LPARAM lParam)
{
	if (m_pChip)
	{
		int addr = (int)lParam;
		WORD value = (WORD)wParam;

		if (addr == PSW)
			m_pChip->SetPSW (value);
		else
			m_pChip->SetWordIndirect (addr, value);

		if (addr == PC)
			m_debugger.SetCurrentAddress (value);
	}

	return TRUE;
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::SetDebugCtrlsState ()
{
	// Set memory ctrl CPU registers
	m_memoryCtrl.SetAddress (R0, m_debugger.GetWordValue (R0));
	m_memoryCtrl.SetAddress (R1, m_debugger.GetWordValue (R1));
	m_memoryCtrl.SetAddress (R2, m_debugger.GetWordValue (R2));
	m_memoryCtrl.SetAddress (R3, m_debugger.GetWordValue (R3));
	m_memoryCtrl.SetAddress (R4, m_debugger.GetWordValue (R4));
	m_memoryCtrl.SetAddress (R5, m_debugger.GetWordValue (R5));
	m_memoryCtrl.SetAddress (SP, m_debugger.GetWordValue (SP));
	m_memoryCtrl.SetAddress (PC, m_debugger.GetWordValue (PC));

	// Set memory ctrl PSW
	m_memoryCtrl.SetPSW (m_debugger.GetPSW ());

	// Set memory ctrl System registers
	m_memoryCtrl.SetAddress (0177660, m_debugger.GetWordValue (0177660));
	m_memoryCtrl.SetAddress (0177662, m_debugger.GetWordValue (0177662));
	m_memoryCtrl.SetAddress (0177664, m_debugger.GetWordValue (0177664));
	m_memoryCtrl.SetAddress (0177706, m_debugger.GetWordValue (0177706));
	m_memoryCtrl.SetAddress (0177710, m_debugger.GetWordValue (0177710));
	m_memoryCtrl.SetAddress (0177712, m_debugger.GetWordValue (0177712));
	m_memoryCtrl.SetAddress (0177714, m_debugger.GetWordValue (0177714));
	m_memoryCtrl.SetAddress (0177716, m_debugger.GetWordValue (0177716));
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::SetDebugCtrlsState (WORD debugAddress, WORD dumpAddress)
{
	// Set debug ctrl line
	m_debugger.SetCurrentAddress (debugAddress);

	// Set memory ctrl memory dump
	WORD buff[1024];
	m_debugger.GetMemoryDump (buff, dumpAddress, m_memoryCtrl.GetDumpRowsCount () * m_memoryCtrl.GetDumpColsCount ());
	m_memoryCtrl.SetDump (dumpAddress, buff);

	SetDebugCtrlsState ();
}	



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::SetFocusToBK ()
{
	AfxGetMainWnd ()->SetFocus ();
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::SetFocusToDebug ()
{
	m_debugCtrl.SetFocus ();
}


/////////////////////////////////////////////////////////////////////////////
void CMainFrame::OnEditTapeRecord() 
{
	m_tapeCtrl.PressRecordButton ();
}



/////////////////////////////////////////////////////////////////////////////
void CMainFrame::Print (CPrintDlg* pPrintDlg, WORD startAddress, WORD endAddress)
{
	CDC dc;

	// Attach a printer DC
	dc.Attach (pPrintDlg->GetPrinterDC ());
    dc.m_bPrinting = TRUE;
	
	
	// Initialise print document details
    DOCINFO di;
    ::ZeroMemory (&di, sizeof (DOCINFO));
    di.cbSize = sizeof (DOCINFO);
    di.lpszDocName = m_strPrintTitle;

	// Begin a new print job
    BOOL bPrintingOK = dc.StartDoc (&di);

    // Get the printing extents and store in the m_rectDraw field of a 
    // CPrintInfo object
    CPrintInfo Info;
	
	if (m_bPrintScreen)
		Info.SetMaxPage (1);
	else
		Info.SetMaxPage (pPrintDlg->m_nPages);
	
    Info.m_rectDraw.SetRect(0,0, dc.GetDeviceCaps (HORZRES), dc.GetDeviceCaps (VERTRES));
	
    //OnBeginPrinting (&dc, &Info);

    for (UINT page = Info.GetMinPage (); page <= Info.GetMaxPage () && bPrintingOK; page++)
    {
		// begin new page
        dc.StartPage();                         
        Info.m_nCurPage = page;

        startAddress = PrintPage (&dc, &Info, startAddress, endAddress); 

        bPrintingOK = (dc.EndPage () > 0);  // end page
    }

    //OnEndPrinting (&dc, &Info);

    if (bPrintingOK)
		// end a print job
        dc.EndDoc ();
    else
		// abort job.
        dc.AbortDoc ();

	// detach the printer DC
    dc.Detach();
}



////////////////////////////////////////////////////////////////////////////
WORD CMainFrame::PrintPage (CDC* pDC, CPrintInfo* pInfo, WORD currAddress, WORD endAddress)
{
	// Get paper sizes
	int width = pInfo->m_rectDraw.Width ();
	int height = pInfo->m_rectDraw.Height ();

	int lines_per_page = PRINT_LINES_PER_PAGE; // Lines per page
	int fnt_height = height / lines_per_page;
	int tabs = width / PRINT_TAB_LENGTH;

	CPen pen;
	pen.CreatePen (PS_SOLID, 10, RGB (0, 0, 0));

	CFont fnt;
	fnt.CreatePointFont (fnt_height, "Arial", pDC);
	
	CFont* pOld = pDC->SelectObject (&fnt);
	CPen* pOldPen = pDC->SelectObject (&pen);

	pDC->SetTextColor (RGB (0, 0, 0));
	pDC->SetBkColor (RGB (255, 255, 255));

	if (m_bPrintScreen)
	{
		// Print Screenshot
		CDC dc;
		dc.CreateCompatibleDC (pDC);

		// Get Screenshot handle
		HBITMAP hScreenShot = m_screen.GetScreenshot ();

		BITMAP bm;
		::GetObject (hScreenShot, sizeof (bm), &bm);

		HBITMAP hOld = (HBITMAP)dc.SelectObject (hScreenShot);
		
		// Calculate print rectangle
		int len = pInfo->m_rectDraw.Width () < pInfo->m_rectDraw.Height () 
			? pInfo->m_rectDraw.Width () : pInfo->m_rectDraw.Height ();

		len = len / 2;

		int xOfs = (pInfo->m_rectDraw.Width () - len) / 2;
		int yOfs = (pInfo->m_rectDraw.Height () - len) / 2;

		// Stretch Screenshot to Print page
		int stretchMode = m_bPrintInverse ? SRCINVERT : SRCCOPY;

		pDC->StretchBlt (xOfs, yOfs, len, len, &dc, 0, 0, bm.bmWidth, bm.bmHeight, stretchMode);
		dc.SelectObject (hOld);

		::DeleteObject (hScreenShot);
	}
	else
	{
		for (int line = 3; line < lines_per_page - 3; line++)
		{
			CRect rcString (0, line * height / lines_per_page, width, line * height / lines_per_page + fnt_height);

			CString strInstr;
			CString strArg;
			WORD codes[3];
			WORD addrOfs = m_debugger.DebugInstruction (currAddress, strInstr, strArg, codes) * 2;

			CString str;
			str.Format ("%07o  |", currAddress);

			rcString.left = tabs * 1;
			pDC->DrawText (str, &rcString, DT_LEFT|DT_EXPANDTABS);

			rcString.left = tabs * 2;
			pDC->DrawText (strInstr, &rcString, DT_LEFT|DT_EXPANDTABS);
			
			rcString.left = tabs * 3;
			pDC->DrawText (strArg, &rcString, DT_LEFT|DT_EXPANDTABS);

			switch (addrOfs / 2)
			{
			case 1:	str.Format ("  |%06o", codes[0]); break;
			case 2:	str.Format ("  |%06o   %06o", codes[0], codes[1]); break;
			case 3:	str.Format ("  |%06o   %06o   %06o", codes[0], codes[1], codes[2]); break;
			}

			rcString.left = tabs * 5;
			pDC->DrawText (str, &rcString, DT_LEFT|DT_EXPANDTABS);

			currAddress += addrOfs;

			if (currAddress >= endAddress)
				break;
		}
	}

	pDC->MoveTo (0, 3 * height / lines_per_page - fnt_height / 2);
	pDC->LineTo (width, 3 * height / lines_per_page - fnt_height / 2);
	pDC->MoveTo (0, (lines_per_page - 3) *  height / lines_per_page + fnt_height / 2);
	pDC->LineTo (width, (lines_per_page - 3) * height / lines_per_page + fnt_height / 2);

	CRect rcString;
	rcString.SetRect (0, height / lines_per_page, width, height / lines_per_page + fnt_height);
	
	CString strTitle;
	strTitle.LoadString (IDS_EMUL_TITLE);

	int oldMode = pDC->SetBkMode (TRANSPARENT);
	pDC->DrawText (strTitle, &rcString, DT_LEFT|DT_EXPANDTABS);
	pDC->DrawText (m_strPrintTitle, &rcString, DT_RIGHT|DT_EXPANDTABS);
	pDC->SetBkMode (oldMode);

	pDC->SetTextColor (RGB (255, 255, 255));
	pDC->SetBkColor (RGB (0, 0, 0));

	rcString.SetRect (0, (lines_per_page - 2) * height / lines_per_page, width, (lines_per_page - 2) * height / lines_per_page + fnt_height);

	CString strPages;
	strPages.Format (IDS_PRINT_PAGES, pInfo->m_nCurPage, pInfo->GetMaxPage ());

	pDC->PatBlt (rcString.left, rcString.top, rcString.Width (), rcString.Height (), BLACKNESS);
	pDC->DrawText (strPages, &rcString, DT_RIGHT|DT_EXPANDTABS);

	pDC->SelectObject (pOld);
	pDC->SelectObject (pOldPen);

	return currAddress;
}
