/*************************************************
*            BK Emulator for Window 3.0          *
*************************************************/

/*
File: MSFManager.cpp
Version: 1.0.0
Written by: Yuriy Kalmykov <kalmykov@stoik.com>
    Copyright (c) 2002-2004 Yuriy Kalmykov

    BK Emulator is a program emulated hardware environment for running
code for BK 0010(01) in different configurations. 
           
    This code may be used in compiled form in any way you desire.
This file or it's parts can't be redistributed without the authors
written consent, but can be modified for your private needs.
    Providing that this notice and the authors name and all copyright
notices remains intact.

    Please, an email me to know how you are using it and where. You can
ask me for any information about this below code or any attendant
knowledge.
    
    This file is provided "as is" with no expressed or implied warranty.
The author accepts no liability for any damage or loss of business that
this product may cause.
*/




// MSFManager.cpp: implementation of the CMSFManager class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "bk.h"
#include "globals.h"
#include "MSFManager.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CMSFManager::CMSFManager()
{
	m_header.type = MSF_STATE_ID;
	m_header.version = 10;
	m_header.configuration = MSF_CONF_BK1001;
}

CMSFManager::~CMSFManager()
{
	ReleaseFile ();
}



//////////////////////////////////////////////////////////////////////
BOOL CMSFManager::ReleaseFile ()
{
	if (!m_fileMSF.GetFileName ().IsEmpty ())
	{
		if (!GetFileName (m_fileMSF.GetFileName ().IsEmpty ()))
		{
			m_fileMSF.Close ();
			m_blocks.RemoveAll ();
			return TRUE;
		}
	}

	return FALSE;
}



//////////////////////////////////////////////////////////////////////
BOOL CMSFManager::Seek (int nOffset, DWORD nFrom)
{
	try
	{
		m_fileMSF.Seek (nOffset, nFrom);
	}
	catch (CFileException)
	{
		return FALSE;
	}

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CMSFManager::OpenFile (CString strPath, BOOL bLoad)
{
	ReleaseFile ();
		
	DWORD flag = CFile::modeRead;

	if (!bLoad)
		flag |= CFile::modeCreate|CFile::modeWrite;

	if (!m_fileMSF.Open (strPath, flag))
		return FALSE;

	m_bOpenForLoad = bLoad;
	int nFilePos = 0;

	if (!m_bOpenForLoad)
	{
		m_fileMSF.Write (&m_header, sizeof (m_header));
	}
	else
	{
		if (m_fileMSF.Read (&m_header, sizeof (m_header)) != sizeof (m_header))
			return FALSE;

		nFilePos += sizeof (m_header);

		if (m_header.type != MSF_STATE_ID || m_header.version > 10)
			return FALSE;

		MSF_BLOCK_INFO bi;
		while (m_fileMSF.Read (&bi.header, sizeof (MSF_BLOCK_HEADER)) == sizeof (MSF_BLOCK_HEADER))
		{
			bi.offset = nFilePos;
			m_blocks.Add (bi);
			
			if (!Seek (bi.header.length - sizeof (MSF_BLOCK_HEADER), SEEK_CUR))
				return FALSE;

			nFilePos += bi.header.length;
		}
	}
		
	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CMSFManager::FindBlock (DWORD blockType, MSF_BLOCK_INFO* pBi)
{
	for (int n = 0; n < m_blocks.GetSize (); n++)
	{
		MSF_BLOCK_INFO bi = m_blocks[n];

		if (bi.header.type == blockType)
		{
			*pBi = bi;
			return TRUE;
		}
	}

	return FALSE;
}



//////////////////////////////////////////////////////////////////////
BOOL CMSFManager::GetBlockHeader (MSF_BLOCK_INFO* pBi)
{
	if (!Seek (pBi->offset, SEEK_SET))
		return FALSE;
	
	if (!Seek (sizeof (MSF_BLOCK_HEADER), SEEK_CUR))
		return FALSE;

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CMSFManager::GetBlockData (BYTE* pBuff, DWORD length)
{
	if (m_fileMSF.Read (pBuff, length) != length)
		return FALSE;

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CMSFManager::GetBlockPreview (HBITMAP* hScreenshot)
{
	MSF_BLOCK_INFO bi;

	if (!FindBlock (MSF_BLOCKTYPE_PREVIEW, &bi))
		return FALSE;

	if (!GetBlockHeader (&bi))
		return FALSE;

	BITMAPINFOHEADER bmih;
	if (!GetBlockData ((BYTE*)&bmih, sizeof (bmih)))
		return FALSE;

	BYTE* pBits = NULL;
	HBITMAP hBm = NULL;;
	if (!(hBm = ::CreateDIBSection (NULL, (BITMAPINFO*)&bmih, DIB_RGB_COLORS, (void**)&pBits, NULL, 0)))
		return FALSE;
	
	if (!GetBlockData (pBits, bmih.biSizeImage))
		return FALSE;

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CMSFManager::GetBlockBaseMemory (BYTE* pMemory)
{
	MSF_BLOCK_INFO bi;

	if (!FindBlock (MSF_BLOCKTYPE_BASEMEMORY, &bi))
		return FALSE;

	if (!GetBlockHeader (&bi))
		return FALSE;

	if (!GetBlockData (pMemory, 0200000))
		return FALSE;
	
	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CMSFManager::GetBlockCPURegisters (MSF_CPU_REGISTERS* pCPURegisters)
{
	MSF_BLOCK_INFO bi;

	if (!FindBlock (MSF_BLOCKTYPE_CPU_REGISTERS, &bi))
		return FALSE;

	if (!GetBlockHeader (&bi))
		return FALSE;

	if (!GetBlockData ((BYTE*)pCPURegisters, sizeof (MSF_CPU_REGISTERS)))
		return FALSE;
	
	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CMSFManager::GetBlockExt32Memory (DWORD* pnPage, BYTE* pMemoryExt)
{
	MSF_BLOCK_INFO bi;

	if (!FindBlock (MSF_BLOCKTYPE_10EXT32, &bi))
		return FALSE;
	
	if (!GetBlockHeader (&bi))
		return FALSE;

	if (!GetBlockData ((BYTE*)&pnPage, sizeof (DWORD)))
		return FALSE;
	
	if (!GetBlockData (pMemoryExt, 4 * 8192))
		return FALSE;

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CMSFManager::GetBlockExt16Memory (BOOL* pbBasic, BYTE* pMemoryBasic, BYTE* pMemoryExt)
{
	MSF_BLOCK_INFO bi;

	if (!FindBlock (MSF_BLOCKTYPE_10EXT16, &bi))
		return FALSE;
	
	if (!GetBlockHeader (&bi))
		return FALSE;

	if (!GetBlockData ((BYTE*)pbBasic, sizeof (BOOL)))
		return FALSE;
	
	if (!GetBlockData (pMemoryBasic, 3 * 8192))
		return FALSE;

	if (!GetBlockData (pMemoryExt, 3 * 8192))
		return FALSE;

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CMSFManager::GetBlockTape (BYTE* pPackedWave, DWORD* pnBitsLength, DWORD* pnPackLength)
{
	MSF_BLOCK_INFO bi;

	if (!FindBlock (MSF_BLOCKTYPE_WAVE, &bi))
		return FALSE;

	if (!GetBlockHeader (&bi))
		return FALSE;

	DWORD nPackLength = bi.header.length - sizeof (MSF_BLOCK_HEADER) - sizeof (DWORD);
	
	if (!GetBlockData ((BYTE*)pnBitsLength, sizeof (DWORD)))
		return FALSE;

	if (!pPackedWave)
	{
		*pnPackLength = nPackLength;
		return TRUE;
	}

	if (!GetBlockData ((BYTE*)pPackedWave, nPackLength))
		return FALSE;
	
	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CMSFManager::SetBlockHeader (MSF_BLOCK_INFO* pBi)
{
	try
	{
		m_fileMSF.Write (&pBi->header, sizeof (MSF_BLOCK_HEADER));
	}
	catch (CFileException)
	{
		return FALSE;
	}

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CMSFManager::SetBlockData (BYTE* pBuff, DWORD length)
{
	try
	{
		m_fileMSF.Write (pBuff, length);
	}
	catch (CFileException)
	{
		return FALSE;
	}

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CMSFManager::SetBlockPreview (HBITMAP hScreenshot)
{
	DIBSECTION ds;
	if (!::GetObject (hScreenshot, sizeof (ds), &ds))
		return FALSE;

	MSF_BLOCK_INFO bi;
	bi.header.type = MSF_BLOCKTYPE_PREVIEW;
	bi.header.length = sizeof (MSF_BLOCK_HEADER) + sizeof (ds.dsBmih) + ds.dsBmih.biSizeImage;

	if (!SetBlockHeader (&bi))
		return FALSE;

	if (!SetBlockData ((BYTE*)&ds.dsBmih, sizeof (ds.dsBmih)))
		return FALSE;

	if (!SetBlockData ((BYTE*)ds.dsBm.bmBits, ds.dsBmih.biSizeImage))
		return FALSE;
	
	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CMSFManager::SetBlockBaseMemory (BYTE* pMemory)
{
	MSF_BLOCK_INFO bi;
	bi.header.type = MSF_BLOCKTYPE_BASEMEMORY;
	bi.header.length = sizeof (MSF_BLOCK_HEADER) + 0200000;

	if (!SetBlockHeader (&bi))
		return FALSE;

	if (!SetBlockData (pMemory, 0200000))
		return FALSE;
	
	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CMSFManager::SetBlockCPURegisters (MSF_CPU_REGISTERS* pCPURegisters)
{
	MSF_BLOCK_INFO bi;
	bi.header.type = MSF_BLOCKTYPE_CPU_REGISTERS;
	bi.header.length = sizeof (MSF_BLOCK_HEADER) + sizeof (MSF_CPU_REGISTERS);

	if (!SetBlockHeader (&bi))
		return FALSE;

	if (!SetBlockData ((BYTE*)pCPURegisters, sizeof (MSF_CPU_REGISTERS)))
		return FALSE;

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CMSFManager::SetBlockExt32Memory (DWORD nPage, BYTE* pMemoryExt)
{
	MSF_BLOCK_INFO bi;
	bi.header.type = MSF_BLOCKTYPE_10EXT32;
	bi.header.length = sizeof (MSF_BLOCK_HEADER) + sizeof (DWORD) + 4 * 8192;

	if (!SetBlockHeader (&bi))
		return FALSE;

	if (!SetBlockData ((BYTE*)&nPage, sizeof (DWORD)))
		return FALSE;

	if (!SetBlockData (pMemoryExt, 4 * 8192))
		return FALSE;

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CMSFManager::SetBlockExt16Memory (BOOL bBasic, BYTE* pMemoryBasic, BYTE* pMemoryExt)
{
	MSF_BLOCK_INFO bi;
	bi.header.type = MSF_BLOCKTYPE_10EXT16;
	bi.header.length = sizeof (MSF_BLOCK_HEADER) + sizeof (BOOL) + (3 + 3) * 8192;

	if (!SetBlockHeader (&bi))
		return FALSE;

	if (!SetBlockData ((BYTE*)&bBasic, sizeof (DWORD)))
		return FALSE;

	if (!SetBlockData (pMemoryBasic, 3 * 8192))
		return FALSE;

	if (!SetBlockData (pMemoryExt, 3 * 8192))
		return FALSE;

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CMSFManager::SetBlockTape (BYTE* pPackedWave, DWORD nBitsLength, DWORD nPackLength)
{
	MSF_BLOCK_INFO bi;
	bi.header.type = MSF_BLOCKTYPE_WAVE;
	bi.header.length = sizeof (MSF_BLOCK_HEADER) + sizeof (DWORD) + nPackLength;

	if (!SetBlockHeader (&bi))
		return FALSE;

	if (!SetBlockData ((BYTE*)&nBitsLength, sizeof (DWORD)))
		return FALSE;

	if (!SetBlockData ((BYTE*)pPackedWave, nPackLength))
		return FALSE;

	return TRUE;
}
