/*************************************************
*            BK Emulator for Window 3.0          *
*************************************************/

/*
File: Floater.cpp
Version: 1.0.0
Written by: Yuriy Kalmykov <kalmykov@stoik.com>
    Copyright (c) 2002-2004 Yuriy Kalmykov

    BK Emulator is a program emulated hardware environment for running
code for BK 0010(01) in different configurations. 
           
    This code may be used in compiled form in any way you desire.
This file or it's parts can't be redistributed without the authors
written consent, but can be modified for your private needs.
    Providing that this notice and the authors name and all copyright
notices remains intact.

    Please, an email me to know how you are using it and where. You can
ask me for any information about this below code or any attendant
knowledge.
    
    This file is provided "as is" with no expressed or implied warranty.
The author accepts no liability for any damage or loss of business that
this product may cause.
*/




// Floater.cpp : implementation file
//

#include "stdafx.h"
#include "Panel.h"
#include "Floater.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


/////////////////////////////////////////////////////////////////////////////
// CFloater

CFloater::CFloater()
{
	m_bFloat = TRUE;
	m_fLastAligned = ALIGN_NOWHERE;
	m_bEnableSplitter = FALSE;
	m_bMoveSplitter = FALSE;
	m_dockingSize = 0.0;
	m_fixedDockingSize = 0;
	m_captionSize = DEFAULT_CAPTION_SIZE;

	m_pDoubleFactory = NULL;
	m_pIntFactory = NULL;
	m_pWnd = NULL;
	m_pParentWnd = NULL;
	m_pLayout = NULL;

	m_pFloaterLayout = NULL;
	m_pWrapperPanel = NULL;
	m_pDockPanel = NULL;
	m_pLastDockPanel = NULL;
}

CFloater::~CFloater()
{
	delete m_pFloaterLayout;

	if (m_bFloat)
		delete m_pWrapperPanel;
}


BEGIN_MESSAGE_MAP(CFloater, CWnd)
	//{{AFX_MSG_MAP(CFloater)
	ON_WM_SIZE()
	ON_WM_NCHITTEST()
	ON_WM_LBUTTONDOWN()
	ON_WM_LBUTTONUP()
	ON_WM_MOUSEMOVE()
	ON_WM_CAPTURECHANGED()
	ON_WM_PAINT()
	ON_WM_CLOSE()
	ON_WM_LBUTTONDBLCLK()
	ON_WM_NCCALCSIZE()
	ON_WM_NCPAINT()
	ON_WM_SHOWWINDOW()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()


/////////////////////////////////////////////////////////////////////////////
// CFloater message handlers


BOOL CFloater::PreCreateWindow(CREATESTRUCT& cs) 
{
	// Create own class for popup window
	cs.lpszClass = AfxRegisterWndClass (CS_HREDRAW|CS_VREDRAW|CS_DBLCLKS, ::LoadCursor (NULL, IDC_ARROW));
	
	return CWnd::PreCreateWindow(cs);
}



/////////////////////////////////////////////////////////////////////////////
void CFloater::CreateFloater (CWnd* pParent, CPanel* pLayout, double dockingSize
							  , PANELFACTORY_LAYOUT pFactory, PANELFACTORY_DOUBLE pDoubleFactory)
{
	ASSERT (!m_hWnd);

	// Attach floater to specified by pLayout layout
	ASSERT (pLayout);

	// Create floater window
	CreateEx (WS_EX_TOOLWINDOW, NULL, _T("New Floater"), WS_CLIPSIBLINGS|WS_CAPTION|WS_POPUPWINDOW|WS_THICKFRAME/*|WS_VISIBLE*/, CRect (0, 0, 100, 100), pParent, 0);

	// Create floater internal layout
	if (!pFactory)
		m_pFloaterLayout = new CPanel (this);
	else
		m_pFloaterLayout = pFactory (this);

	// Save layout pointer
	m_pLayout = pLayout;
	m_pDoubleFactory = pDoubleFactory;
	
	// Save initial window sizes
	m_dockingSize = dockingSize;
	m_bFixedSize = FALSE;

	GetWindowRect (&m_rcWindow);
}



/////////////////////////////////////////////////////////////////////////////
void CFloater::CreateFloater (CWnd* pParent, CPanel* pLayout, int dockingSize
							  , PANELFACTORY_LAYOUT pFactory, PANELFACTORY_INT pIntFactory)
{
	ASSERT (!m_hWnd);

	// Attach floater to specified by pLayout layout
	ASSERT (pLayout);

	// Create floater window
	CreateEx (WS_EX_TOOLWINDOW, NULL, _T("New Floater"), WS_CLIPSIBLINGS|WS_CAPTION|WS_POPUPWINDOW|WS_THICKFRAME/*|WS_VISIBLE*/, CRect (0, 0, 100, 100), pParent, 0);

	// Create floater internal layout
	if (!pFactory)
		m_pFloaterLayout = new CPanel (this);
	else
		m_pFloaterLayout = ((PANELFACTORY_LAYOUT)pFactory) (this);

	// Save layout pointer
	m_pLayout = pLayout;
	m_pIntFactory = pIntFactory;

	// Save initial window sizes
	m_fixedDockingSize = dockingSize;
	m_bFixedSize = TRUE;

	GetWindowRect (&m_rcWindow);
}



/////////////////////////////////////////////////////////////////////////////
void CFloater::DrawDebugInfo (BOOL bEnabled)
{
	ASSERT (m_pFloaterLayout);
	m_bDrawDebugInfo = bEnabled;
	m_pFloaterLayout->DrawDebugInfo (m_bDrawDebugInfo);
}



/////////////////////////////////////////////////////////////////////////////
void CFloater::InsertWindow (CWnd* pWnd, int windowAlign)
{
	// Save inserted window pointers
	m_pWnd = pWnd;
	m_pParentWnd = m_pWnd->GetParent ();

	// Resize inserted window
	m_pWnd->SetWindowPos (NULL, 0, 0, 100, 100, SWP_NOZORDER);

	// Now Floater is parent for inserted window
	m_pWnd->SetParent (this);

	// Add inserted window to internal layout
	m_pFloaterLayout->InsertWindow (pWnd, windowAlign);
	//m_pFloaterLayout->SetBorder (6);
	m_pFloaterLayout->Redraw ();
}



/////////////////////////////////////////////////////////////////////////////
void CFloater::InsertWindow (CWnd* pWnd, int ctrlID, int windowAlign)
{
	// Save inserted window pointers
	m_pWnd = pWnd;
	m_pParentWnd = m_pWnd->GetParent ();

	// Resize inserted window
	m_pWnd->SetWindowPos (NULL, 0, 0, 100, 100, SWP_NOZORDER);

	// Now Floater is parent for inserted window
	m_pWnd->SetParent (this);

	// Add inserted window to internal layout
	m_pFloaterLayout->InsertWindow (pWnd, ctrlID, windowAlign);
	m_pFloaterLayout->SetBorder (6);
	m_pFloaterLayout->Redraw ();
}



/////////////////////////////////////////////////////////////////////////////
void CFloater::	InsertWindow (CWnd* pWnd, int firstID, int lastID, int windowAlign)
{
	// Save inserted window pointers
	m_pWnd = pWnd;
	m_pParentWnd = m_pWnd->GetParent ();

	// Resize inserted window
	m_pWnd->SetWindowPos (NULL, 0, 0, 100, 100, SWP_NOZORDER);

	// Now Floater is parent for inserted window
	m_pWnd->SetParent (this);

	// Add inserted window to internal layout
	m_pFloaterLayout->InsertWindow (pWnd, firstID, lastID, windowAlign);
	m_pFloaterLayout->SetBorder (6);
	m_pFloaterLayout->Redraw ();
}



/////////////////////////////////////////////////////////////////////////////
void CFloater::EnableFloating (BOOL bFloat, CPoint ptPos, BOOL bShow)
{
	// If window floating and already recreated - don't destroy it
	if (bFloat == m_bFloat && bFloat)
		return;

	// Panel can't dockin to it self
	if (m_pDockPanel == m_pWrapperPanel)
		return;

	// Delete wrapper panel if exists
	if (m_pWrapperPanel)
	{
		delete m_pLayout->RemovePanel (m_pWrapperPanel);
		m_pWrapperPanel = NULL;
	}

	// Save window caption
	CString strCaption;
	GetWindowText (strCaption);

	// Get current window position and sizes
	CRect rcCurrent;
	GetWindowRect (&rcCurrent);

	// Set style
	int style = WS_CLIPSIBLINGS|WS_VISIBLE;

	if (!bFloat && m_fAligned != ALIGN_NOWHERE)
	{
		// If window go to docking
		
		// save current sizes only if panel floating
		if (m_bFloat)
			m_rcWindow = rcCurrent;

		// Set new position and sizes
		rcCurrent.right = rcCurrent.left + m_rcWindow.Width ();
		rcCurrent.bottom = rcCurrent.top + m_rcWindow.Height ();

		// Set style
		style |= WS_CHILD;

		// Recreate window
		m_pFloaterLayout->SetBase (NULL);
		m_pWnd->ShowWindow (SW_HIDE);
		m_pWnd->SetParent (NULL);
		
		DestroyWindow ();
		CreateEx (WS_EX_TOOLWINDOW, NULL, strCaption, style, rcCurrent, m_pParentWnd, 0);
		m_pWnd->SetParent (this);
		m_pWnd->ShowWindow (SW_SHOW);

		m_pFloaterLayout->SetBase (this);
		m_pFloaterLayout->Redraw (TRUE);

		// Dock floater to panel
		DockPanel ();

		// Set new state is docking
		m_bFloat = FALSE;

		// Save last states for double click
		m_fLastAligned = m_fAligned;
		m_pLastDockPanel = m_pDockPanel;
	}
	else
	{
		// If window go to floating

		// Get window saved sizes
		int width = m_rcWindow.Width ();
		int height = m_rcWindow.Height ();

		// Set new position and sizes
		rcCurrent.left = ptPos.x - width / 2;
		rcCurrent.top = ptPos.y - height / 2;
		rcCurrent.right = rcCurrent.left + width;
		rcCurrent.bottom = rcCurrent.top + height;

		// Recreate window
		m_pFloaterLayout->SetBase (NULL);
		
		m_pWnd->ShowWindow (SW_HIDE);
		m_pWnd->SetParent (NULL);
		DestroyWindow ();
		CreateEx (WS_EX_TOOLWINDOW, NULL, strCaption, WS_CLIPSIBLINGS|WS_CAPTION|WS_POPUPWINDOW|WS_THICKFRAME|WS_VISIBLE
			, rcCurrent, m_pParentWnd, 0);
		m_pWnd->SetParent (this);
		m_pWnd->ShowWindow (SW_SHOW);

		m_pFloaterLayout->SetBase (this);
		m_pFloaterLayout->Redraw (TRUE);

		// Set new state is floating
		m_bFloat = TRUE;
	}

	if (!bShow)
		// Hide panel after docking
		ShowWindow (SW_HIDE);

	// Redraw layout
	m_pLayout->Redraw (TRUE);
}



/////////////////////////////////////////////////////////////////////////////
void CFloater::DrawDragMark (CPoint ptPos)
{
	// Calculate offset
	CPoint ptOffs (ptPos);
	ptOffs -= m_ptInitPos;

	// Calculate frame sizes
	CRect rcWindow;
	GetWindowRect (&rcWindow);

	rcWindow.right = rcWindow.left + m_rcWindow.Width ();
	rcWindow.bottom = rcWindow.top + m_rcWindow.Height ();

	rcWindow += ptOffs;
	rcWindow.DeflateRect (2, 2);

	int pen_size = 3;

	// If horizontal aligned stretch frame by vertical
	if (m_fAligned == ALIGN_LEFT || m_fAligned == ALIGN_RIGHT)
	{
		rcWindow.top = m_rcPanel.top;
		rcWindow.bottom = m_rcPanel.bottom;
		rcWindow.InflateRect ((m_fixedDockingSize - m_rcWindow.Width ()) / 2, 0);

		pen_size = 1;
	}

	// If vertical aligned stretch frame by horizontal
	if (m_fAligned == ALIGN_TOP || m_fAligned == ALIGN_BOTTOM)
	{
		rcWindow.left = m_rcPanel.left;
		rcWindow.right = m_rcPanel.right;
		rcWindow.InflateRect (0, (m_fixedDockingSize - m_rcWindow.Height ()) / 2);

		pen_size = 1;
	}

	// If cursor out of frame - center frame correspondently by cursor
	if (!rcWindow.PtInRect (ptPos))
	{
		int width = rcWindow.Width ();
		int height = rcWindow.Height ();
		
		if (m_fAligned != ALIGN_TOP && m_fAligned != ALIGN_BOTTOM)
		{
			rcWindow.left = ptPos.x - width / 2;
			rcWindow.right = rcWindow.left + width;
		}

		if (m_fAligned != ALIGN_LEFT && m_fAligned != ALIGN_RIGHT)
		{
			rcWindow.top = ptPos.y - height / 2;
			rcWindow.bottom = rcWindow.top + height;
		}
	}
		
	// Draw frame
	CWnd* pWnd = CWnd::GetDesktopWindow ();
	CDC* pDC = pWnd->GetDCEx (NULL, DCX_WINDOW|DCX_CACHE|DCX_LOCKWINDOWUPDATE);

	CPen pen;
	pen.CreatePen (PS_SOLID, pen_size, RGB (128, 128, 128));

	int oldMode = pDC->SetROP2 (R2_XORPEN);
	CPen* pOld = pDC->SelectObject (&pen);
	
	pDC->MoveTo (rcWindow.left, rcWindow.top);
	pDC->LineTo (rcWindow.right, rcWindow.top);
	pDC->LineTo (rcWindow.right, rcWindow.bottom);
	pDC->LineTo (rcWindow.left, rcWindow.bottom);
	pDC->LineTo (rcWindow.left, rcWindow.top);

	pDC->SelectObject (pOld);
	pDC->SetROP2 (oldMode);
}




/////////////////////////////////////////////////////////////////////////////
void CFloater::RepositionFloater (CPoint ptPos)
{
	// Panel can't dockin to it self
	if (m_pDockPanel == m_pWrapperPanel && !m_bFloat)
		return;

	// Calculate offset
	CPoint ptOffs (ptPos);
	ptOffs -= m_ptInitPos;

	// Get window position
	CRect rcWindow;
	GetWindowRect (&rcWindow);

	// If window docking recalculate position in parent coordinates
	if (!m_bFloat)
		GetParent ()->ScreenToClient (&rcWindow);

	// Move window
	rcWindow += ptOffs;

	ShowWindow (SW_HIDE);
	SetWindowPos (NULL, rcWindow.left, rcWindow.top, 0, 0, SWP_NOSIZE|SWP_NOZORDER|SWP_SHOWWINDOW);
}



/////////////////////////////////////////////////////////////////////////////
int CFloater::GetPanelSide (CPoint& ptPos, CPanel* pPanel)
{
	// Get alignment for floater in specified panel
	m_pDockPanel = pPanel;
	CRect rcPanel = pPanel->GetScreenRect ();

	int width4 = rcPanel.Width () / 8;
	int height4 = rcPanel.Height () / 8;

	if (ptPos.x >= rcPanel.left && ptPos.x < rcPanel.left + width4)
		return ALIGN_LEFT;
	
	if (ptPos.x >= rcPanel.right - width4 && ptPos.x < rcPanel.right)
		return ALIGN_RIGHT;

	if (ptPos.y >= rcPanel.top && ptPos.y < rcPanel.top + height4)
		return ALIGN_TOP;

	if (ptPos.y >= rcPanel.bottom - height4 && ptPos.y < rcPanel.bottom)
		return ALIGN_BOTTOM;

	m_pDockPanel = NULL;
	return ALIGN_NOWHERE;
}



/////////////////////////////////////////////////////////////////////////////
void CFloater::DockPanel ()
{
	// Create wrapper panel
	if (m_bFixedSize)
		if (m_pIntFactory)
			m_pWrapperPanel = m_pIntFactory (NULL, m_fixedDockingSize);
		else
			m_pWrapperPanel = new CPanel (NULL, m_fixedDockingSize);
	else
		if (m_pDoubleFactory)
			m_pWrapperPanel = m_pDoubleFactory (NULL, m_dockingSize);
		else
			m_pWrapperPanel = new CPanel (NULL, m_dockingSize);
	
	// Insert floater in to panel
	m_pWrapperPanel->InsertWindow (this, WINDOW_STRETCHED);
	m_pWrapperPanel->DrawDebugInfo (m_bDrawDebugInfo);

	// Add splitter if nesassary
	if (m_bEnableSplitter)
		m_pWrapperPanel->EnableSplitter ();
	else
		m_pWrapperPanel->DisableSplitter ();

	// Get dock alignment
	int dock_align = m_pDockPanel->GetAlign ();

	// Add panel with window to layout
	if (!m_pDockPanel->GetParent () 
		|| ((m_fAligned & ALIGN_VER) != (dock_align & ALIGN_VER)))
	{
		//BOOL bAtHead = (m_fAligned & ALIGN_VER) == ALIGN_VER;
		// Reme for BK Emulator
		BOOL bAtHead = FALSE;
		m_pDockPanel->Add (m_pWrapperPanel, m_fAligned, bAtHead);
	}
	else
	{
		if (dock_align == m_fAligned)
			m_pDockPanel->AddBefor (m_pWrapperPanel);
		else
			m_pDockPanel->AddAfter (m_pWrapperPanel);
	}
}



/////////////////////////////////////////////////////////////////////////////
void CFloater::EnableSplitter ()
{
	m_bEnableSplitter = TRUE;
}



/////////////////////////////////////////////////////////////////////////////
void CFloater::DisableSplitter ()
{
	m_bEnableSplitter = FALSE;
}


/////////////////////////////////////////////////////////////////////////////
void CFloater::SetDockingSize (int size)
{
	m_fixedDockingSize = size;
}



/////////////////////////////////////////////////////////////////////////////
int	CFloater::GetDockingSize ()
{
	return m_fixedDockingSize;
}



/////////////////////////////////////////////////////////////////////////////
void CFloater::SetDockingPriority (BOOL bVertical)
{
	m_fDockingPriority = bVertical ? ALIGN_VER : ALIGN_HOR;
}



/////////////////////////////////////////////////////////////////////////////
BOOL CFloater::GetDockingPriority ()
{
	return m_fDockingPriority == ALIGN_VER ? TRUE : FALSE;
}



/////////////////////////////////////////////////////////////////////////////
void CFloater::OnDrawCaption (CDC* pDC, CRect& rcBound)
{
	// Draw caption background
	COLORREF clrBackground = m_pFloaterLayout->GetBackgroundColor ();

	CBrush br;
	br.CreateSolidBrush (clrBackground);

	CBrush* pOld = pDC->SelectObject (&br);
	pDC->PatBlt (rcBound.left, rcBound.top, rcBound.Width (), rcBound.Height (), PATCOPY);
	pDC->SelectObject (pOld);

	// Draw caption
	COLORREF clrLight = RGB (min (GetRValue (clrBackground) + 200, 255)
		, min (GetGValue (clrBackground) + 200, 255)
		, min (GetBValue (clrBackground) + 200, 255));
	COLORREF clrShadow = RGB (max (GetRValue (clrBackground) - 200, 0)
		, max (GetGValue (clrBackground) - 200, 0)
		, max (GetBValue (clrBackground) - 200, 0));

	CPoint ptStart (0, 0);
	CPoint ptOffset (0, 0);
	
	if (m_fAligned == ALIGN_TOP || m_fAligned == ALIGN_BOTTOM)
	{
		ptStart.x = m_captionSize / 3;
		ptStart.y = 0;
		ptOffset.x = 4;
		ptOffset.y = 0;
		rcBound.left = -1;
		rcBound.right = 1;
		rcBound.DeflateRect (0, 2);
	}

	if (m_fAligned == ALIGN_LEFT || m_fAligned == ALIGN_RIGHT)
	{
		ptStart.x = 0;
		ptStart.y = m_captionSize / 3;
		ptOffset.x = 0;
		ptOffset.y = 4;
		rcBound.top = -1;
		rcBound.bottom = 1;
		rcBound.DeflateRect (2, 0);
	}

	rcBound += ptStart;
	pDC->Draw3dRect (rcBound, clrLight, clrShadow);

	rcBound += ptOffset;
	pDC->Draw3dRect (rcBound, clrLight, clrShadow);
}



/////////////////////////////////////////////////////////////////////////////
void CFloater::OnSize(UINT nType, int cx, int cy) 
{
	CWnd::OnSize(nType, cx, cy);
	
	// Save window sizes if floating
	if (m_bFloat)
		GetWindowRect (&m_rcWindow);

	// Remove layout

	if (m_pLayout)
	{
		m_pFloaterLayout->ResizeToWindow ();
		m_pFloaterLayout->RecalcWindowsZorder ();
	}
}



/////////////////////////////////////////////////////////////////////////////
BOOL CFloater::OnCmdMsg(UINT nID, int nCode, void* pExtra, AFX_CMDHANDLERINFO* pHandlerInfo) 
{
	// Remap OnCmdMsg to parent window
	// Now notifications available in Floater parent
	if (m_pParentWnd)
		m_pParentWnd->OnCmdMsg (nID, nCode, pExtra, pHandlerInfo);
	
	return CWnd::OnCmdMsg(nID, nCode, pExtra, pHandlerInfo);
}



/////////////////////////////////////////////////////////////////////////////
UINT CFloater::OnNcHitTest(CPoint point) 
{
	// Identify any window place as client
	DWORD htest = CWnd::OnNcHitTest(point);

	if (htest == HTCAPTION || htest == HTNOWHERE)
		htest = HTCLIENT;

	return htest;
}



/////////////////////////////////////////////////////////////////////////////
void CFloater::OnLButtonDown(UINT nFlags, CPoint point) 
{
	ClientToScreen (&point);

	m_bMoveSplitter = TRUE;
	m_ptInitPos = point;
	m_ptOldPos = point;

	// Lock windows update while dragging
	CWnd* pWnd = CWnd::GetDesktopWindow ();
	pWnd->LockWindowUpdate ();

	// Current alignment is nowhere
	m_fAligned = ALIGN_NOWHERE;

	// Draw draging frame
	DrawDragMark (point);
	SetCapture ();
	
	CWnd::OnLButtonDown(nFlags, point);
}



/////////////////////////////////////////////////////////////////////////////
void CFloater::OnMouseMove(UINT nFlags, CPoint point) 
{
	ClientToScreen (&point);

	if (m_bMoveSplitter)
	{
		DrawDragMark (m_ptOldPos);
		m_ptOldPos = point;
		
		// Draw draging frame in new position
		DrawDragMark (point);

		CPoint ptPos (point);
		
		// Get panel under cursor
		int align = ALIGN_NOWHERE;
		CPanel* pPanel = m_pLayout->GetPanel (ptPos);
		
		// If any panel under cursor calculate aling
		if (pPanel)
			align = GetPanelSide (ptPos, pPanel);
		else
			m_pDockPanel = NULL;
	
		// If current align not equal old hide cursor and show new 
		if (align != m_fAligned)
		{
			DrawDragMark (point);	
			
			m_fAligned = align;

			if (pPanel)
				m_rcPanel = pPanel->GetScreenRect ();

			DrawDragMark (point);
		}
	}
	
	CWnd::OnMouseMove(nFlags, point);
}



/////////////////////////////////////////////////////////////////////////////
void CFloater::OnLButtonUp(UINT nFlags, CPoint point) 
{
	ClientToScreen (&point);

	// Save current position and release mouse
	m_ptEndPos = point;
	ReleaseCapture ();

	CWnd::OnLButtonUp(nFlags, point);
}



/////////////////////////////////////////////////////////////////////////////
void CFloater::OnCaptureChanged(CWnd *pWnd) 
{
	if (m_bMoveSplitter)
	{
		// Clear draging frame
		DrawDragMark (m_ptOldPos);
		m_bMoveSplitter = FALSE;

		// Unlock windows update
		CWnd* pWnd = CWnd::GetDesktopWindow();
		pWnd->UnlockWindowUpdate ();

		// Move window to new position
		RepositionFloater (m_ptEndPos);
		// Change floating state if nesessary
		EnableFloating (m_fAligned == ALIGN_NOWHERE, m_ptEndPos);
	}
	
	CWnd::OnCaptureChanged(pWnd);
}



/////////////////////////////////////////////////////////////////////////////
void CFloater::OnPaint() 
{
	CPaintDC dc(this); // device context for painting
	
	// Draw layout and windows on paint
	if (m_pFloaterLayout)
		m_pFloaterLayout->DrawToWindow ();
	
	// Do not call CWnd::OnPaint() for painting messages
}



/////////////////////////////////////////////////////////////////////////////
void CFloater::OnClose() 
{
	// Hode window on close button
	ShowWindow (SW_HIDE);	
	//CWnd::OnClose();
}



/////////////////////////////////////////////////////////////////////////////
void CFloater::OnLButtonDblClk(UINT nFlags, CPoint point) 
{
	if (m_bFloat)
	{
		// Floater going to docking
		m_fAligned = m_fLastAligned;
		m_pDockPanel = m_pLastDockPanel;
		EnableFloating (FALSE, m_rcWindow.TopLeft ());
	}
	else
	{
		// Floater going to float
		m_fAligned = ALIGN_NOWHERE;
		m_pDockPanel = NULL;
		EnableFloating (TRUE, m_rcWindow.TopLeft ());
	}
	
	CWnd::OnLButtonDblClk(nFlags, point);
}



/////////////////////////////////////////////////////////////////////////////
void CFloater::OnNcCalcSize(BOOL bCalcValidRects, NCCALCSIZE_PARAMS FAR* lpncsp) 
{
	// If docking add place for caption
	if(!m_bFloat)
	{
		// Add caption place to left
		if (m_fAligned == ALIGN_TOP || m_fAligned == ALIGN_BOTTOM)
			lpncsp->rgrc[0].left += m_captionSize;

		// Add caption place to top
		if (m_fAligned == ALIGN_LEFT || m_fAligned == ALIGN_RIGHT)
			lpncsp->rgrc[0].top += m_captionSize;
	}
	
	CWnd::OnNcCalcSize(bCalcValidRects, lpncsp);
}



/////////////////////////////////////////////////////////////////////////////
void CFloater::OnNcPaint() 
{	
	if (!m_bFloat)
	{
		CRect rcWindow;
		GetWindowRect (&rcWindow);
		ScreenToClient (&rcWindow);

		if (m_fAligned == ALIGN_TOP || m_fAligned == ALIGN_BOTTOM)
		{
			rcWindow.left = 0;
			rcWindow.right = m_captionSize;
		}

		if (m_fAligned == ALIGN_LEFT || m_fAligned == ALIGN_RIGHT)
		{
			rcWindow.top = 0;
			rcWindow.bottom = m_captionSize;
		}

		CWindowDC dc (this);

		// Call virtual draw method
		OnDrawCaption (&dc, rcWindow);
	}

	Default ();
	
	// Do not call CWnd::OnNcPaint() for painting messages
}



/////////////////////////////////////////////////////////////////////////////
void CFloater::OnShowWindow(BOOL bShow, UINT nStatus) 
{
	CWnd::OnShowWindow(bShow, nStatus);

	static BOOL bLock = FALSE;

	// Lock for preserv recursive call in Panel::Show function
	if (bLock)
		return; // If locking return
	
	// Lock
	bLock = TRUE;

	if (bShow)
	{
		// If panel in floating state
		if (!m_bFloat)
		{
			// If wrapper panel exists
			if (m_pWrapperPanel) 
				m_pWrapperPanel->Show (TRUE);
		}
	}
	else
	{
		// If panel in floating state
		if (!m_bFloat)
		{
			// If wrapper panel exists
			if (m_pWrapperPanel)
				m_pWrapperPanel->Show (FALSE);
		}
	}

	// Unlock
	bLock = FALSE;
}



/////////////////////////////////////////////////////////////////////////////
LRESULT CFloater::WindowProc(UINT message, WPARAM wParam, LPARAM lParam) 
{
	if (message >= WM_USER)
	{
		// Deligate all user messages to parent window
		GetParent ()->SendMessage (message, wParam, lParam);
	}
	
	return CWnd::WindowProc(message, wParam, lParam);
}



/////////////////////////////////////////////////////////////////////////////
void CFloater::DockFloater (CPanel* pPanel, int align, BOOL bShow)
{
	// bDock now is not used
	// Get Rect of panel floater to docking
	CRect rcPanel = pPanel->GetScreenRect ();

	CPoint ptDockPoint;
	BOOL bFound = FALSE;

	// Find point from specifien align 
	for (int y = rcPanel.top; y < rcPanel.bottom; y++)
	{
		for (int x = rcPanel.left; x < rcPanel.right; x++)
		{
			ptDockPoint.x = x;
			ptDockPoint.y = y;

			if (GetPanelSide (ptDockPoint, pPanel) == align)
			{
				// If point found - break circle
				bFound = TRUE;
				break;
			}
		}

		if (bFound)
			break;
	}

	if (!bFound)
	{
		// If point not found return 
		// May be internal error accure
		ASSERT (FALSE);
		return;
	}

	m_fAligned = align; // Set specified align as current
	
	// Dock current floater panel
	EnableFloating (FALSE, ptDockPoint, bShow);
}
