/*************************************************
*            BK Emulator for Window 3.0          *
*************************************************/

/*
File: Debugger.cpp
Version: 1.0.0
Written by: Yuriy Kalmykov <kalmykov@stoik.com>
    Copyright (c) 2002-2004 Yuriy Kalmykov

    BK Emulator is a program emulated hardware environment for running
code for BK 0010(01) in different configurations. 
           
    This code may be used in compiled form in any way you desire.
This file or it's parts can't be redistributed without the authors
written consent, but can be modified for your private needs.
    Providing that this notice and the authors name and all copyright
notices remains intact.

    Please, an email me to know how you are using it and where. You can
ask me for any information about this below code or any attendant
knowledge.
    
    This file is provided "as is" with no expressed or implied warranty.
The author accepts no liability for any damage or loss of business that
this product may cause.
*/




// Debugger.cpp: implementation of the CDebugger class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "bk.h"
#include "Debugger.h"
#include "Defines.h"
#include "Globals.h"
#include "Board.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

int CDebugger::m_outLevel = 0;

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CDebugger::CDebugger()
{
	m_address = 0100000;
	m_pChip = NULL;
	m_pCtrl = NULL;

	m_bBreakCondition = FALSE;
}

CDebugger::~CDebugger()
{

}



//////////////////////////////////////////////////////////////////////
void CDebugger::OnReset ()
{
}



//////////////////////////////////////////////////////////////////////
void CDebugger::GetByte (int addr, BYTE* pValue)
{
	ASSERT (FALSE);
}



//////////////////////////////////////////////////////////////////////
void CDebugger::GetWord (int addr, WORD* pValue)
{
	ASSERT (addr == DEBUG_PC_BREAK);

	if (addr == DEBUG_PC_BREAK)
		*pValue = m_bBreakCondition;
}



//////////////////////////////////////////////////////////////////////
void CDebugger::SetByte (int addr, BYTE value)
{
	ASSERT (FALSE);
}



//////////////////////////////////////////////////////////////////////
void CDebugger::SetWord (int addr, WORD value)
{
	ASSERT (addr == DEBUG_PC_BREAK);
	BOOL bBreak = FALSE;
	
	if (addr == DEBUG_PC_BREAK)
	{
		if (m_breakpointList.GetCount ())
		{
			POSITION pos = m_breakpointList.GetHeadPosition ();

			while (pos)
			{
				CBreakPoint& breakpoint = m_breakpointList.GetNext (pos);

				if (breakpoint.IsAddress ())
				{
					if (breakpoint.GetAddress () == value)
					{
						bBreak = TRUE;
						break;
					}
				}
			}
		}
	}

	m_bBreakCondition = bBreak;
}



//////////////////////////////////////////////////////////////////////
int	CDebugger::CalcInstructionLength (WORD instruction)
{	
	// No fields
	switch (instruction)
	{
	case PI_HALT:
	case PI_HALT10:
	case PI_HALT11:
	case PI_HALT12:
	case PI_HALT13:
	case PI_HALT14:
	case PI_HALT15:
	case PI_HALT16:
	case PI_HALT17:
	case PI_WAIT:
	case PI_RTI:
	case PI_BPT:
	case PI_IOT:
	case PI_RESET:
	case PI_RTT:
	case PI_NOP:
	case PI_CLC:
	case PI_CLV:
	case PI_CLVC:
	case PI_CLZ:
	case PI_CLZC:
	case PI_CLZV:
	case PI_CLZVC:
	case PI_CLN:
	case PI_CLNC:
	case PI_CLNV:
	case PI_CLNVC:
	case PI_CLNZ:
	case PI_CLNZC:
	case PI_CLNZV:
	case PI_CCC:
	case PI_NOP260:
	case PI_SEC:
	case PI_SEV:
	case PI_SEVC:
	case PI_SEZ:
	case PI_SEZC:
	case PI_SEZV:
	case PI_SEZVC:
	case PI_SEN:
	case PI_SENC:
	case PI_SENV:
	case PI_SENVC:
	case PI_SENZ:
	case PI_SENZC:
	case PI_SENZV:
	case PI_SCC:
		return 2;
	}

	// Branchs & interrupts
	switch (instruction & ~(WORD)0377)
	{
	case PI_BR:
	case PI_BNE:
	case PI_BEQ:
	case PI_BGE:
	case PI_BLT:
	case PI_BGT:
	case PI_BLE:
	case PI_BPL:
	case PI_BMI:
	case PI_BHI:
	case PI_BLOS:
	case PI_BVC:
	case PI_BVS:
	case PI_BHIS:
	case PI_BLO:
	case PI_EMT:
	case PI_TRAP:
		return 2;
	}

	// One field
	if ((instruction & ~(WORD)7) == PI_RTS)
		return 2;
	
	// Two fields
	switch (instruction & ~(WORD)077)
	{
	case PI_MARK:
		return 1;
	case PI_JMP:
	case PI_SWAB:
	case PI_SXT:
	case PI_MTPS:
	case PI_MFPS:
		return 2 + CalcArgLength (instruction, DST);
	}
	
	switch (instruction & ~(WORD)0100077)
	{
	case PI_CLR:
	case PI_COM:
	case PI_INC:
	case PI_DEC:
	case PI_NEG:
	case PI_ADC:
	case PI_SBC:
	case PI_TST:
	case PI_ROR:
	case PI_ROL:
	case PI_ASR:
	case PI_ASL:
		return 2 + CalcArgLength (instruction, DST);
	}
	
	// Three fields
	switch(instruction & ~(WORD)0777)
	{
	case PI_JSR:
	case PI_XOR:
		return 2 + CalcArgLength (instruction, DST);
	case PI_SOB:
		return 2;
	}

	// Four fields
	switch(instruction & ~(WORD)0107777)
	{
	case PI_MOV:
	case PI_CMP:
	case PI_BIT:
	case PI_BIC:
	case PI_BIS:
		return 2 + CalcArgLength (instruction, SRC) + CalcArgLength (instruction, DST);
	}

	switch (instruction & ~(WORD)0007777)
	{
	case PI_ADD:
	case PI_SUB:
		return 2 + CalcArgLength (instruction, SRC) + CalcArgLength (instruction, DST);
	}

	return 2;
}



//////////////////////////////////////////////////////////////////////
int CDebugger::CalcArgLength (WORD instruction, int pos)
{
	int reg = (GetDigit (instruction, pos) << 1) + R0;
	int meth = GetDigit (instruction, pos + 1);
	
	int arg = 0;

	switch (meth)
	{
	case 0: // R0,		PC 
	case 1: // (R0),	(PC)
		break;
		
	case 2: // (R0)+,	#012345
		if (reg == PC)
			arg = 2;
		break;

	case 3: // @(R0)+,	@#012345
		if (reg == PC)
			arg = 2;
		break;
	
	case 4: // -(R0),	-(PC)
	case 5: // @-(R0),	@-(PC)
		break;

	case 6: // 345(R0),	345	
	case 7: // @345(R0),@345
		arg = 2;
		break;
	}

	return arg;
}



//////////////////////////////////////////////////////////////////////
WORD CDebugger::GetLineAddress (int iNum)
{
	if (iNum < 0)
		return 0;

	WORD cur_addr = m_address;
	
	while (iNum--)
		cur_addr += CalcInstructionLength (m_pChip->GetWordIndirect (cur_addr));
	
	return cur_addr;
}



//////////////////////////////////////////////////////////////////////
int CDebugger::DebugInstruction (WORD addr, CString& strInstr, CString& strArg, WORD* codes)
{
	ASSERT (m_pChip);

	WORD instr = m_pChip->GetWordIndirect (addr);

	int length = 1;
	codes[0] = instr;

	strArg = "";
	CString strReg;
	CString strSrc;
	CString strDst;
	CString strByte = "";
	// No fields

	switch (instr)
	{
	case PI_HALT:
		strInstr.LoadString (IDS_PI_HALT);
		return 1;
	case PI_WAIT:
		strInstr.LoadString (IDS_PI_WAIT);
		return 1;
	case PI_RTI:
		strInstr.LoadString (IDS_PI_RTI);
		return 1;
	case PI_BPT:
		strInstr.LoadString (IDS_PI_BPT);
		return 1;
	case PI_IOT:
		strInstr.LoadString (IDS_PI_IOT);
		return 1;
	case PI_RESET:
		strInstr.LoadString (IDS_PI_RESET);
		return 1;
	case PI_RTT:
		strInstr.LoadString (IDS_PI_RTT);
		return 1;
	case PI_NOP:
		strInstr.LoadString (IDS_PI_NOP);
		return 1;
	case PI_CLC:
		strInstr.LoadString (IDS_PI_CLC);
		return 1;
	case PI_CLV:
		strInstr.LoadString (IDS_PI_CLV);
		return 1;
	case PI_CLZ:
		strInstr.LoadString (IDS_PI_CLZ);
		return 1;
	case PI_CLN:
		strInstr.LoadString (IDS_PI_CLN);
		return 1;
	case PI_CCC:
		strInstr.LoadString (IDS_PI_CCC);
		return 1;
	case PI_NOP260:
		strInstr.LoadString (IDS_PI_NOP260);
		return 1;
	case PI_SEC:
		strInstr.LoadString (IDS_PI_SEC);
		return 1;
	case PI_SEV:
		strInstr.LoadString (IDS_PI_SEV);
		return 1;
	case PI_SEZ:
		strInstr.LoadString (IDS_PI_SEZ);
		return 1;
	case PI_SEN:
		strInstr.LoadString (IDS_PI_SEN);
		return 1;
	case PI_SCC:
		strInstr.LoadString (IDS_PI_SCC);
		return 1;
	}

	// One field
	if ((instr & ~(WORD)7) == PI_RTS)
	{
		strInstr.LoadString (IDS_PI_RTS);

		strReg.LoadString (IDS_R0 + GetDigit (instr, 0));
		strArg = strReg;
		
		return 1;
	}

	// Two fields

	length += ConvertDstToString (instr, addr + 2, strDst, codes[1]);

	switch (instr & ~(WORD)077)
	{
	case PI_JMP:
		strInstr.LoadString (IDS_PI_JMP);
		strArg = strDst;
		return length;
	case PI_SWAB:
		strInstr.LoadString (IDS_PI_SWAB);
		strArg = strDst;
		return length;
	case PI_MARK:
		strInstr.LoadString (IDS_PI_MARK);
		strArg = strDst;
		return length;
	case PI_SXT:
		strInstr.LoadString (IDS_PI_SXT);
		strArg = strDst;
		return length;
	case PI_MTPS:
		strInstr.LoadString (IDS_PI_MTPS);
		strArg = strDst;
		return length;
	case PI_MFPS:
		strInstr.LoadString (IDS_PI_MFPS);
		strArg = strDst;
		return length;
	}
	
	if (instr & 0100000)
		strByte = 'B';

	switch (instr & ~(WORD)0100077)
	{
	case PI_CLR:
		strInstr.LoadString (IDS_PI_CLR);
		strInstr += strByte;
		strArg = strDst;
		return length;
	case PI_COM:
		strInstr.LoadString (IDS_PI_COM);
		strInstr += strByte;
		strArg = strDst;
		return length;
	case PI_INC:
		strInstr.LoadString (IDS_PI_INC);
		strInstr += strByte;
		strArg = strDst;
		return length;
	case PI_DEC:
		strInstr.LoadString (IDS_PI_DEC);
		strInstr += strByte;
		strArg = strDst;
		return length;
	case PI_NEG:
		strInstr.LoadString (IDS_PI_NEG);
		strInstr += strByte;
		strArg = strDst;
		return length;
	case PI_ADC:
		strInstr.LoadString (IDS_PI_ADC);
		strInstr += strByte;
		strArg = strDst;
		return length;
	case PI_SBC:
		strInstr.LoadString (IDS_PI_SBC);
		strInstr += strByte;
		strArg = strDst;
		return length;
	case PI_TST:
		strInstr.LoadString (IDS_PI_TST);
		strInstr += strByte;
		strArg = strDst;
		return length;
	case PI_ROR:
		strInstr.LoadString (IDS_PI_ROR);
		strInstr += strByte;
		strArg = strDst;
		return length;
	case PI_ROL:
		strInstr.LoadString (IDS_PI_ROL);
		strInstr += strByte;
		strArg = strDst;
		return length;
	case PI_ASR:
		strInstr.LoadString (IDS_PI_ASR);
		strInstr += strByte;
		strArg = strDst;
		return length;
	case PI_ASL:
		strInstr.LoadString (IDS_PI_ASL);
		strInstr += strByte;
		strArg = strDst;
		return length;
	}

	length = 1;
	strDst.Format ("%o", addr + ((short)(char)LOBYTE (instr) * 2) + 2);

	// Branchs & interrupts
	switch (instr & ~(WORD)0377)
	{
	case PI_BR:
		strInstr.LoadString (IDS_PI_BR);
		strArg = strDst;
		return 1;
	case PI_BNE:
		strInstr.LoadString (IDS_PI_BNE);
		strArg = strDst;
		return 1;
	case PI_BEQ:
		strInstr.LoadString (IDS_PI_BEQ);
		strArg = strDst;
		return 1;
	case PI_BGE:
		strInstr.LoadString (IDS_PI_BGE);
		strArg = strDst;
		return 1;
	case PI_BLT:
		strInstr.LoadString (IDS_PI_BLT);
		strArg = strDst;
		return 1;
	case PI_BGT:
		strInstr.LoadString (IDS_PI_BGT);
		strArg = strDst;
		return 1;
	case PI_BLE:
		strInstr.LoadString (IDS_PI_BLE);
		strArg = strDst;
		return 1;
	case PI_BPL:
		strInstr.LoadString (IDS_PI_BPL);
		strArg = strDst;
		return 1;
	case PI_BMI:
		strInstr.LoadString (IDS_PI_BMI);
		strArg = strDst;
		return 1;
	case PI_BHI:
		strInstr.LoadString (IDS_PI_BHI);
		strArg = strDst;
		return 1;
	case PI_BLOS:
		strInstr.LoadString (IDS_PI_BLOS);
		strArg = strDst;
		return 1;
	case PI_BVC:
		strInstr.LoadString (IDS_PI_BVC);
		strArg = strDst;
		return 1;
	case PI_BVS:
		strInstr.LoadString (IDS_PI_BVS);
		strArg = strDst;
		return 1;
	case PI_BHIS:
		strInstr.LoadString (IDS_PI_BHIS);
		strArg = strDst;
		return 1;
	case PI_BLO:
		strInstr.LoadString (IDS_PI_BLO);
		strArg = strDst;
		return 1;
	}

	strDst.Format ("%o", LOBYTE (instr));

	switch (instr & ~(WORD)0377)
	{
	case PI_EMT:
		strInstr.LoadString (IDS_PI_EMT);
		strArg = strDst;
		return 1;
	case PI_TRAP:
		strInstr.LoadString (IDS_PI_TRAP);
		strArg = strDst;
		return 1;
	}

	// Three fields
	switch(instr & ~(WORD)0777)
	{
	case PI_JSR:
		strInstr.LoadString (IDS_PI_JSR);
		strReg.LoadString (IDS_R0 + GetDigit (instr, 2));
		length += ConvertDstToString (instr, addr + 2, strDst, codes[1]);
		strArg = strReg + ',' + strDst;
		return length;
	case PI_XOR:
		strInstr.LoadString (IDS_PI_XOR);
		strReg.LoadString (IDS_R0 + GetDigit (instr, 2));
		length += ConvertDstToString (instr, addr + 2, strDst, codes[1]);
		strArg = strReg + ',' + strDst;
		return length;
	case PI_SOB:
		strInstr.LoadString (IDS_PI_SOB);
		strReg.LoadString (IDS_R0 + GetDigit (instr, 2));
		strDst.Format ("%o", addr - (GetDigit (instr, 1) * 8 + GetDigit (instr, 0))  * 2 + 2);
		strArg = strReg + ',' + strDst;
		return 1;
	}

	// Four fields

	if (instr & 0100000)
		strByte = 'B';
	else
		strByte = "";

	length += ConvertSrcToString (instr, addr + 2, strSrc, codes[1]);
	length += ConvertDstToString (instr, addr + 2 + (length - 1) * 2, strDst, codes[length]);

	switch(instr & ~(WORD)0107777)
	{
	case PI_MOV:
		strInstr.LoadString (IDS_PI_MOV);
		strInstr += strByte;
		strArg = strSrc + ',' + strDst;
		return length;
	case PI_CMP:
		strInstr.LoadString (IDS_PI_CMP);
		strInstr += strByte;
		strArg = strSrc + ',' + strDst;
		return length;
	case PI_BIT:
		strInstr.LoadString (IDS_PI_BIT);
		strInstr += strByte;
		strArg = strSrc + ',' + strDst;
		return length;
	case PI_BIC:
		strInstr.LoadString (IDS_PI_BIC);
		strInstr += strByte;
		strArg = strSrc + ',' + strDst;
		return length;
	case PI_BIS:
		strInstr.LoadString (IDS_PI_BIS);
		strInstr += strByte;
		strArg = strSrc + ',' + strDst;
		return length;
	}

	switch (instr & ~(WORD)0007777)
	{
	case PI_ADD:
		strInstr.LoadString (IDS_PI_ADD);
		strArg = strSrc + ',' + strDst;
		return length;
	case PI_SUB:
		strInstr.LoadString (IDS_PI_SUB);
		strArg = strSrc + ',' + strDst;
		return length;
	}
	
	strInstr.LoadString (IDS_PI_WORD);
	strArg.Format ("%o", instr);

	return 1;
}



//////////////////////////////////////////////////////////////////////
BOOL CDebugger::ConvertSrcToString (WORD instr, WORD addr, CString& strSrc, WORD& code)
{
	ASSERT (m_pChip);

	int reg = GetDigit (instr, 2);
	int param = GetDigit (instr, 3);

	CString strReg;

	strReg.LoadString (IDS_R0 + reg);

	reg = (reg << 1) + R0;

	if (reg >= R0 && reg <= SP)
	{
		strSrc.LoadString (IDS_ADDR_0 + param);

		if (param == 6 || param == 7)
		{
			WORD word = m_pChip->GetWordIndirect (addr);

			strSrc.Format (CString (strSrc), word, strReg);
			code = word;
			return TRUE;
		}
		else
			strSrc.Format (strSrc, strReg);
	}
	else
	{
		strSrc.LoadString (IDS_ADDR_PC_0 + param);

		if (param == 2 || param == 3)
		{
			WORD word = m_pChip->GetWordIndirect (addr);

			strSrc.Format (CString (strSrc), word);
			code = word;
			return TRUE;
		}
		else if (param == 6 || param == 7)
		{
			WORD word = m_pChip->GetWordIndirect (addr);

			strSrc.Format (CString (strSrc), (WORD)(addr + word + 2));
			code = word;
			return TRUE;
		}
		else
			strSrc.Format (strSrc, strReg);
	}


	return FALSE;
}



//////////////////////////////////////////////////////////////////////
BOOL CDebugger::ConvertDstToString (WORD instr, WORD addr, CString& strDst, WORD& code)
{
	ASSERT (m_pChip);

	int reg = GetDigit (instr, 0);
	int param = GetDigit (instr, 1);

	CString strReg;

	strReg.LoadString (IDS_R0 + reg);

	reg = (reg << 1) + R0;

	if (reg >= R0 && reg <= SP)
	{
		strDst.LoadString (IDS_ADDR_0 + param);

		if (param == 6 || param == 7)
		{
			strDst.Format (CString(strDst), m_pChip->GetWordIndirect (addr), strReg);
			code = m_pChip->GetWordIndirect (addr);
			return TRUE;
		}
		else
			strDst.Format (strDst, strReg);
	}
	else
	{
		strDst.LoadString (IDS_ADDR_PC_0 + param);

		if (param == 2 || param == 3)
		{
			strDst.Format (CString(strDst), m_pChip->GetWordIndirect (addr));
			code = m_pChip->GetWordIndirect (addr);
			return TRUE;
		}
		else if (param == 6 || param == 7)
		{
			strDst.Format (CString(strDst), (WORD)(addr + m_pChip->GetWordIndirect (addr) + 2));
			code = m_pChip->GetWordIndirect (addr);
			return TRUE;
		}
		else
			strDst.Format (strDst, strReg);
	}

	return FALSE;
}



//////////////////////////////////////////////////////////////////////
void CDebugger::DrawDeguggerLine (int nNum, CDC* pDC, CRect& rcLine, CRect* pRcSubs)
{
	if (!m_pChip) // No Chip nothing to draw
		return;

	// Convert instruction code to text
	WORD instruction[3];
	CString strInstruction;
	CString strArgument;
	WORD line_addr = GetLineAddress (nNum);

	int len = DebugInstruction (line_addr, strInstruction, strArgument, instruction); 

	// Draw Marker
	DrawMarker (line_addr, pDC, pRcSubs[MARK_COLUMN]);

	// Draw address
	pDC->DrawText (IntToString (line_addr, 8), &pRcSubs[ADDR_COLUMN], DT_LEFT|DT_VCENTER|DT_END_ELLIPSIS);

	// Draw Instruction
	pDC->DrawText (strInstruction, &pRcSubs[INSTR_COLUMN], DT_LEFT|DT_VCENTER|DT_END_ELLIPSIS);

	// Draw Arguments
	pDC->DrawText (strArgument, &pRcSubs[ARG_COLUMN], DT_LEFT|DT_VCENTER|DT_END_ELLIPSIS);
}



//////////////////////////////////////////////////////////////////////
void CDebugger::DrawMarker (WORD address, CDC* pDC, CRect& rc)
{
	HICON hIcon = NULL;

	if (IsBpeakpointAtAddress (address))
	{
		hIcon = ::LoadIcon (AfxGetInstanceHandle (), MAKEINTRESOURCE (IDI_DBG_BPT));
		::DrawIconEx (pDC->m_hDC, rc.left, rc.top, hIcon, 16, 16, 0, NULL, DI_NORMAL);
	}

	if (address == m_pChip->GetWordIndirect (PC))
	{
		hIcon = ::LoadIcon (AfxGetInstanceHandle (), MAKEINTRESOURCE (IDI_DBG_CUR));
		::DrawIconEx (pDC->m_hDC, rc.left, rc.top, hIcon, 16, 16, 0, NULL, DI_NORMAL);
	}
}



//////////////////////////////////////////////////////////////////////
void CDebugger::StepForward ()
{
	m_address += CalcInstructionLength (m_pChip->GetWordIndirect (m_address));
}



//////////////////////////////////////////////////////////////////////
void CDebugger::StepBackward ()
{
	m_address -= sizeof (WORD);
}



//////////////////////////////////////////////////////////////////////
BOOL CDebugger::IsInstructionOut (WORD instruction)
{
	// No fields
	switch (instruction)
	{
	case PI_BPT:
	case PI_IOT:
		m_outLevel++;
		return FALSE;
	}

	// Branchs & interrupts
	switch (instruction & ~(WORD)0377)
	{
	case PI_EMT:
	case PI_TRAP:
		m_outLevel++;
		return FALSE;
	}
		
	// Three fields
	switch (instruction & ~(WORD)0777)
	{
	case PI_JSR:
		m_outLevel++;
		return FALSE;
	}

	BOOL bOutInstruction = FALSE;

	// No fields
	switch (instruction)
	{
		case PI_RTI:
		case PI_RTT:
			m_outLevel--;
			bOutInstruction = TRUE;
			break;
	}

	// One field
	if ((instruction & ~(WORD)7) == PI_RTS)
	{
		m_outLevel--;
		bOutInstruction = TRUE;
	}

	if (bOutInstruction && m_outLevel == -1)
		return TRUE;

	return FALSE;
}



//////////////////////////////////////////////////////////////////////
BOOL CDebugger::IsInstructionOver (WORD instruction)
{
	// No fields
	switch (instruction)
	{
	case PI_BPT:
	case PI_IOT:
		return TRUE;
	}

	// Branchs & interrupts
	switch (instruction & ~(WORD)0377)
	{
	case PI_EMT:
	case PI_TRAP:
		return TRUE;
	}
		
	// Three fields
	switch (instruction & ~(WORD)0777)
	{
	case PI_JSR:
	case PI_SOB:
		return TRUE;
	}

	return FALSE;
}



//////////////////////////////////////////////////////////////////////
WORD CDebugger::GetCursorAddress ()
{
	ASSERT (m_pCtrl);
	int nLine = m_pCtrl->GetSelectionMark ();

	return GetLineAddress (nLine);
}



//////////////////////////////////////////////////////////////////////
WORD CDebugger::GetBottomAddress ()
{
	ASSERT (m_pCtrl);
	int nLast = m_pCtrl->GetItemCount ();

	return GetLineAddress (nLast);
}



//////////////////////////////////////////////////////////////////////
WORD CDebugger::GetPSW ()
{
	ASSERT (m_pChip);

	if (!m_pChip)
		return 0;

	return m_pChip->GetPSW ();
}



//////////////////////////////////////////////////////////////////////
WORD CDebugger::GetWordValue (int addr)
{
	ASSERT (!(addr & 1));
	addr &= ~1;
	ASSERT (m_pChip);

	if (!m_pChip)
		return 0;

	return m_pChip->GetWordIndirect (addr);
}



//////////////////////////////////////////////////////////////////////
void CDebugger::GetMemoryDump (WORD* pBuff, WORD addr, int length)
{
	ASSERT (!(addr & 1));
	addr &= ~1;
	ASSERT (m_pChip);

	if (m_pChip)
	{
		int length1 = length;
		int length2 = 0;

		if (65536 - addr < length * 2)
		{
			length1 = (65536 - addr) / 2;
			length2 = (length - length1) / 2;
		}

		for (int n = 0; n < length1; n++)
			pBuff[n] = m_pChip->GetWordIndirect (addr + n * 2);

		for (n = 0; n < length2; n++)
			pBuff[n + length1] = m_pChip->GetWordIndirect (n * 2);
	}
}



//////////////////////////////////////////////////////////////////////
void CDebugger::ApplyBreakpoints ()
{
	if (m_pChip)
	{
		if (m_breakpointList.GetCount ())
			m_pChip->AttachDebugger (this);
		else
			m_pChip->AttachDebugger (NULL);
	}	
}



//////////////////////////////////////////////////////////////////////
BOOL CDebugger::IsBpeakpointExists (CBreakPoint& breakpoint)
{
	POSITION pos = m_breakpointList.GetHeadPosition ();

	while (pos)
	{
		CBreakPoint& curr = m_breakpointList.GetNext (pos);

		if ((curr.IsAddress () == breakpoint.IsAddress ()) 
			&& (curr.GetAddress () == breakpoint.GetAddress ()))
			return TRUE;
	}

	return FALSE;
}



//////////////////////////////////////////////////////////////////////
BOOL CDebugger::IsBpeakpointAtAddress (int addr)
{
	POSITION pos = m_breakpointList.GetHeadPosition ();

	while (pos)
	{
		CBreakPoint& curr = m_breakpointList.GetNext (pos);

		if (curr.IsAddress () && curr.GetAddress () == addr)
			return TRUE;
	}

	return FALSE;
}



//////////////////////////////////////////////////////////////////////
BOOL CDebugger::SetSimpleBreakpoint (WORD addr)
{
	CBreakPoint breakpoint (addr);

	if (IsBpeakpointExists (breakpoint))
		return FALSE;

	m_breakpointList.AddTail (breakpoint);
	ApplyBreakpoints ();

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CDebugger::SetSimpleBreakpoint ()
{
	return SetSimpleBreakpoint (GetCursorAddress ());
}



//////////////////////////////////////////////////////////////////////
BOOL CDebugger::RemoveBreakpoint (WORD addr)
{
	POSITION pos = m_breakpointList.GetHeadPosition ();

	while (pos)
	{
		POSITION oldPos = pos;
		CBreakPoint& curr = m_breakpointList.GetNext (pos);

		if (curr.IsAddress () && curr.GetAddress () == addr)
		{
			m_breakpointList.RemoveAt (oldPos);
			return TRUE;
		}
	}

	return FALSE;
}



//////////////////////////////////////////////////////////////////////
BOOL CDebugger::RemoveBreakpoint ()
{
	return RemoveBreakpoint (GetCursorAddress ());
}