/*************************************************
*            BK Emulator for Window 3.0          *
*************************************************/

/*
File: CPU.h
Version: 1.0.0
Written by: Yuriy Kalmykov <kalmykov@stoik.com>
    Copyright (c) 2002-2004 Yuriy Kalmykov

    BK Emulator is a program emulated hardware environment for running
code for BK 0010(01) in different configurations. 
           
    This code may be used in compiled form in any way you desire.
This file or it's parts can't be redistributed without the authors
written consent, but can be modified for your private needs.
    Providing that this notice and the authors name and all copyright
notices remains intact.

    Please, an email me to know how you are using it and where. You can
ask me for any information about this below code or any attendant
knowledge.
    
    This file is provided "as is" with no expressed or implied warranty.
The author accepts no liability for any damage or loss of business that
this product may cause.
*/




// CPU.h: interface for the CCPU class.
//
//////////////////////////////////////////////////////////////////////

#if !defined(AFX_CPU_H__13517A42_EBC7_49E6_9498_CC54D76F44EB__INCLUDED_)
#define AFX_CPU_H__13517A42_EBC7_49E6_9498_CC54D76F44EB__INCLUDED_

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

//////////////////////////////////////////////////////////////////////
#include "Globals.h"
#include "Defines.h"

//////////////////////////////////////////////////////////////////////
class CMotherBoard;

#pragma inline_recursion(on)
//////////////////////////////////////////////////////////////////////
class CCPU  
{
	CMotherBoard*	m_pChip;
	WORD			m_instruction;
	WORD			m_psw;
	LONGLONG		m_internalTick; // In tenth of mks

	BOOL			m_bStopInterrupt;
	BOOL			m_bKeyboardInterrupt;

	int				GetByteArg (int pos);
	int				GetWordArg (int pos);
	int				GetDstWordArgAsBranch ();
	
	void			SetC (BOOL bFlag);
	BOOL			GetC ();
	void			SetV (BOOL bFlag);
	BOOL			GetV ();
	void			SetN (BOOL bFlag);
	BOOL			GetN ();
	void			SetZ (BOOL bFlag);
	BOOL			GetZ ();

	BOOL			CheckForNegative (BYTE byte);
	BOOL			CheckForNegative (WORD word);
	BOOL			CheckForZero (BYTE byte);
	BOOL			CheckForZero (WORD word);
	BOOL			CheckAddForOverflow (BYTE a, BYTE b);
	BOOL			CheckAddForOverflow (WORD a, WORD b);
	BOOL			CheckSubForOverflow (BYTE a, BYTE b);
	BOOL			CheckSubForOverflow (WORD a, WORD b);
	BOOL			CheckAddForCarry (BYTE a, BYTE b);
	BOOL			CheckAddForCarry (WORD a, WORD b);
	BOOL			CheckSubForCarry (BYTE a, BYTE b);
	BOOL			CheckSubForCarry (WORD a, WORD b);
	
	// No fields
	void			ExecuteHALT ();
	void			ExecuteWAIT ();
	void			ExecuteRTI ();
	void			ExecuteBPT ();
	void			ExecuteIOT ();
	void			ExecuteRESET ();
	void			ExecuteRTT ();
	void			ExecuteNOP ();
	void			ExecuteCLC ();
	void			ExecuteCLV ();
	void			ExecuteCLVC ();
	void			ExecuteCLZ ();
	void			ExecuteCLZC ();
	void			ExecuteCLZV ();
	void			ExecuteCLZVC ();
	void			ExecuteCLN ();
	void			ExecuteCLNC ();
	void			ExecuteCLNV ();
	void			ExecuteCLNVC ();
	void			ExecuteCLNZ ();
	void			ExecuteCLNZC ();
	void			ExecuteCLNZV ();
	void			ExecuteCCC ();
	void			ExecuteSEC ();
	void			ExecuteSEV ();
	void			ExecuteSEVC ();
	void			ExecuteSEZ ();
	void			ExecuteSEZC ();
	void			ExecuteSEZV ();
	void			ExecuteSEZVC ();
	void			ExecuteSEN ();
	void			ExecuteSENC ();
	void			ExecuteSENV ();
	void			ExecuteSENVC ();
	void			ExecuteSENZ ();
	void			ExecuteSENZC ();
	void			ExecuteSENZV ();
	void			ExecuteSCC ();

	// One fiels
	void			ExecuteRTS ();

	// Two fields
	void			ExecuteJMP ();
	void			ExecuteSWAB ();
	void			ExecuteCLR ();
	void			ExecuteCOM ();
	void			ExecuteINC ();
	void			ExecuteDEC ();
	void			ExecuteNEG ();
	void			ExecuteADC ();
	void			ExecuteSBC ();
	void			ExecuteTST ();
	void			ExecuteROR ();
	void			ExecuteROL ();
	void			ExecuteASR ();
	void			ExecuteASL ();
	void			ExecuteMARK ();
	void			ExecuteSXT ();
	void			ExecuteMTPS ();
	void			ExecuteMFPS ();
	
	// Branchs & interrupts
	void			ExecuteBR ();
	void			ExecuteBNE ();
	void			ExecuteBEQ ();
	void			ExecuteBGE ();
	void			ExecuteBLT ();
	void			ExecuteBGT ();
	void			ExecuteBLE ();
	void			ExecuteBPL ();
	void			ExecuteBMI ();
	void			ExecuteBHI ();
	void			ExecuteBLOS ();
	void			ExecuteBVC ();
	void			ExecuteBVS ();
	void			ExecuteBHIS ();
	void			ExecuteBLO ();

	void			ExecuteEMT ();
	void			ExecuteTRAP ();

	// Three fields
	void			ExecuteJSR ();
	void			ExecuteXOR ();
	void			ExecuteSOB ();

	// Four fields
	void			ExecuteMOV ();
	void			ExecuteCMP ();
	void			ExecuteBIT ();
	void			ExecuteBIC ();
	void			ExecuteBIS ();

	void			ExecuteADD ();
	void			ExecuteSUB ();

public:
	CCPU();
	virtual ~CCPU();

	void			AttachChip (CMotherBoard* pChip) {m_pChip = pChip;}
	void			ResetInternalTicks (int mod) {m_internalTick %= mod;}
	LONGLONG		GetInternalTicks () {return m_internalTick;}
	void			TranslateInstruction ();

	void			MakeInterrupt (WORD interrupt);
	void			StopInterrupt ();
	void			KeyboardInterript ();

	void			SetPSW (WORD value) {m_psw = value;}
	WORD			GetPSW () {return m_psw;}
	WORD			GetCurrentInstruction () {return m_instruction;}

	BOOL			IsTrace ();
};



//////////////////////////////////////////////////////////////////////
inline BOOL CCPU::CheckForNegative (BYTE byte)
{
	return (byte & 0200) != 0;
}



//////////////////////////////////////////////////////////////////////
inline BOOL CCPU::CheckForNegative (WORD word)
{
	return (word & 0100000) != 0;
}



//////////////////////////////////////////////////////////////////////
inline BOOL CCPU::CheckForZero (BYTE byte)
{
	return byte == 0;
}



//////////////////////////////////////////////////////////////////////
inline BOOL CCPU::CheckForZero (WORD word)
{
	return word == 0;
}



//////////////////////////////////////////////////////////////////////
inline BOOL CCPU::CheckAddForOverflow (BYTE a, BYTE b)
{
	//WORD sum = a < 0200 ? (WORD)a + (WORD)b + 0200 : (WORD)a + (WORD)b - 0200;
	//return HIBYTE (sum) != 0;

	BOOL bOverflow = FALSE;

	_asm
	{
		pushf
		push cx
		mov cl,byte ptr [a]
		add cl,byte ptr [b]
		jno end
		mov dword ptr [bOverflow],1
	end:							
		pop cx
		popf
	}

	return bOverflow;
}



//////////////////////////////////////////////////////////////////////
inline BOOL CCPU::CheckAddForOverflow (WORD a, WORD b)
{
	//DWORD sum =  a < 0100000 ? (DWORD)a + (DWORD)b + 0100000 : (DWORD)a + (DWORD)b - 0100000;
	//return HIWORD (sum) != 0;

	BOOL bOverflow = FALSE;

	_asm
	{
		pushf
		push cx
		mov cx,word ptr [a]
		add cx,word ptr [b]
		jno end
		mov dword ptr [bOverflow],1
	end:							
		pop cx
		popf
	}

	return bOverflow;
}



//////////////////////////////////////////////////////////////////////
inline BOOL CCPU::CheckSubForOverflow (BYTE a, BYTE b)
{
	//WORD sum = a < 0200 ? (WORD)a - (WORD)b + 0200 : (WORD)a - (WORD)b - 0200;
	//return HIBYTE (sum) != 0;

	BOOL bOverflow = FALSE;

	_asm
	{
		pushf
		push cx
		mov cl,byte ptr [a]
		sub cl,byte ptr [b]
		jno end
		mov dword ptr [bOverflow],1
	end:							
		pop cx
		popf
	}

	return bOverflow;
}



//////////////////////////////////////////////////////////////////////
inline BOOL CCPU::CheckSubForOverflow (WORD a, WORD b)
{
	//DWORD sum =  a < 0100000 ? (DWORD)a - (DWORD)b + 0100000 : (DWORD)a - (DWORD)b - 0100000;
	//return HIWORD (sum) != 0;

	BOOL bOverflow = FALSE;

	_asm
	{
		pushf
		push cx
		mov cx,word ptr [a]
		sub cx,word ptr [b]
		jno end
		mov dword ptr [bOverflow],1
	end:							
		pop cx
		popf
	}

	return bOverflow;
}


	
//////////////////////////////////////////////////////////////////////
inline BOOL CCPU::CheckAddForCarry (BYTE a, BYTE b)
{
	WORD sum = (WORD)a + (WORD)b;
	return HIBYTE (sum) != 0;
}



//////////////////////////////////////////////////////////////////////
inline BOOL CCPU::CheckAddForCarry (WORD a, WORD b)
{
	DWORD sum = (DWORD)a + (DWORD)b;
	return HIWORD (sum) != 0;
}



//////////////////////////////////////////////////////////////////////
inline BOOL CCPU::CheckSubForCarry (BYTE a, BYTE b)
{
	WORD sum = (WORD)a - (WORD)b;
	return HIBYTE (sum) != 0;
}



//////////////////////////////////////////////////////////////////////
inline BOOL CCPU::CheckSubForCarry (WORD a, WORD b)
{
	DWORD sum = (DWORD)a - (DWORD)b;
	return HIWORD (sum) != 0;
}



//////////////////////////////////////////////////////////////////////
inline void CCPU::SetC (BOOL bFlag)
{
	if (bFlag)
		m_psw |= PSW_C;
	else
		m_psw &= ~PSW_C;
}



//////////////////////////////////////////////////////////////////////
inline BOOL CCPU::GetC ()
{
	return (m_psw & PSW_C) != 0;
}



//////////////////////////////////////////////////////////////////////
inline void CCPU::SetV (BOOL bFlag)
{
	if (bFlag)
		m_psw |= PSW_V;
	else
		m_psw &= ~PSW_V;
}



//////////////////////////////////////////////////////////////////////
inline BOOL CCPU::GetV ()
{
	return (m_psw & PSW_V) != 0;
}



//////////////////////////////////////////////////////////////////////
inline void CCPU::SetZ (BOOL bFlag)
{
	if (bFlag)
		m_psw |= PSW_Z;
	else
		m_psw &= ~PSW_Z;
}



//////////////////////////////////////////////////////////////////////
inline BOOL CCPU::GetZ ()
{
	return (m_psw & PSW_Z) != 0;
}



//////////////////////////////////////////////////////////////////////
inline void CCPU::SetN (BOOL bFlag)
{
	if (bFlag)
		m_psw |= PSW_N;
	else
		m_psw &= ~PSW_N;
}



//////////////////////////////////////////////////////////////////////
inline BOOL CCPU::GetN ()
{
	return (m_psw & PSW_N) != 0;
}

#pragma inline_recursion(off)

#endif // !defined(AFX_CPU_H__13517A42_EBC7_49E6_9498_CC54D76F44EB__INCLUDED_)
