/*************************************************
*            BK Emulator for Window 3.0          *
*************************************************/

/*
File: CPU.cpp
Version: 1.0.0
Written by: Yuriy Kalmykov <kalmykov@stoik.com>
    Copyright (c) 2002-2004 Yuriy Kalmykov

    BK Emulator is a program emulated hardware environment for running
code for BK 0010(01) in different configurations. 
           
    This code may be used in compiled form in any way you desire.
This file or it's parts can't be redistributed without the authors
written consent, but can be modified for your private needs.
    Providing that this notice and the authors name and all copyright
notices remains intact.

    Please, an email me to know how you are using it and where. You can
ask me for any information about this below code or any attendant
knowledge.
    
    This file is provided "as is" with no expressed or implied warranty.
The author accepts no liability for any damage or loss of business that
this product may cause.
*/




// CPU.cpp: implementation of the CCPU class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "BK.h"
#include "CPU.h"
#include "Board.h"
#include "ExceptionHalt.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

// CPU timing
#define PI_TIME_BASE	  40
#define PI_TIME_HALT	 280
#define PI_TIME_RTI		 134
#define PI_TIME_RESET	3800
#define PI_TIME_BR_BASE	  54
#define PI_TIME_RTS		 108
#define PI_TIME_MARK	 118
#define PI_TIME_EMT		 228
#define PI_TIME_SOB		  68
#define PI_TIME_INT		 200

DWORD timing_A_A1[8]	={0,  40,  40,  68,  40,  68,  68, 94}; 
DWORD timing_AB[8]		={0,  54,  54,  80,  54,  80,  80, 108};
DWORD timing_B[8]		={0,  68,  68, 108,  68, 108, 108, 134};
DWORD timing_A2_Nj[8]	={0,  68,  68,  94,  68,  94,  94, 120};
DWORD timing_Ns[8]		={0, 108, 108, 134, 108, 134, 134, 160};

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CCPU::CCPU()
{
	m_pChip = NULL;
	m_psw = 0340;
	m_internalTick = 0;

	m_bStopInterrupt = FALSE;
	m_bKeyboardInterrupt = FALSE;
}

CCPU::~CCPU()
{
}



//////////////////////////////////////////////////////////////////////
void CCPU::TranslateInstruction ()
{
	m_instruction = m_pChip->GetWord (m_pChip->GetWord (PC));
	m_pChip->SetWord (PC, m_pChip->GetWord (PC) + 2);

	if (m_bStopInterrupt)
	{
		// Is "Stop" key is pressed
		m_bStopInterrupt = FALSE;
		MakeInterrupt (INTERRUPT_4); // Halt timing
		m_internalTick += PI_TIME_HALT;
		return;
	}

	// No fields
	switch (m_instruction)
	{
	case PI_HALT:
	case PI_HALT10:
	case PI_HALT11:
	case PI_HALT12:
	case PI_HALT13:
	case PI_HALT14:
	case PI_HALT15:
	case PI_HALT16:
	case PI_HALT17:
		ExecuteHALT ();	//HALT timing
		m_internalTick += PI_TIME_HALT;
		return;
	case PI_WAIT:
		ExecuteWAIT ();
		return;
	case PI_RTI:		// RTI timing
		ExecuteRTI ();
		m_internalTick += PI_TIME_RTI;
		return;
	case PI_BPT:
		ExecuteBPT ();	// EMT timing
		m_internalTick += PI_TIME_EMT;
		return;
	case PI_IOT:
		ExecuteIOT ();	// EMT timing
		m_internalTick += PI_TIME_EMT;
		return;
	case PI_RESET:		// Reset timing
		ExecuteRESET ();
		m_internalTick += PI_TIME_RESET;
		return;
	case PI_RTT:		// RTI timing
		ExecuteRTT ();
		m_internalTick += PI_TIME_RTI;
		return;
	case PI_NOP:		
		ExecuteNOP ();	// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_CLC:
		ExecuteCLC ();	// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_CLV:
		ExecuteCLV ();	// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_CLVC:
		ExecuteCLVC ();	// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_CLZ:
		ExecuteCLZ ();	// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_CLZC:
		ExecuteCLZC ();	// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_CLZV:
		ExecuteCLZV ();	// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_CLZVC:
		ExecuteCLZVC ();// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_CLN:
		ExecuteCLN ();	// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_CLNC:
		ExecuteCLNC ();	// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_CLNV:
		ExecuteCLNV ();	// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_CLNVC:
		ExecuteCLNVC ();// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_CLNZ:
		ExecuteCLNZ ();	// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_CLNZC:
		ExecuteCLNZC ();// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_CLNZV:
		ExecuteCLNZV ();// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_CCC:
		ExecuteCCC ();	// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_NOP260:
		ExecuteNOP ();	// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_SEC:
		ExecuteSEC ();	// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_SEV:
		ExecuteSEV ();	// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_SEVC:
		ExecuteSEVC ();	// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_SEZ:
		ExecuteSEZ ();	// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_SEZC:
		ExecuteSEZC ();	// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_SEZV:
		ExecuteSEZV ();	// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_SEZVC:
		ExecuteSEZVC ();// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_SEN:
		ExecuteSEN ();	// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_SENC:
		ExecuteSENC ();	// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_SENV:
		ExecuteSENV ();	// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_SENVC:
		ExecuteSENVC ();// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_SENZ:
		ExecuteSENZ ();	// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_SENZC:
		ExecuteSENZC ();// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_SENZV:
		ExecuteSENZV ();// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	case PI_SCC:
		ExecuteSCC ();	// Base timing
		m_internalTick += PI_TIME_BASE;
		return;
	}

	// Branchs & interrupts
	switch (m_instruction & ~(WORD)0377)
	{
	case PI_BR:
		ExecuteBR ();	// Base Branch timing
		m_internalTick += PI_TIME_BR_BASE;
		return;
	case PI_BNE:
		ExecuteBNE ();	// Base Branch timing
		m_internalTick += PI_TIME_BR_BASE;
		return;
	case PI_BEQ:
		ExecuteBEQ ();	// Base Branch timing
		m_internalTick += PI_TIME_BR_BASE;
		return;
	case PI_BGE:
		ExecuteBGE ();	// Base Branch timing
		m_internalTick += PI_TIME_BR_BASE;
		return;
	case PI_BLT:
		ExecuteBLT ();	// Base Branch timing
		m_internalTick += PI_TIME_BR_BASE;
		return;
	case PI_BGT:
		ExecuteBGT ();	// Base Branch timing
		m_internalTick += PI_TIME_BR_BASE;
		return;
	case PI_BLE:
		ExecuteBLE ();	// Base Branch timing
		m_internalTick += PI_TIME_BR_BASE;
		return;
	case PI_BPL:
		ExecuteBPL ();	// Base Branch timing
		m_internalTick += PI_TIME_BR_BASE;
		return;
	case PI_BMI:
		ExecuteBMI ();	// Base Branch timing
		m_internalTick += PI_TIME_BR_BASE;
		return;
	case PI_BHI:
		ExecuteBHI ();	// Base Branch timing
		m_internalTick += PI_TIME_BR_BASE;
		return;
	case PI_BLOS:
		ExecuteBLOS ();	// Base Branch timing
		m_internalTick += PI_TIME_BR_BASE;
		return;
	case PI_BVC:
		ExecuteBVC ();	// Base Branch timing
		m_internalTick += PI_TIME_BR_BASE;
		return;
	case PI_BVS:
		ExecuteBVS ();	// Base Branch timing
		m_internalTick += PI_TIME_BR_BASE;
		return;
	case PI_BHIS:
		ExecuteBHIS ();	// Base Branch timing
		m_internalTick += PI_TIME_BR_BASE;
		return;
	case PI_BLO:
		ExecuteBLO ();	// Base Branch timing
		m_internalTick += PI_TIME_BR_BASE;
		return;

	case PI_EMT:
		ExecuteEMT();	// EMT timing
		m_internalTick += PI_TIME_EMT;
		return;
	case PI_TRAP:
		ExecuteTRAP ();	// EMT timing
		m_internalTick += PI_TIME_EMT;
		return;
	}

	// One field
	if ((m_instruction & ~(WORD)7) == PI_RTS)
	{
		ExecuteRTS (); // RTS timing
		m_internalTick += PI_TIME_RTS;
		return;
	}

	int meth = GetDigit (m_instruction, 1);

	// Two fields
	switch (m_instruction & ~(WORD)077)
	{
	case PI_JMP:	// Nj timing
		ExecuteJMP ();
		m_internalTick += timing_A2_Nj[meth];
		return;
	case PI_SWAB:	// AB timing
		ExecuteSWAB ();
		m_internalTick += PI_TIME_BASE + timing_AB[meth];
		return;
	case PI_MARK:	// MARK timing
		ExecuteMARK ();
		m_internalTick += PI_TIME_MARK;
		return;
	case PI_SXT:	// AB timing
		ExecuteSXT ();
		m_internalTick += PI_TIME_BASE + timing_AB[meth];
		return;
	case PI_MTPS:	// AB timing
		ExecuteMTPS ();
		m_internalTick += PI_TIME_BASE + timing_AB[meth];
		return;
	case PI_MFPS:	// AB timing
		ExecuteMFPS ();
		m_internalTick += PI_TIME_BASE + timing_AB[meth];
		return;
	}
	
	switch (m_instruction & ~(WORD)0100077)
	{
	case PI_CLR:	// AB timing
		ExecuteCLR ();
		m_internalTick += PI_TIME_BASE + timing_AB[meth];
		return;
	case PI_COM:	// AB timing
		ExecuteCOM ();
		m_internalTick += PI_TIME_BASE + timing_AB[meth];
		return;
	case PI_INC:	// AB timing
		ExecuteINC ();
		m_internalTick += PI_TIME_BASE + timing_AB[meth];
		return;
	case PI_DEC:	// AB timing
		ExecuteDEC ();
		m_internalTick += PI_TIME_BASE + timing_AB[meth];
		return;
	case PI_NEG:	// AB timing
		ExecuteNEG ();
		m_internalTick += PI_TIME_BASE + timing_AB[meth];
		return;
	case PI_ADC:	// AB timing
		ExecuteADC ();
		m_internalTick += PI_TIME_BASE + timing_AB[meth];
		return;
	case PI_SBC:	// AB timing
		ExecuteSBC ();
		m_internalTick += PI_TIME_BASE + timing_AB[meth];
		return;
	case PI_TST:	// A1 timing
		ExecuteTST ();
		m_internalTick += PI_TIME_BASE + timing_A_A1[meth];
		return;
	case PI_ROR:	// AB timing
		ExecuteROR ();
		m_internalTick += PI_TIME_BASE + timing_AB[meth];
		return;
	case PI_ROL:	// AB timing
		ExecuteROL ();
		m_internalTick += PI_TIME_BASE + timing_AB[meth];
		return;
	case PI_ASR:	// AB timing
		ExecuteASR ();
		m_internalTick += PI_TIME_BASE + timing_AB[meth];
		return;
	case PI_ASL:	// AB timing
		ExecuteASL ();
		m_internalTick += PI_TIME_BASE + timing_AB[meth];
		return;
	}
	
	// Three fields
	switch(m_instruction & ~(WORD)0777)
	{
	case PI_JSR:	// Ns timing
		ExecuteJSR ();
		m_internalTick += timing_A2_Nj[meth];
		return;
	case PI_XOR:	// A2 timing
		ExecuteXOR ();
		m_internalTick += PI_TIME_BASE + timing_A2_Nj[meth];
		return;
	case PI_SOB:	// SOB timing
		ExecuteSOB ();
		m_internalTick += PI_TIME_SOB;
		return;
	}

	int methA = GetDigit (m_instruction, 3);
	int methB = GetDigit (m_instruction, 1);

	// Four fields
	switch(m_instruction & ~(WORD)0107777)
	{
	case PI_MOV:	// A + B timing
		ExecuteMOV ();
		m_internalTick += PI_TIME_BASE + timing_A_A1[methA] + timing_B[methB];
		return;
	case PI_CMP:	// A1 + A2 timing
		ExecuteCMP ();
		m_internalTick += PI_TIME_BASE + timing_A_A1[methA] + timing_A2_Nj[methB];
		return;
	case PI_BIT:	// A1 + A2 timing
		ExecuteBIT ();
		m_internalTick += PI_TIME_BASE + timing_A_A1[methA] + timing_A2_Nj[methB];
		return;
	case PI_BIC:	// A + B timing
		ExecuteBIC ();
		m_internalTick += PI_TIME_BASE + timing_A_A1[methA] + timing_B[methB];
		return;
	case PI_BIS:	// A + B timing
		ExecuteBIS ();
		m_internalTick += PI_TIME_BASE + timing_A_A1[methA] + timing_B[methB];
		return;
	}

	switch (m_instruction & ~(WORD)0007777)
	{
	case PI_ADD:	// A + B timing
		ExecuteADD ();
		m_internalTick += PI_TIME_BASE + timing_A_A1[methA] + timing_B[methB];
		return;
	case PI_SUB:	// A + B timing
		ExecuteSUB ();
		m_internalTick += PI_TIME_BASE + timing_A_A1[methA] + timing_B[methB];
		return;
	}

	//m_pChip->MakeInterrupt (INTERRUPT_10);
	MakeInterrupt (INTERRUPT_10); // Halt timing
}



//////////////////////////////////////////////////////////////////////
int CCPU::GetByteArg (int pos)
{
	int reg = (GetDigit (m_instruction, pos) << 1) + R0;
	int meth = GetDigit (m_instruction, pos + 1);
	
	int arg;

	switch (meth)
	{
	case 0:  // R0,		PC 
		if (pos == SRC)
			arg = m_pChip->GetByte (reg);
		else
			arg = reg;
		break;
		
	case 1: // (R0),	(PC)
		if (pos == SRC)
			arg = m_pChip->GetByte (m_pChip->GetWord (reg));
		else
			arg = m_pChip->GetWord (reg);
		break;
		
	case 2: // (R0)+,	#012345
		if (pos == SRC)
			arg = m_pChip->GetByte (m_pChip->GetWord (reg));
		else
			arg = m_pChip->GetWord (reg);
		
		if (reg == SP || reg == PC)
			m_pChip->SetWord (reg, m_pChip->GetWord (reg) + 2);
		else
			m_pChip->SetWord (reg, m_pChip->GetWord (reg) + 1);
		break;

	case 3: // @(R0)+,	@#012345
		if (pos == SRC)
			arg = m_pChip->GetByte (m_pChip->GetWord (m_pChip->GetWord (reg)));
		else
			arg = m_pChip->GetWord (m_pChip->GetWord (reg));
		
		m_pChip->SetWord (reg, m_pChip->GetWord (reg) + 2);
		break;

	case 4: // -(R0),	-(PC)
		if (reg == SP || reg == PC)
			m_pChip->SetWord (reg, m_pChip->GetWord (reg) - 2);
		else
			m_pChip->SetWord (reg, m_pChip->GetWord (reg) - 1);

		if (pos == SRC)
			arg = m_pChip->GetByte (m_pChip->GetWord (reg));
		else
			arg = m_pChip->GetWord (reg);
		break;

	case 5: // @-(R0),	@-(PC)
		m_pChip->SetWord (reg, m_pChip->GetWord (reg) - 2);
			
		if (pos == SRC)
			arg = m_pChip->GetByte (m_pChip->GetWord (m_pChip->GetWord (reg)));
		else
			arg = m_pChip->GetWord (m_pChip->GetWord (reg));
		break;

	case 6: // 345(R0),	345
		{
			WORD pc = m_pChip->GetWord (m_pChip->GetWord (PC));
			
			m_pChip->SetWord (PC, m_pChip->GetWord (PC) + 2);

			if (pos == SRC)
				arg = m_pChip->GetByte ((WORD)(pc + m_pChip->GetWord (reg)));
			else
				arg = (WORD)(pc + m_pChip->GetWord (reg));
			break;
		}

	case 7: // @345(R0),@345
		{
			WORD pc = m_pChip->GetWord (m_pChip->GetWord (PC));
			
			m_pChip->SetWord (PC, m_pChip->GetWord (PC) + 2);

			if (pos == SRC)
				arg = m_pChip->GetByte (m_pChip->GetWord ((WORD)(pc + m_pChip->GetWord (reg))));
			else
				arg = m_pChip->GetWord ((WORD)(pc + m_pChip->GetWord (reg)));
			break;
		}
	}

	return arg;
}



//////////////////////////////////////////////////////////////////////
int CCPU::GetWordArg (int pos)
{
	int reg = (GetDigit (m_instruction, pos) << 1) + R0;
	int meth = GetDigit (m_instruction, pos + 1);
	
	int arg;

	switch (meth)
	{
	case 0: // R0,		PC 
		if (pos == SRC)
			arg = m_pChip->GetWord (reg);
		else
			arg = reg;
		break;

	case 1: // (R0),	(PC)
		if (pos == SRC)
			arg = m_pChip->GetWord (m_pChip->GetWord (reg));
		else
			arg = m_pChip->GetWord (reg);
		break;

	case 2: // (R0)+,	#012345
		if (pos == SRC)
			arg = m_pChip->GetWord (m_pChip->GetWord (reg));
		else
			arg = m_pChip->GetWord (reg);

		m_pChip->SetWord (reg, m_pChip->GetWord (reg) + 2);
		break;

	case 3: // @(R0)+,	@#012345
		if (pos == SRC)
			arg = m_pChip->GetWord (m_pChip->GetWord (m_pChip->GetWord (reg)));
		else
			arg = m_pChip->GetWord (m_pChip->GetWord (reg));
		
		m_pChip->SetWord (reg, m_pChip->GetWord (reg) + 2);
		break;

	case 4: // -(R0),	-(PC)
		m_pChip->SetWord (reg, m_pChip->GetWord (reg) - 2);

		if (pos == SRC)
			arg = m_pChip->GetWord (m_pChip->GetWord (reg));
		else
			arg = m_pChip->GetWord (reg);
		break;

	case 5: // @-(R0),	@-(PC)
		m_pChip->SetWord (reg, m_pChip->GetWord (reg) - 2);

		if (pos == SRC)
			arg = m_pChip->GetWord (m_pChip->GetWord (m_pChip->GetWord (reg)));
		else
			arg = m_pChip->GetWord (m_pChip->GetWord (reg));
		break;

	case 6: // 345(R0),	345
		{
			WORD pc = m_pChip->GetWord (m_pChip->GetWord (PC));
			
			m_pChip->SetWord (PC, m_pChip->GetWord (PC) + 2);

			if (pos == SRC)
				arg = m_pChip->GetWord ((WORD)(pc + m_pChip->GetWord (reg)));
			else
				arg = (WORD)(pc + m_pChip->GetWord (reg));

			break;
		}
	case 7: // @345(R0),@345
		{
			WORD pc = m_pChip->GetWord (m_pChip->GetWord (PC));
			
			m_pChip->SetWord (PC, m_pChip->GetWord (PC) + 2);

			if (pos == SRC)
				arg = m_pChip->GetWord (m_pChip->GetWord ((WORD)(pc + m_pChip->GetWord (reg))));
			else
				arg = m_pChip->GetWord ((WORD)(pc + m_pChip->GetWord (reg)));

			break;
		}
	}

	return arg;
}



//////////////////////////////////////////////////////////////////////
int CCPU::GetDstWordArgAsBranch ()
{
	int reg = (GetDigit (m_instruction, 0) << 1) + R0;
	int meth = GetDigit (m_instruction, 1);
	
	int arg;

	try
	{
		switch (meth)
		{
		case 0: // R0,		PC 
			throw CExceptionHalt (m_instruction, "Wrong addressing");
			break;
		case 1: // (R0),	(PC)
			arg = m_pChip->GetWord (reg);
			break;
		case 2: // (R0)+,	#012345
			arg = m_pChip->GetWord (reg);
			m_pChip->SetWord (reg, m_pChip->GetWord (reg) + 2);
			break;
		case 3: // @(R0)+,	@#012345
			arg = m_pChip->GetWord (m_pChip->GetWord (reg));
			m_pChip->SetWord (reg, m_pChip->GetWord (reg) + 2);
			break;
		case 4: // -(R0),	-(PC)
			m_pChip->SetWord (reg, m_pChip->GetWord (reg) - 2);
			arg = m_pChip->GetWord (reg);
			break;
		case 5: // @-(R0),	@-(PC)
			m_pChip->SetWord (reg, m_pChip->GetWord (reg) - 2);
			arg = m_pChip->GetWord (m_pChip->GetWord (reg));
			break;
		case 6: // 345(R0),	345
			{
				WORD pc = m_pChip->GetWord (m_pChip->GetWord (PC));
				m_pChip->SetWord (PC, m_pChip->GetWord (PC) + 2);
				
				arg = (WORD)(pc + m_pChip->GetWord (reg));
			}
			break;
		case 7: // @345(R0),@345
			{
				WORD pc = m_pChip->GetWord (m_pChip->GetWord (PC));
				m_pChip->SetWord (PC, m_pChip->GetWord (PC) + 2);
				
				arg = m_pChip->GetWord ((WORD)(pc + m_pChip->GetWord (reg)));
			}
			break;
		}
	}
	catch (CExceptionHalt error)
	{
		if (meth & 2)
			m_pChip->SetWord (PC, m_pChip->GetWord (PC) + 2);

		throw CExceptionHalt (error.m_addr, error.m_info);
	}

	return arg;
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteWAIT ()
{
	if (m_bKeyboardInterrupt == FALSE && m_bStopInterrupt == FALSE)
		m_pChip->SetWord (PC, m_pChip->GetWord (PC) - 2);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteHALT ()
{
	m_pChip->MakeInterrupt (INTERRUPT_4);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteRTI ()
{
	m_pChip->SetWord (PC, m_pChip->GetWord (m_pChip->GetWord (SP)));
	m_pChip->SetWord (SP, m_pChip->GetWord (SP) + 2);
	
	m_psw = m_pChip->GetWord (m_pChip->GetWord (SP));
	m_pChip->SetWord (SP, m_pChip->GetWord (SP) + 2);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteBPT ()
{
	m_pChip->MakeInterrupt (INTERRUPT_14);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteIOT ()
{
	m_pChip->MakeInterrupt (INTERRUPT_20);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteRESET ()
{
	m_pChip->SetWordIndirect (0177660, 0100);
	m_pChip->SetWordIndirect (0177662, 0);
	m_pChip->SetWordIndirect (0177716, m_pChip->GetWordIndirect (0177716) | 0300);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteRTT ()
{
	m_pChip->SetWord (PC, m_pChip->GetWord (m_pChip->GetWord (SP)));
	m_pChip->SetWord (SP, m_pChip->GetWord (SP) + 2);
	
	m_psw = m_pChip->GetWord (m_pChip->GetWord (SP));
	m_pChip->SetWord (SP, m_pChip->GetWord (SP) + 2);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteRTS ()
{
	int reg = (GetDigit (m_instruction, 0) << 1) + R0;

	/*
	if (reg == PC)
		m_pChip->SetWord (PC, m_pChip->GetWord (m_pChip->GetWord (SP)));
	else
	{
	*/
		m_pChip->SetWord (PC, m_pChip->GetWord (reg));
		m_pChip->SetWord (reg, m_pChip->GetWord (m_pChip->GetWord (SP)));
	//}

	m_pChip->SetWord (SP, m_pChip->GetWord (SP) + 2);
}


//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteNOP ()
{
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteCLC ()
{
	SetC (FALSE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteCLV ()
{
	SetV (FALSE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteCLVC ()
{
	SetV (FALSE);
	SetC (FALSE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteCLZ ()
{
	SetZ (FALSE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteCLZC ()
{
	SetZ (FALSE);
	SetC (FALSE);
}


//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteCLZV ()
{
	SetZ (FALSE);
	SetV (FALSE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteCLZVC ()
{
	SetZ (FALSE);
	SetV (FALSE);
	SetC (FALSE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteCLN ()
{
	SetN (FALSE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteCLNC ()
{
	SetN (FALSE);
	SetC (FALSE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteCLNV ()
{
	SetN (FALSE);
	SetV (FALSE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteCLNVC ()
{
	SetN (FALSE);
	SetV (FALSE);
	SetZ (FALSE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteCLNZ ()
{
	SetN (FALSE);
	SetZ (FALSE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteCLNZC ()
{
	SetN (FALSE);
	SetZ (FALSE);
	SetC (FALSE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteCLNZV ()
{
	SetN (FALSE);
	SetZ (FALSE);
	SetV (FALSE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteCCC ()
{
	SetC (FALSE);
	SetV (FALSE);
	SetZ (FALSE);
	SetN (FALSE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteSEC ()
{
	SetC (TRUE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteSEV ()
{
	SetV (TRUE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteSEVC ()
{
	SetV (TRUE);
	SetC (TRUE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteSEZ ()
{
	SetZ (TRUE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteSEZC ()
{
	SetZ (TRUE);
	SetC (TRUE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteSEZV ()
{
	SetZ (TRUE);
	SetV (TRUE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteSEZVC ()
{
	SetZ (TRUE);
	SetV (TRUE);
	SetC (TRUE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteSEN ()
{
	SetN (TRUE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteSENC ()
{
	SetN (TRUE);
	SetZ (TRUE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteSENV ()
{
	SetN (TRUE);
	SetV (TRUE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteSENVC ()
{
	SetN (TRUE);
	SetV (TRUE);
	SetC (TRUE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteSENZ ()
{
	SetN (TRUE);
	SetZ (TRUE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteSENZC ()
{
	SetN (TRUE);
	SetZ (TRUE);
	SetC (TRUE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteSENZV ()
{
	SetN (TRUE);
	SetZ (TRUE);
	SetV (TRUE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteSCC ()
{
	SetC (TRUE);
	SetV (TRUE);
	SetZ (TRUE);
	SetN (TRUE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteJMP ()
{
	m_pChip->SetWord (PC, GetDstWordArgAsBranch ());
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteSWAB ()
{
	int dstAddr = GetWordArg (DST);
	WORD dst = m_pChip->GetWord (dstAddr);

	dst = (WORD)(dst << 8) | (WORD)(dst >> 8);

	m_pChip->SetWord (dstAddr, dst);

	SetC (FALSE);
	SetV (FALSE);
	SetZ (CheckForZero ((BYTE)dst));
	SetN (CheckForNegative ((BYTE)dst));
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteCLR ()
{
	if (m_instruction & 0100000)
	{
		int dstAddr = GetByteArg (DST);
		m_pChip->SetByte (dstAddr, 0);
	}
	else
	{
		int dstAddr = GetWordArg (DST);
		m_pChip->SetWord (dstAddr, 0);
	}

	SetC (FALSE);
	SetV (FALSE);
	SetZ (TRUE);
	SetN (FALSE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteCOM ()
{
	if (m_instruction & 0100000)
	{
		int dstAddr = GetByteArg (DST);
		BYTE dst = m_pChip->GetByte (dstAddr);

		dst = ~dst;
		
		m_pChip->SetByte (dstAddr, dst);

		SetN (CheckForNegative (dst));
		SetZ (CheckForZero (dst));
	}
	else
	{
		int dstAddr = GetWordArg (DST);
		WORD dst = m_pChip->GetWord (dstAddr);

		dst = ~dst;
		
		m_pChip->SetWord (dstAddr, dst);

		SetN (CheckForNegative (dst));
		SetZ (CheckForZero (dst));
	}

	SetC (TRUE);
	SetV (FALSE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteINC ()
{
	if (m_instruction & 0100000)
	{
		int dstAddr = GetByteArg (DST);
		BYTE dst = m_pChip->GetByte (dstAddr);

		BOOL bOverflow = CheckAddForOverflow (dst, 1);

		dst += 1;
		
		m_pChip->SetByte (dstAddr, dst);

		SetN (CheckForNegative (dst));
		SetZ (CheckForZero (dst));
		SetV (bOverflow);
	}
	else
	{
		int dstAddr = GetWordArg (DST);
		WORD dst = m_pChip->GetWord (dstAddr);

		BOOL bOverflow = CheckAddForOverflow (dst, 1);

		dst += 1;
		
		m_pChip->SetWord (dstAddr, dst);

		SetN (CheckForNegative (dst));
		SetZ (CheckForZero (dst));
		SetV (bOverflow);
	}
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteDEC ()
{
	if (m_instruction & 0100000)
	{
		int dstAddr = GetByteArg (DST);
		BYTE dst = m_pChip->GetByte (dstAddr);

		BOOL bOverflow = CheckSubForOverflow (dst, 1);
		
		dst -= 1;

		m_pChip->SetByte (dstAddr, dst);
		
		SetN (CheckForNegative (dst));
		SetZ (CheckForZero (dst));
		SetV (bOverflow);
	}
	else
	{
		int dstaddr = GetWordArg (DST);
		WORD dst = m_pChip->GetWord (dstaddr);

		BOOL bOverflow = CheckSubForOverflow (dst, 1);
		
		dst -= 1;

		m_pChip->SetWord (dstaddr, dst);
		
		SetN (CheckForNegative (dst));
		SetZ (CheckForZero (dst));
		SetV (bOverflow);
	}
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteNEG ()
{
	if (m_instruction & 0100000)
	{
		int dstAddr = GetByteArg (DST);
		BYTE dst = m_pChip->GetByte (dstAddr);

		dst = ~dst + 1;

		m_pChip->SetByte (dstAddr, dst);

		SetN (CheckForNegative (dst));
		SetZ (CheckForZero (dst));
		SetV (dst == 0200);
		SetC (dst != 0200);
	}
	else
	{
		int dstAddr = GetWordArg (DST);
		WORD dst = m_pChip->GetWord (dstAddr);

		dst = ~dst + 1;
		
		m_pChip->SetWord (dstAddr, dst);

		SetN (CheckForNegative (dst));
		SetZ (CheckForZero (dst));
		SetV (dst == 0100000);
		SetC (dst != 0100000);
	}
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteADC ()
{
	BOOL bOverflow = FALSE;
	BOOL bCarry = FALSE;

	if (m_instruction & 0100000)
	{
		int dstAddr = GetByteArg (DST);
		BYTE dst = m_pChip->GetByte (dstAddr);

		if (GetC ())
		{
			bOverflow = CheckAddForOverflow (dst, 1);
			bCarry = CheckAddForCarry (dst, 1);

			dst += 1;
			
			m_pChip->SetByte (dstAddr, dst);
		}

		SetN (CheckForNegative (dst));
		SetZ (CheckForZero (dst));
	}
	else
	{
		int dstAddr = GetWordArg (DST);
		WORD dst = m_pChip->GetWord (dstAddr);

		if (GetC ())
		{
			bOverflow = CheckAddForOverflow (dst, 1);
			bCarry = CheckAddForCarry (dst, 1);

			dst += 1;
			
			m_pChip->SetWord (dstAddr, dst);
		}

		SetN (CheckForNegative (dst));
		SetZ (CheckForZero (dst));
	}

	SetV (bOverflow);
	SetC (bCarry);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteSBC ()
{
	BOOL bOverflow = FALSE;
	BOOL bCarry = FALSE;

	if (m_instruction & 0100000)
	{
		int dstaddr = GetByteArg (DST);
		BYTE dst = m_pChip->GetByte (dstaddr);

		if (GetC ())
		{
			bOverflow = CheckSubForOverflow (dst, 1);
			bCarry = CheckSubForCarry (dst, 1);

			dst -= 1;
			
			m_pChip->SetByte (dstaddr, dst);
		}

		SetN (CheckForNegative (dst));
		SetZ (CheckForZero (dst));
	}
	else
	{
		int dstAddr = GetWordArg (DST);
		WORD dst = m_pChip->GetWord (dstAddr);

		if (GetC ())
		{
			bOverflow = CheckSubForOverflow (dst, 1);
			bCarry = CheckSubForCarry (dst, 1);

			dst -= 1;
			
			m_pChip->SetWord (dstAddr, dst);
		}

		SetN (CheckForNegative (dst));
		SetZ (CheckForZero (dst));
	}

	SetV (bOverflow);
	SetC (bCarry);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteTST ()
{
	if (m_instruction & 0100000)
	{
		int dstAddr = GetByteArg (DST);
		BYTE dst = m_pChip->GetByte (dstAddr);

		SetN (CheckForNegative (dst));
		SetZ (CheckForZero (dst));
	}
	else
	{
		int dstAddr = GetWordArg (DST);
		WORD dst = m_pChip->GetWord (dstAddr);

		SetN (CheckForNegative (dst));
		SetZ (CheckForZero (dst));
	}

	SetV (FALSE);
	SetC (FALSE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteROR ()
{
	if (m_instruction & 0100000)
	{
		int dstAddr = GetByteArg (DST);
		BYTE dst = m_pChip->GetByte (dstAddr);

		BYTE befor = dst;
		BOOL bCarry = dst & (BYTE)1;

		dst >>= 1;
		
		if (GetC ())
			dst |= 0200;

		m_pChip->SetByte (dstAddr, dst);

		SetN (CheckForNegative (dst));
		SetZ (CheckForZero (dst));
		SetV (GetN () ^ bCarry);
		SetC (bCarry);
	}
	else
	{
		int dstAddr = GetWordArg (DST);
		WORD dst = m_pChip->GetWord (dstAddr);

		WORD befor = dst;
		BOOL bCarry = dst & (WORD)1;
		
		dst >>= 1;

		if (GetC ())
			dst |= 0100000;

		m_pChip->SetWord (dstAddr, dst);

		SetN (CheckForNegative (dst));
		SetZ (CheckForZero (dst));
		SetV (GetN () ^ bCarry);
		SetC (bCarry);
	}
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteROL ()
{
	if (m_instruction & 0100000)
	{
		int dstAddr = GetByteArg (DST);
		BYTE dst = m_pChip->GetByte (dstAddr);

		BYTE befor = dst;
		BOOL bCarry = dst & (BYTE)0200;

		dst <<= 1;
		
		if (GetC ())
			dst |= 1;

		m_pChip->SetByte (dstAddr, dst);

		SetN (CheckForNegative (dst));
		SetZ (CheckForZero (dst));
		SetV (GetN () ^ bCarry);
		SetC (bCarry);
	}
	else
	{
		int dstAddr = GetWordArg (DST);
		WORD dst = m_pChip->GetWord (dstAddr);

		WORD befor = dst;
		BOOL bCarry = dst & (WORD)0100000;
		
		dst <<= 1;

		if (GetC ())
			dst |= 1;

		m_pChip->SetWord (dstAddr, dst);

		SetN (CheckForNegative (dst));
		SetZ (CheckForZero (dst));
		SetV (GetN () ^ bCarry);
		SetC (bCarry);
	}
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteASR ()
{
	if (m_instruction & 0100000)
	{
		int dstAddr = GetByteArg (DST);
		BYTE dst = m_pChip->GetByte (dstAddr);

		BYTE befor = dst;
		BOOL bCarry = dst & (BYTE)1;

		dst >>= 1;
		
		if (CheckForNegative ((BYTE)befor))
			dst |= 0200;

		m_pChip->SetByte (dstAddr, dst);

		SetN (CheckForNegative (dst));
		SetZ (CheckForZero (dst));
		SetV (GetN () ^ bCarry);
		SetC (bCarry);
	}
	else
	{
		int dstAddr = GetWordArg (DST);
		WORD dst = m_pChip->GetWord (dstAddr);

		WORD befor = dst;
		BOOL bCarry = dst & (WORD)1;
		
		dst >>= 1;

		if (CheckForNegative ((WORD)befor))
			dst |= 0100000;

		m_pChip->SetWord (dstAddr, dst);

		SetN (CheckForNegative (dst));
		SetZ (CheckForZero (dst));
		SetV (GetN () ^ bCarry);
		SetC (bCarry);
	}
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteASL ()
{
	if (m_instruction & 0100000)
	{
		int dstAddr = GetByteArg (DST);
		BYTE dst = m_pChip->GetByte (dstAddr);

		BYTE befor = dst;
		BOOL bCarry = dst & (BYTE)0200;

		dst <<= 1;

		m_pChip->SetByte (dstAddr, dst);
		
		SetN (CheckForNegative (dst));
		SetZ (CheckForZero (dst));
		SetV (GetN () ^ bCarry);
		SetC (bCarry);
	}
	else
	{
		int dstAddr = GetWordArg (DST);
		WORD dst = m_pChip->GetWord (dstAddr);

		WORD befor = dst;
		BOOL bCarry = dst & (WORD)0100000;
		
		dst <<= 1;

		m_pChip->SetWord (dstAddr, dst);

		SetN (CheckForNegative (dst));
		SetZ (CheckForZero (dst));
		SetV (GetN () ^ bCarry);
		SetC (bCarry);
	}
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteSXT ()
{
	int dstAddr = GetWordArg (DST);
	
	if (GetN())
		m_pChip->SetWord (dstAddr, 0177777);
	else
		m_pChip->SetWord (dstAddr, 0);

	WORD dst = m_pChip->GetWord (dstAddr);

	SetZ (CheckForZero (dst));
	SetV (FALSE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteMTPS ()
{
	m_psw = m_pChip->GetByte (GetByteArg (DST));
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteMFPS ()
{
	int dstAddr = GetByteArg (DST);

	if (dstAddr >= R0 && dstAddr <= PC)
	{
		short exSrc = (char)m_psw;
		m_pChip->SetWord (dstAddr, exSrc);
	}
	else
		m_pChip->SetByte (dstAddr, m_psw);

	//m_pChip->SetByte (GetByteArg (DST), m_psw);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteBR ()
{
	m_pChip->SetWord (PC, m_pChip->GetWord (PC) + ((short)(char)LOBYTE (m_instruction)) * 2);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteBNE ()
{
	if (!GetZ ())
		m_pChip->SetWord (PC, m_pChip->GetWord (PC) + ((short)(char)LOBYTE (m_instruction)) * 2);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteBEQ ()
{
	if (GetZ ())
		m_pChip->SetWord (PC, m_pChip->GetWord (PC) + ((short)(char)LOBYTE (m_instruction)) * 2);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteBGE ()
{
	if (GetN () == GetV ())
		m_pChip->SetWord (PC, m_pChip->GetWord (PC) + ((short)(char)LOBYTE (m_instruction)) * 2);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteBLT ()
{
	if (GetN () != GetV ())
		m_pChip->SetWord (PC, m_pChip->GetWord (PC) + ((short)(char)LOBYTE (m_instruction)) * 2);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteBGT ()
{
	if (!((GetN () != GetV ()) || GetZ ()))
		m_pChip->SetWord (PC, m_pChip->GetWord (PC) + ((short)(char)LOBYTE (m_instruction)) * 2);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteBLE ()
{
	if ((GetN () != GetV ()) || GetZ ())
		m_pChip->SetWord (PC, m_pChip->GetWord (PC) + ((short)(char)LOBYTE (m_instruction)) * 2);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteBPL ()
{
	if (!GetN ())
		m_pChip->SetWord (PC, m_pChip->GetWord (PC) + ((short)(char)LOBYTE (m_instruction)) * 2);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteBMI ()
{
	if (GetN ())
		m_pChip->SetWord (PC, m_pChip->GetWord (PC) + ((short)(char)LOBYTE (m_instruction)) * 2);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteBHI ()
{
	if (!(GetZ () || GetC ()))
		m_pChip->SetWord (PC, m_pChip->GetWord (PC) + ((short)(char)LOBYTE (m_instruction)) * 2);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteBLOS ()
{
	if (GetZ () || GetC ())
		m_pChip->SetWord (PC, m_pChip->GetWord (PC) + ((short)(char)LOBYTE (m_instruction)) * 2);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteBVC ()
{
	if (!GetV ())
		m_pChip->SetWord (PC, m_pChip->GetWord (PC) + ((short)(char)LOBYTE (m_instruction)) * 2);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteBVS ()
{
	if (GetV ())
		m_pChip->SetWord (PC, m_pChip->GetWord (PC) + ((short)(char)LOBYTE (m_instruction)) * 2);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteBHIS ()
{
	if (!GetC ())
		m_pChip->SetWord (PC, m_pChip->GetWord (PC) + ((short)(char)LOBYTE (m_instruction)) * 2);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteBLO ()
{
	if (GetC ())
		m_pChip->SetWord (PC, m_pChip->GetWord (PC) + ((short)(char)LOBYTE (m_instruction)) * 2);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteXOR ()
{
	int reg = (GetDigit (m_instruction, 2) << 1) + R0;
	WORD src = m_pChip->GetWord (reg);

	int dstAddr = GetWordArg (DST);
	WORD dst = m_pChip->GetWord (dstAddr);

	dst = dst ^ src;
	
	m_pChip->SetWord (dstAddr, dst);

	SetN (CheckForNegative (dst));
	SetZ (CheckForZero (dst));
	SetV (FALSE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteSOB ()
{
	int reg = (GetDigit (m_instruction, 2) << 1) + R0;
	WORD dst = m_pChip->GetWord (reg);
	
	if (--dst)
		m_pChip->SetWord (PC, m_pChip->GetWord (PC) - (m_instruction & (WORD)077) * 2);

	m_pChip->SetWord (reg, dst);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteMOV ()
{
	if (m_instruction & 0100000)
	{
		BYTE src = GetByteArg (SRC);
		int dstAddr = GetByteArg (DST);

		if (dstAddr >= R0 && dstAddr <= PC)
		{
			short exSrc = (char)src;
			m_pChip->SetWord (dstAddr, exSrc);
		}
		else
			m_pChip->SetByte (dstAddr, src);

		BYTE dst = m_pChip->GetByte (dstAddr);

		SetN (CheckForNegative (dst));
		SetZ (CheckForZero (dst));
	}
	else
	{
		WORD src = GetWordArg (SRC);
		int dstAddr = GetWordArg (DST);

		m_pChip->SetWord (dstAddr, src);
		WORD dst = m_pChip->GetWord (dstAddr);

		SetN (CheckForNegative (dst));
		SetZ (CheckForZero (dst));
	}

	SetV (FALSE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteCMP ()
{
	if (m_instruction & 0100000)
	{
		BYTE src = GetByteArg (SRC);
		BYTE dst = m_pChip->GetByte (GetByteArg (DST));

		SetN (CheckForNegative ((BYTE)(src - dst)));
		SetZ (CheckForZero ((BYTE)(src - dst)));
		SetV (CheckSubForOverflow (src, dst));
		SetC (CheckSubForCarry (src, dst));
	}
	else
	{
		WORD src = GetWordArg (SRC);
		WORD dst = m_pChip->GetWord (GetWordArg (DST));

		SetN (CheckForNegative ((WORD)(src - dst)));
		SetZ (CheckForZero ((WORD)(src - dst)));
		SetV (CheckSubForOverflow (src, dst));
		SetC (CheckSubForCarry (src, dst));
	}
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteBIT ()
{
	if (m_instruction & 0100000)
	{
		BYTE src = GetByteArg (SRC);
		BYTE dst = m_pChip->GetByte (GetByteArg (DST));

		SetN (CheckForNegative ((BYTE)(src & dst)));
		SetZ (CheckForZero ((BYTE)(src & dst)));
	}
	else
	{
		WORD src = GetWordArg (SRC);
		WORD dst = m_pChip->GetWord (GetWordArg (DST));

		SetN (CheckForNegative ((WORD)(src & dst)));
		SetZ (CheckForZero ((WORD)(src & dst)));
	}

	SetV (FALSE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteBIC ()
{
	if (m_instruction & 0100000)	
	{
		BYTE src = GetByteArg (SRC);
		int dstAddr = GetByteArg (DST);
		BYTE dst = m_pChip->GetByte (dstAddr);

		dst = dst & ~src;
		
		m_pChip->SetByte (dstAddr, dst);

		SetN (CheckForNegative (dst));
		SetZ (CheckForZero (dst));
	}
	else
	{
		WORD src = GetWordArg (SRC);
		int dstAddr = GetWordArg (DST);
		WORD dst = m_pChip->GetWord (dstAddr);

		dst = dst & ~src;
		
		m_pChip->SetWord (dstAddr, dst);

		SetN (CheckForNegative (dst));
		SetZ (CheckForZero (dst));
	}

	SetV (FALSE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteBIS ()
{
	if (m_instruction & 0100000)
	{
		BYTE src = GetByteArg (SRC);
		int dstAddr = GetByteArg (DST);
		BYTE dst = m_pChip->GetByte (dstAddr);

		dst = dst | src;

		m_pChip->SetByte (dstAddr, dst);

		SetN (CheckForNegative (dst));
		SetZ (CheckForZero (dst));
	}
	else
	{
		WORD src = GetWordArg (SRC);
		int dstAddr = GetWordArg (DST);
		WORD dst = m_pChip->GetWord (dstAddr);

		dst = dst | src;

		m_pChip->SetWord (dstAddr, dst);

		SetN (CheckForNegative (dst));
		SetZ (CheckForZero (dst));
	}

	SetV (FALSE);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteADD ()
{
	WORD src = GetWordArg (SRC);
	int dstAddr = GetWordArg (DST);
	WORD dst = m_pChip->GetWord (dstAddr);

	SetN (CheckForNegative ((WORD)(dst + src)));
	SetZ (CheckForZero ((WORD)(dst + src)));
	SetV (CheckAddForOverflow (dst, src));
	SetC (CheckAddForCarry (dst, src));

	dst += src;

	m_pChip->SetWord (dstAddr, dst);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteSUB ()
{
	WORD src = GetWordArg (SRC);
	int dstAddr = GetWordArg (DST);
	WORD dst = m_pChip->GetWord (dstAddr);

	SetN (CheckForNegative ((WORD)(dst - src)));
	SetZ (CheckForZero ((WORD)(dst - src)));
	SetV (CheckSubForOverflow (dst, src));
	SetC (CheckSubForCarry (dst, src));

	dst -= src;

	m_pChip->SetWord (dstAddr, dst);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteEMT ()
{
	m_pChip->MakeInterrupt (INTERRUPT_30);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteTRAP ()
{
	m_pChip->MakeInterrupt (INTERRUPT_34);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteJSR ()
{
	WORD pc = GetDstWordArgAsBranch ();
	
	m_pChip->SetWord (SP, m_pChip->GetWord (SP) - 2);
	
	int reg = (GetDigit (m_instruction, 2) << 1) + R0;

	/*
	if (reg == PC)
		m_pChip->SetWord (m_pChip->GetWord (SP), m_pChip->GetWord (PC));
	else
	{
	*/
		m_pChip->SetWord (m_pChip->GetWord (SP), m_pChip->GetWord (reg));
		m_pChip->SetWord (reg, m_pChip->GetWord (PC));
	//}

	m_pChip->SetWord (PC, pc);
}



//////////////////////////////////////////////////////////////////////
void CCPU::ExecuteMARK ()
{
	m_pChip->SetWord (PC, m_pChip->GetWord (R5));
	m_pChip->SetWord (R5, m_pChip->GetWord (m_pChip->GetWord (SP)));
	m_pChip->SetWord (SP, m_pChip->GetWord (SP) + (m_instruction & 0x003F) * 2 + 2);
}



//////////////////////////////////////////////////////////////////////
void CCPU::MakeInterrupt (WORD interrupt)
{
	m_pChip->SetWord (SP, m_pChip->GetWord (SP) - 2);
	m_pChip->SetWord (m_pChip->GetWord (SP), m_psw);
	m_pChip->SetWord (SP, m_pChip->GetWord (SP) - 2);
	m_pChip->SetWord (m_pChip->GetWord (SP), m_pChip->GetWord (PC));
	m_pChip->SetWord (PC, m_pChip->GetWord (interrupt));
	m_psw = m_pChip->GetWord (interrupt + 2);

	m_internalTick += PI_TIME_INT;
}



//////////////////////////////////////////////////////////////////////
void CCPU::StopInterrupt ()
{
	m_bStopInterrupt = TRUE;
}



//////////////////////////////////////////////////////////////////////
void CCPU::KeyboardInterript ()
{
	m_bKeyboardInterrupt = TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CCPU::IsTrace ()
{
	if (/*((m_psw & PSW_P) == 0) &&*/ (m_psw & PSW_T) && (m_instruction != PI_RTT))
		return TRUE;

	return FALSE;
}