/*************************************************
*            BK Emulator for Window 3.0          *
*************************************************/

/*
File: Board_FDD.cpp
Version: 1.0.0
Written by: Yuriy Kalmykov <kalmykov@stoik.com>
    Copyright (c) 2002-2004 Yuriy Kalmykov

    BK Emulator is a program emulated hardware environment for running
code for BK 0010(01) in different configurations. 
           
    This code may be used in compiled form in any way you desire.
This file or it's parts can't be redistributed without the authors
written consent, but can be modified for your private needs.
    Providing that this notice and the authors name and all copyright
notices remains intact.

    Please, an email me to know how you are using it and where. You can
ask me for any information about this below code or any attendant
knowledge.
    
    This file is provided "as is" with no expressed or implied warranty.
The author accepts no liability for any damage or loss of business that
this product may cause.
*/




// Board_FDD.cpp: implementation of the CMotherBoard_FDD class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "bk.h"
#include "Board_FDD.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CMotherBoard_FDD::CMotherBoard_FDD()
{
	m_bBasic = FALSE;
	m_pBasic = new BYTE [8192 * 3];
	m_pExt = new BYTE [8192 * 3];
}

CMotherBoard_FDD::~CMotherBoard_FDD()
{
	delete []m_pBasic;
	delete []m_pExt;
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard_FDD::InitMemoryModules ()
{
	// Load Basic ROMs in RAM pages
	BYTE* pOldMem = m_pMemory;
	m_pMemory = m_pBasic;
	LoadRomModule (IDS_INI_BASIC0, 0);
	LoadRomModule (IDS_INI_BASIC1, 020000);
	LoadRomModule (IDS_INI_REGISTERS, 040000);

	// Load FDD driver
	m_pMemory = pOldMem;
	LoadRomModule (IDS_INI_MONITOR, 0100000);
	LoadRomModule (IDS_INI_FDR, 0160000);

	// Save Extended memory and FDD driver
	memcpy (m_pExt, &m_pMemory[0120000], 8192 * 3);
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard_FDD::SetByte (int addr, BYTE value)
{
	addr = ConvertAddress (addr);

	int iMemPage = addr >> 13;

	// memory on 0 - 040000
	if ((iMemPage >= 0 && iMemPage < 2) || iMemPage == 8)
	{
		m_pMemory[addr] = value;
		return;
	}

	// screen memory on 040000 - 0100000
	if (iMemPage >= 2 && iMemPage < 4)
	{
		OnSetScreenCell (addr, value);
		m_pMemory[addr] = value;
		return;
	}

	// monitor on 0100000 - 0120000
	if (iMemPage >= 4 && iMemPage < 5)
		throw CExceptionHalt (addr, "Can't write this address.");
	
	// memory on 0120000 - 0160000
	if (iMemPage >= 5 && iMemPage < 7)
	{
		if (!IsBasicPage ())
		{
			m_pMemory[addr] = value;
			return;
		}
		else
			throw CExceptionHalt (addr, "Can't write this address.");
	}

	if (IsRegister (addr))
		OnSetSystemRegister (addr, (WORD*)&m_pMemory[addr], value);
	else
		throw CExceptionHalt (addr, "Can't write this address.");
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard_FDD::SetWord (int addr, WORD value)
{
	addr = ConvertAddress (addr);

	int iMemPage = addr >> 13;
	addr &= 0377776;

	// memory on 0 - 040000
	if ((iMemPage >= 0 && iMemPage < 2) || iMemPage == 8)
	{
		*(WORD*)&m_pMemory[addr] = value;
		return;
	}

	// screen memory on 040000 - 0100000
	if (iMemPage >= 2 && iMemPage < 4)
	{
		OnSetScreenCell (addr, value);
		*(WORD*)&m_pMemory[addr] = value;
		return;
	}

	// monitor on 0100000 - 0120000
	if (iMemPage >= 4 && iMemPage < 5)
		throw CExceptionHalt (addr, "Can't write this address.");
	
	// memory on 0120000 - 0160000
	if (iMemPage >= 5 && iMemPage < 7)
	{
		if (!IsBasicPage ())
		{
			*(WORD*)&m_pMemory[addr] = value;
			return;
		}
		else
			throw CExceptionHalt (addr, "Can't write this address.");
	}

	if (IsRegister (addr))
		OnSetSystemRegister (addr, (WORD*)&m_pMemory[addr], value);
	else
		throw CExceptionHalt (addr, "Can't write this address.");
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard_FDD::OnSetSystemRegister (WORD num, WORD* pDst, WORD src)
{
	CMotherBoard::OnSetSystemRegister (num, pDst, src);

	switch (num)
	{
	case 0177130:

	if (src & 014)
		SetMemoryPage (TRUE);
	else
		SetMemoryPage (FALSE);
	break;
	}
}



//////////////////////////////////////////////////////////////////////
BOOL CMotherBoard_FDD::IsRegister (int addr)
{
	if (addr >= 0177100)
	{
		switch (addr)
		{
		case 0177660:
		case 0177662:
		case 0177664:
		case 0177706:
		case 0177710:
		case 0177712:
		case 0177714:
		case 0177716:
		case 0177130:
		case 0177132:
			return TRUE;
		}
	}

	return FALSE;
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard_FDD::Interception ()
{
	switch (GetWordIndirect (PC))
	{
	case 0116076:
		
		if (m_pParent)
			m_pParent->PostMessage (WM_TAPE_FILE_LOAD, 0, 0);

		break;

	case 0160372:
		EmulateFDD ();
		SetWordIndirect (PC, 0161564);

		break;
	}		
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard_FDD::SetMemoryPage (BOOL bBasic)
{
	if (bBasic == m_bBasic)
		return;

	if (bBasic)
	{
		memcpy (m_pExt, &m_pMemory[0120000], 060000);
		memcpy (&m_pMemory[0120000], m_pBasic, 060000);
	}
	else
		memcpy (&m_pMemory[0120000], m_pExt, 060000);

	m_bBasic = bBasic;
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard_FDD::EmulateFDD ()
{
	TABLE dt;

	WORD psw = m_cpu.GetPSW ();
	WORD table_addr = GetWordIndirect (R3);

	for (int i = 0; i < sizeof (dt) / 2; i++)
	{
		((WORD*)&dt)[i] = GetWordIndirect (table_addr + i * 2);
	}

	ReadDrivesPath ();
	
	// If drive number out of range A: - D:

	if (dt.sign_tab[0] != 0)
	{
		m_cpu.SetPSW (psw | PSW_C);
		SetWordIndirect (052, FDD_BAD_FORMAT);
		return;
	}

	if (dt.drive_num > 3)
	{
		m_cpu.SetPSW (psw | PSW_C);
		SetWordIndirect (052, FDD_0_TRACK_ERROR);
		return;
	}

	//dt.drive_num &= 1; Old for MK-DOS

	CFile imgFile;

	// If read/write length equal zero - return without error
	if (!dt.working_length)
	{
		m_cpu.SetPSW (psw & ~PSW_C);
		SetWordIndirect (052, FDD_NOERROR);
		return;
	}

	BOOL bRes = TRUE;

	// Negative lengh is to write data
	if (dt.working_length > 0)
		// Open img in read mode
		bRes = imgFile.Open (m_strDrives[dt.drive_num], CFile::modeRead);
	else
	{
		// Open img in write mode
		bRes = imgFile.Open (m_strDrives[dt.drive_num], CFile::modeWrite);

		if (!bRes)
		{
			// Of open in write mode error try to open in read mode
			bRes = imgFile.Open (m_strDrives[dt.drive_num], CFile::modeRead);

			if (bRes)
			{
				// If done - file exists but write protected
				m_cpu.SetPSW (psw | PSW_C);
				SetWordIndirect (052, FDD_DISK_PROTECTED);
				return;	
			}
		}
	}

	// If invalid disk path - return with no disk error
	if (!bRes)
	{
		m_cpu.SetPSW (psw | PSW_C);
		SetWordIndirect (052, FDD_NO_DISK);
		return;
	}

	//if img is open
	int pos = ((dt.track_num * 2) + dt.disk_side_num) * dt.sector_per_track * 512 + (dt.sector_num - 1) * 512;

	imgFile.Seek (pos, SEEK_SET);

	BOOL bRead = TRUE;
	int addr = dt.working_addr;
	int length = dt.working_length;

	if (dt.working_length < 0)
	{
		bRead = FALSE;
		length = 0 - dt.working_length;
	}

	try
	{
		for (i = 0; i < length; i++)
		{	
			WORD word;
			int count = sizeof (WORD);
			
			if (bRead)
			{
				count = imgFile.Read (&word, sizeof (WORD));
				SetWord (addr, word);
			}
			else
			{
				word = GetWord (addr);
				imgFile.Write (&word, sizeof (WORD));
			}

			if (count != sizeof (WORD))
				throw 0;

			addr += 2;
		}
	}
	catch (...)
	{
		m_cpu.SetPSW (psw | PSW_C);
		SetWordIndirect (052, FDD_STOP);
		return;
	}


	m_cpu.SetPSW (psw & ~PSW_C);
	SetWordIndirect (052, FDD_NOERROR);
	
	imgFile.Close ();
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard_FDD::ReadDrivesPath ()
{
	CString strIniFileName;
	strIniFileName.LoadString (IDS_INI_FILENAME);

	m_strDrives[0] = GetCurrentPath () + GetIniFileString (IDS_INI_DRIVES, IDS_INI_DRIVEA, GetCurrentPath () + strIniFileName);
	m_strDrives[1] = GetCurrentPath () + GetIniFileString (IDS_INI_DRIVES, IDS_INI_DRIVEB, GetCurrentPath () + strIniFileName);
	m_strDrives[2] = GetCurrentPath () + GetIniFileString (IDS_INI_DRIVES, IDS_INI_DRIVEC, GetCurrentPath () + strIniFileName);
	m_strDrives[3] = GetCurrentPath () + GetIniFileString (IDS_INI_DRIVES, IDS_INI_DRIVED, GetCurrentPath () + strIniFileName);
}



//////////////////////////////////////////////////////////////////////
BOOL CMotherBoard_FDD::RestoreState (CMSFManager& msf, HBITMAP hScreenshot)
{
	if (!CMotherBoard::RestoreState (msf, hScreenshot))
		return FALSE;

	if (msf.IsLoad ())
	{
		if (!msf.GetBlockExt16Memory (&m_bBasic, m_pBasic, m_pExt))
			return FALSE;
	}
	else
	{
		if (!msf.SetBlockExt16Memory (m_bBasic, m_pBasic, m_pExt))
			return FALSE;
	}

	return TRUE;
}
