/*************************************************
*            BK Emulator for Window 3.0          *
*************************************************/

/*
File: Board.cpp
Version: 1.0.0
Written by: Yuriy Kalmykov <kalmykov@stoik.com>
    Copyright (c) 2002-2004 Yuriy Kalmykov

    BK Emulator is a program emulated hardware environment for running
code for BK 0010(01) in different configurations. 
           
    This code may be used in compiled form in any way you desire.
This file or it's parts can't be redistributed without the authors
written consent, but can be modified for your private needs.
    Providing that this notice and the authors name and all copyright
notices remains intact.

    Please, an email me to know how you are using it and where. You can
ask me for any information about this below code or any attendant
knowledge.
    
    This file is provided "as is" with no expressed or implied warranty.
The author accepts no liability for any damage or loss of business that
this product may cause.
*/




// Chip.cpp: implementation of the CMotherBoard class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "BK.h"
#include "Board.h"
#include "Debugger.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//extern CBKApp theApp;

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CMotherBoard::CMotherBoard()
{
	m_cpu.AttachChip (this);
	m_bRunning = FALSE;
	m_bNextCircleInterrupt4 = FALSE;
	m_bAskForBreak = TRUE;

	m_startAddr = 0100000;
	m_nMultiplier = SPEED_PRECISION;	// Use for accelerate or slowdown CPU.
	m_tickCount = 0;	// Current instruction counter. Use for timer.

	m_pScreen = NULL;
	m_pSpeaker = NULL;
	m_pCovox = NULL;
	m_pFDD = NULL;
	m_pDebugger = NULL;
	
	// Initialization of memory module
	m_pMemory = new BYTE [65536 + 16];
	memset (m_pMemory, 0, 65536 + 16);
	
	m_pParent = NULL; // Pointer to window to which notyfication messages send

	m_fKeyboardInterrupt = NO_INTERRUPT;
	m_fGotoAddress = ADDRESS_NONE;
}

CMotherBoard::~CMotherBoard()
{
	delete []m_pMemory;
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::AttachWindow (CWnd* pParent)
{
	m_pParent = pParent;
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::OnReset ()
{
	try
	{
		WORD val = 0;
		BYTE flag = 0;

		for (int i = 0; i < 32768; i += 2, flag--)
		{
			SetWord (i, val);
			
			val = ~val;

			if (flag == 192)
			{
				val = ~val;
				flag = 0;
			}
		}
	}
	catch (CExceptionHalt error)
	{
		CString str;
		str.Format ("%s in address %d", error.m_info, error.m_addr);
		CBKMessageBox message(str);
	}
	
	m_cpu.SetPSW (0340);
	SetWordIndirect (0177660, 0);
	SetWordIndirect (0177662, 0);
	SetWordIndirect (0177664, 01330);
	SetWordIndirect (0177706, 0);
	SetWordIndirect (0177710, 0);
	SetWordIndirect (0177712, 0);
	SetWordIndirect (0177714, 0);
	SetWordIndirect (PC, GetWordIndirect (0177716) & 0177400);
}



//////////////////////////////////////////////////////////////////////
BYTE CMotherBoard::GetByte (int addr)
{
	addr = ConvertAddress (addr);

	int iMemPage = addr >> 13;

	// system register
	if (iMemPage == 7 && addr >= 0177660 && addr < 0200000)
		OnGetSystemRegister (addr & 0377776);

	return m_pMemory[addr];
}



//////////////////////////////////////////////////////////////////////
WORD CMotherBoard::GetWord (int addr)
{
	addr = ConvertAddress (addr);

	int iMemPage = addr >> 13;
	addr &= 0377776;

	// system register
	if (iMemPage == 7 && addr >= 0177660 && addr < 0200000)
		OnGetSystemRegister (addr & 0377776);

	return *(WORD*)&m_pMemory[addr];
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::GetByte (int addr, BYTE* pValue)
{
	*pValue = GetByte (addr);
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::GetWord (int addr, WORD* pValue)
{
	*pValue = GetWord (addr);
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::SetByte (int addr, BYTE value)
{
	addr = ConvertAddress (addr);

	int iMemPage = addr >> 13;

	// memory on 0 - 040000
	if ((iMemPage >= 0 && iMemPage < 2) || iMemPage == 8)
	{
		m_pMemory[addr] = value;
		return;
	}

	// screen memory on 040000 - 0100000
	if (iMemPage >= 2 && iMemPage < 4)
	{
		OnSetScreenCell (addr, value);
		m_pMemory[addr] = value;
		return;
	}

	// ROM memory on 0100000 - 0160000
	if (iMemPage >= 4 && iMemPage < 7)
		throw CExceptionHalt (addr, "Can't write this address.");

	if (IsRegister (addr))
		OnSetSystemRegister (addr, (WORD*)&m_pMemory[addr], value);
	else
		throw CExceptionHalt (addr, "Can't write this address.");
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::SetWord (int addr, WORD value)
{
	addr = ConvertAddress (addr);

	int iMemPage = addr >> 13;
	addr &= 0377776;

	// memory on 0 - 040000
	if ((iMemPage >= 0 && iMemPage < 2) || iMemPage == 8)
	{
		*(WORD*)&m_pMemory[addr] = value;
		return;
	}

	// screen memory on 040000 - 0100000
	if (iMemPage >= 2 && iMemPage < 4)
	{
		OnSetScreenCell (addr, value);
		*(WORD*)&m_pMemory[addr] = value;
		return;
	}

	// ROM memory on 0100000 - 0160000
	if (iMemPage >= 4 && iMemPage < 7)
		throw CExceptionHalt (addr, "Can't write this address.");

	if (IsRegister (addr))
		OnSetSystemRegister (addr, (WORD*)&m_pMemory[addr], value);
	else
		throw CExceptionHalt (addr, "Can't write this address.");
}



//////////////////////////////////////////////////////////////////////
BYTE CMotherBoard::GetByteIndirect (int addr)
{
	addr = ConvertAddress (addr);

	return m_pMemory[addr];
}



//////////////////////////////////////////////////////////////////////
WORD CMotherBoard::GetWordIndirect (int addr)
{
	addr = ConvertAddress (addr);

	return *(WORD*)&m_pMemory[addr & 0377776];
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::SetByteIndirect (int addr, BYTE value)
{
	addr = ConvertAddress (addr);

	m_pMemory[addr] = value;
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::SetWordIndirect (int addr, WORD value)
{
	addr = ConvertAddress (addr);

	*(WORD*)&m_pMemory[addr & 0377776] = value;
}



//////////////////////////////////////////////////////////////////////
WORD CMotherBoard::GetPSW ()
{
	return m_cpu.GetPSW ();
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::SetPSW (WORD value)
{
	m_cpu.SetPSW (value);
}



//////////////////////////////////////////////////////////////////////
BOOL CMotherBoard::IsRegister (int addr)
{
	if (addr >= 0177600)
	{
		switch (addr)
		{
		case 0177660:
		case 0177662:
		case 0177664:
		case 0177706:
		case 0177710:
		case 0177712:
		case 0177714:
		case 0177716:
			return TRUE;
		}
	}

	return FALSE;
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::ResetHot (WORD addrStart)
{
	if (addrStart == 0)
		addrStart = m_startAddr; // Use standart address

	SetWordIndirect (0177716, (addrStart & 0177400) | 0300);

	InitMemoryModules ();
	Reset ();

	if (m_pSpeaker)
		m_pSpeaker->Reset ();

	if (m_pCovox)
		m_pCovox->Reset ();
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::ResetCold (WORD addrStart)
{
	if (addrStart == 0)
		addrStart = m_startAddr; // Use standart address

	SetWordIndirect (0177716, (addrStart & 0177400) | 0300);
	SetWordIndirect (PC, GetWordIndirect (0177716) & 0177400);
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::RefreshScreen ()
{
	for (int i = 040000; i < 0100000; i += 2)
	{
		SetWord (i, GetWord (i));
	}

	// Refresh screen scroll position
	SetWord (0177664, GetWord (0177664));
}



//////////////////////////////////////////////////////////////////////
int CMotherBoard::ExecuteNextInstruction ()
{	
	int nTicks = 0;

	try
	{
		// Make necessary interrupts
		if (m_fKeyboardInterrupt)
		{
			// Get keyboard register
			WORD reg177660 = GetWordIndirect (0177660);

			// Get PSW
			WORD psw = GetPSW ();

			// if keyboard interrupt bit is "0" make keyboard interrupt
			if ((!(reg177660 & 0100)) && (!(psw & PSW_P)))
			{
				MakeInterrupt (m_fKeyboardInterrupt);
				m_fKeyboardInterrupt = NO_INTERRUPT;
			}
		}

		// If debug cursor address break;
		if (m_fGotoAddress == GetPrevPC ())
		{
			BreakCPU ();
			return nTicks;
		}

		if (m_pDebugger && m_fGotoAddress == ADDRESS_NONE)
		{
			// Ask debugger for break condition
			m_pDebugger->SetWord (DEBUG_PC_BREAK, GetPrevPC ()); 

			WORD bBreak = FALSE;
			m_pDebugger->GetWord (DEBUG_PC_BREAK, &bBreak);
			
			if (bBreak)
			{
				// If TRUE break
				BreakCPU ();
				return nTicks;
			}
		}
		
		m_cpu.TranslateInstruction (); // Translate instruction
		Interception ();

		int mod = 10000000 / (CPU_TICKS_PER_SET * CPU_SETS_PER_SECOND);
		nTicks = m_cpu.GetInternalTicks () / mod;
		m_cpu.ResetInternalTicks (mod);

		//m_tickCount++;
		
		for (int n = 0; n < nTicks; n++)
		{
			// Increase devices ticks
			if (m_pSpeaker)
				m_pSpeaker->NextTick ();

			if (m_pCovox)
				m_pCovox->NextTick ();

			if (!(m_tickCount % 6))
				Timerprocess (); // Increase timer

			m_tickCount++;
		}

		if (m_cpu.IsTrace ())
		{
			// If trace flag is set do interrupt 14
			// and previous instruction is not RTT
			m_cpu.MakeInterrupt (INTERRUPT_14);
		}

		// Save current PC for debug
		m_previousPC = GetWordIndirect (PC);

		// Debug break if one step only
		if (m_fGotoAddress == GO_INTO)
		{
			BreakCPU ();
			return nTicks;
		}

		// Debug break if out command
		if (m_fGotoAddress == GO_OUT && CDebugger::IsInstructionOut (m_cpu.GetCurrentInstruction ()))
		{
			BreakCPU ();
			return nTicks;
		}
	}
	catch (CExceptionHalt error)
	{
		MakeInterrupt (INTERRUPT_4);
	}

	return nTicks;
}


/*
//////////////////////////////////////////////////////////////////////
void CMotherBoard::ExternalInterrupt (WORD interrupt)
{
	m_cpu.MakeInterrupt (interrupt);
}
*/



//////////////////////////////////////////////////////////////////////
void CMotherBoard::StopInterrupt ()
{
	m_cpu.StopInterrupt ();
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::KeyboardInterrupt (WORD interrupt)
{
	m_fKeyboardInterrupt = interrupt;

	m_cpu.KeyboardInterript ();
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::AccelerateCPU ()
{	
	if (m_nMultiplier < SPEED_PRECISION * SPEED_STEP)
		m_nMultiplier += SPEED_PRECISION / SPEED_STEP;
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::SlowdownCPU ()
{	
	if (m_nMultiplier > SPEED_PRECISION / SPEED_STEP) 
		m_nMultiplier -= SPEED_PRECISION / SPEED_STEP;
}



//////////////////////////////////////////////////////////////////////
BOOL CMotherBoard::CanAccelerate ()
{
	if (m_nMultiplier >= SPEED_PRECISION * SPEED_STEP)
		return FALSE;

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CMotherBoard::CanSlowDown ()
{
	if (m_nMultiplier <= SPEED_PRECISION / SPEED_STEP)
		return FALSE;

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::NormalizeCPU ()
{
	m_nMultiplier = SPEED_PRECISION;
}



//////////////////////////////////////////////////////////////////////
float CMotherBoard::GetCPUSpeed ()
{
	return (float)m_nMultiplier / SPEED_PRECISION;
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::MakeInterrupt (WORD interrupt)
{
	m_cpu.MakeInterrupt (interrupt);
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::OnSetSystemRegister (WORD num, WORD* pDst, WORD src)
{
	WORD mask = 0177777;
	WORD old = GetWordIndirect (num);

	switch (num)
	{
	case 0177660:
		mask = 0100;
		break;

	case 0177662:
		mask = 0;
		throw CExceptionHalt (num, "Can't write this address.");
		
	case 0177664:
		//src &= (old & 01000) | (src & ~0400);
		mask = 01377;

		if (m_pScreen)
			m_pScreen->SetWord (0177664, src);
		break;

	case 0177706:
		mask = 0177777;
		break;

	case 0177710:
		return;

	case 0177712:
		mask = 0377;
		SetWordIndirect (0177710, GetWordIndirect (0177706));
		break;

	case 0177714:
		if (m_pCovox)
			m_pCovox->SetWord (0177714, src);
		return;

	case 0177716:

		if (m_pSpeaker)
			m_pSpeaker->SetWord (0177716, src);
		return;
	}

	*pDst = (WORD)((~mask & old) | (mask & src));
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::OnGetSystemRegister (WORD num)
{
	if (num == 0177662)
		SetWordIndirect (0177660, GetWordIndirect (0177660) & ~0200);

	if (num == 0177716)
	{
		WORD word177716 = GetWordIndirect (0177716);
		m_pSpeaker->GetWord (0177716, &word177716);
		SetWordIndirect (0177716, word177716);
	}
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::OnSetScreenCell (WORD addr, BYTE byte)
{
	if (m_pScreen)
		m_pScreen->SetByte (addr, byte);
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::OnSetScreenCell (WORD addr, WORD word)
{
	if (m_pScreen)
		m_pScreen->SetWord (addr, word);
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::Timerprocess ()
{
	static int speed = 64;

	int dec = 64;
	
	WORD reg_177706 = GetWordIndirect (0177706); // Initialize
	WORD reg_177710 = GetWordIndirect (0177710); // Timer
	WORD reg_177712 = GetWordIndirect (0177712); // Control
	
	if (reg_177712 & 040)
		//dec >>= 2; // Divide speed by 4
		dec >>= 4;

	if (reg_177712 & 0100)
		//dec >>= 4; // Divide speed by 16
		dec >>= 2;
		
	if ((reg_177712 & 1) == 1 || (reg_177712 & 020) == 0)
		SetWordIndirect (0177710, reg_177706);
	else
	{
		speed -= dec; // Decriment timer

		if (speed <= 0)
		{
			// Decrease current timer value by 1
			WORD word = reg_177710 - 1;
			
			if (!word) // If timer zero
			{
				if ((reg_177712 & 02) == 0)
				{
					// If no prohibit zero pass
					word = reg_177706;

					if (reg_177712 & 04)
						SetWordIndirect (0177712, reg_177712 | 0200);

					if (reg_177712 & 010)
						SetWordIndirect (0177712, reg_177712 & ~020);
				}
			}
			
			SetWordIndirect (0177710, word);

			speed = 64;
		}
	}
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::RunCPU ()
{
	if (m_pParent)
		m_pParent->SendMessage (WM_CPU_RUN);

	//WORD reg177716 = GetWordIndirect (0177716);
	//SetWordIndirect (0177716, reg177716 | 0100);

	//PlaySound ();

	// Save previous PC state for DEBUG
	m_previousPC = GetWordIndirect (PC);

	// Set running flag
	m_bRunning = TRUE;
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::RunInto ()
{
	// Run one commant with go into command
	m_fGotoAddress = GO_INTO;	

	RunCPU ();
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::RunOver ()
{
	// Run one commant with go over command
	if (CDebugger::IsInstructionOver (GetWordIndirect (GetWordIndirect (PC))))
		RunToAddr (GetWordIndirect (PC) + CDebugger::CalcInstructionLength (GetWordIndirect (GetWordIndirect (PC))));
	else
		RunInto ();
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::RunToAddr (WORD addr)
{
	// Run all commands to address
	m_fGotoAddress = addr;

	RunCPU ();
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::RunOut ()
{
	// Run all commands to function end
	m_fGotoAddress = GO_OUT;
	CDebugger::InitOutMode ();

	RunCPU ();
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::StopCPU ()
{
	if (m_pParent)
		m_pParent->SendMessage (WM_CPU_STOP);
	
	m_fGotoAddress = ADDRESS_NONE;
	m_bRunning = FALSE;
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::BreakCPU ()
{
	StopCPU ();

	CDebugger::InitOutMode ();

	if (m_pParent)
		m_pParent->SendMessage (WM_CPU_DEBUGBREAK);
}



//////////////////////////////////////////////////////////////////////
BOOL CMotherBoard::IsCPURun ()
{
	return m_bRunning;
}


//////////////////////////////////////////////////////////////////////
void CMotherBoard::ExecuteInstructionSet (int count, BOOL bCount)
{
	//int nInstructions = 0;

	//Multiply on speed
	count = count * m_nMultiplier / SPEED_PRECISION;

	while (count > 0)
	{
		int nTicks = 0;

		try
		{
			if (m_bNextCircleInterrupt4)
			{
				// If previous instruction make halt interruption 4 
				MakeInterrupt (INTERRUPT_4);
				m_bNextCircleInterrupt4 = FALSE;
			}

			// Execute current instruction
			nTicks = ExecuteNextInstruction ();
			//nInstructions++;

			if (!m_bRunning)
			{
				if (m_pParent)
					m_pParent->SendMessage (WM_CPU_DEBUGBREAK);

				break; // Any debug condition occure
			}
		}
		catch (CExceptionHalt halt)
		{
			// BK Violation exception. For Example ROM memory write or etc.
			WORD pc = GetPrevPC ();
			
			CString strMessage;
			strMessage.Format (IDS_ERRMSG_ROM, halt.m_addr, pc, halt.m_info);
			
			if (m_bAskForBreak)
			{
				CBKMessageBox message;
				int answer = message.Show (strMessage, MB_ABORTRETRYIGNORE|MB_DEFBUTTON2|MB_ICONSTOP);
				
				if (answer == IDABORT)
				{
					BreakCPU (); // Stop CPU for debug
					break; // Exit instruction set
				}
				else 
				{
					m_bNextCircleInterrupt4 = TRUE; // Do interrupt 4 (halt) on next circle
				
					if (answer == IDIGNORE)
						m_bAskForBreak = FALSE;
				}
			}
			else
				m_bNextCircleInterrupt4 = TRUE; // Do interrupt 4 (halt) on next circle
		}
		catch (...)
		{
			// Any ather exception. Unknown exception or WINDOWS memory access violation.
			// This error may be BK unknown error or my emulator error
			WORD pc = GetPrevPC ();

			CString strMessage;
			strMessage.LoadString (IDS_ERRMSG_INTERNAL);

			CBKMessageBox (strMessage, MB_OK|MB_ICONSTOP);
			BreakCPU ();
			break; // Exit instruction set
		}

		count -= nTicks;
	}

	//TRACE ("\nInstructions = %i", nInstructions);
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::Interception ()
{
	switch (GetWordIndirect (PC))
	{
	case 0116076:
		{
			if (m_pParent)
				m_pParent->SendMessage (WM_TAPE_FILE_LOAD);
		break;
		}
	}	
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::LoadRomModule (int iniRomNameIndex, WORD addr)
{
	CString strFileName;
	strFileName.LoadString (IDS_INI_FILENAME);

	CString strPath = GetCurrentPath () +
		GetIniFileString (IDS_INI_ROMMODULES, iniRomNameIndex, 
		GetCurrentPath () + strFileName);

	CFile file;
	if (!file.Open (strPath, CFile::modeRead))
	{
		CString strError;
		strError.LoadString (IDS_ERROR_CANTOPENFILE);
		throw CExceptionFile (strError + _T('\'') + strPath + _T('\''));
	}

	file.Read (&m_pMemory[addr], file.GetLength ());
	file.Close ();
}



//////////////////////////////////////////////////////////////////////
void CMotherBoard::InitMemoryModules ()
{
	LoadRomModule (IDS_INI_MONITOR, 0100000);
	LoadRomModule (IDS_INI_BASIC0, 0120000);
	LoadRomModule (IDS_INI_BASIC1, 0140000);
	LoadRomModule (IDS_INI_REGISTERS, 0160000);
}



//////////////////////////////////////////////////////////////////////
BOOL CMotherBoard::RestoreState (CMSFManager& msf, HBITMAP hScreenshot)
{
	if (!RestorePreview (msf, hScreenshot))
		return FALSE;

	if (!RestoreRegisters (msf))
		return FALSE;

	if (!RestoreMemory (msf))
		return FALSE;

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CMotherBoard::RestoreRegisters (CMSFManager& msf)
{
	if (msf.IsLoad ())
	{
		MSF_CPU_REGISTERS cpu_reg;
		if (!msf.GetBlockCPURegisters (&cpu_reg))
			return FALSE;

		SetWordIndirect (R0, cpu_reg.r0);
		SetWordIndirect (R1, cpu_reg.r1);
		SetWordIndirect (R2, cpu_reg.r2);
		SetWordIndirect (R3, cpu_reg.r3);
		SetWordIndirect (R4, cpu_reg.r4);
		SetWordIndirect (R5, cpu_reg.r5);
		SetWordIndirect (SP, cpu_reg.sp);
		SetWordIndirect (PC, cpu_reg.pc);
		SetPSW (cpu_reg.psw);
	}
	else
	{
		MSF_CPU_REGISTERS cpu_reg;

		cpu_reg.r0 = GetWordIndirect (R0);
		cpu_reg.r1 = GetWordIndirect (R1);
		cpu_reg.r2 = GetWordIndirect (R2);
		cpu_reg.r3 = GetWordIndirect (R3);
		cpu_reg.r4 = GetWordIndirect (R4);
		cpu_reg.r5 = GetWordIndirect (R5);
		cpu_reg.sp = GetWordIndirect (SP);
		cpu_reg.pc = GetWordIndirect (PC);
		cpu_reg.psw = GetPSW ();

		if (!msf.SetBlockCPURegisters (&cpu_reg))
			return FALSE;
	}

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CMotherBoard::RestoreMemory (CMSFManager& msf)
{
	if (msf.IsLoad ())
	{
		if (!msf.GetBlockBaseMemory (m_pMemory))
			return FALSE;
	}
	else
	{
		if (!msf.SetBlockBaseMemory (m_pMemory))
			return FALSE;
	}

	return TRUE;
}



//////////////////////////////////////////////////////////////////////
BOOL CMotherBoard::RestorePreview (CMSFManager& msf, HBITMAP hScreenshot)
{
	if (msf.IsSave ())
	{
		ASSERT (hScreenshot);
		
		CClientDC dc(NULL);
		CDC memDC;
		memDC.CreateCompatibleDC (&dc);

		BITMAPINFOHEADER bmih;
		ZeroMemory (&bmih, sizeof (BITMAPINFOHEADER));
		bmih.biSize = sizeof (BITMAPINFOHEADER);
		bmih.biWidth = 200;
		bmih.biHeight = 200;
		bmih.biCompression = BI_RGB;
		bmih.biPlanes = 1;
		bmih.biBitCount = 32;

		int bpr = (bmih.biWidth * (bmih.biBitCount / 8) + 3) / 4 * 4;
		int length = bpr * bmih.biHeight;

		void* pBits;
		HBITMAP hBm = ::CreateDIBSection (NULL, (BITMAPINFO*)&bmih, DIB_RGB_COLORS, (void**)&pBits, NULL, 0);

		if (!hBm)
			return FALSE;

		::SumStretchBlt (hBm, hScreenshot);

		if (!msf.SetBlockPreview (hBm))
			return FALSE;
	}

	return TRUE;
}
