/*************************************************
*            BK Emulator for Window 3.0          *
*************************************************/

/*
File: BkSound.cpp
Version: 1.0.0
Written by: Yuriy Kalmykov <kalmykov@stoik.com>
    Copyright (c) 2002-2004 Yuriy Kalmykov

    BK Emulator is a program emulated hardware environment for running
code for BK 0010(01) in different configurations. 
           
    This code may be used in compiled form in any way you desire.
This file or it's parts can't be redistributed without the authors
written consent, but can be modified for your private needs.
    Providing that this notice and the authors name and all copyright
notices remains intact.

    Please, an email me to know how you are using it and where. You can
ask me for any information about this below code or any attendant
knowledge.
    
    This file is provided "as is" with no expressed or implied warranty.
The author accepts no liability for any damage or loss of business that
this product may cause.
*/




// BkSound.cpp: implementation of the CBkSound class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "Bk.h"
#include "BkSound.h"

extern CBKApp theApp;

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CBkSound::CBkSound (CWnd* pWnd, DWORD cpu_sps)
{
	m_cpu_sps = cpu_sps;
	
	CDSTHResult hr;

	// Init members
	m_bBufferReady = FALSE;
	m_bSoundEnable = TRUE;
	
	m_wfx.cbSize			= sizeof (WAVEFORMATEX);
	m_wfx.wFormatTag		= WAVE_FORMAT_PCM;
	m_wfx.nSamplesPerSec	= (BUFFER_FREQUENCY + m_cpu_sps - 1) / m_cpu_sps * m_cpu_sps;
	m_wfx.wBitsPerSample	= BUFFER_BPS;
	m_wfx.nChannels			= 1;
	m_wfx.nBlockAlign		= (m_wfx.wBitsPerSample >> 3) * m_wfx.nChannels;
	m_wfx.nAvgBytesPerSec	= m_wfx.nSamplesPerSec * m_wfx.nBlockAlign;

	m_pSampleBuffer = new BYTE[GetSampleCount ()];

	try
	{
		// Init Direct Sound
		hr = DirectSoundCreate (NULL, &m_pDirectSound, NULL);
		hr = m_pDirectSound->SetCooperativeLevel (pWnd->m_hWnd, DSSCL_EXCLUSIVE);
		
		m_BD.dwSize				= sizeof (DSBUFFERDESC);
		m_BD.dwFlags			= DSBCAPS_GLOBALFOCUS|DSBCAPS_GETCURRENTPOSITION2|DSBCAPS_CTRLPOSITIONNOTIFY;
		m_BD.dwReserved			= 0;
		m_BD.dwBufferBytes		= m_wfx.nAvgBytesPerSec;
		m_BD.lpwfxFormat		= &m_wfx;

		// Create Secondary Direct Sound buffer
		hr = m_pDirectSound->CreateSoundBuffer (&m_BD, &m_pBuffer, NULL);

		// Create buffer notify object
		hr = m_pBuffer->QueryInterface (IID_IDirectSoundNotify, (void**)&m_pNotify);
		
		InitEvents ();

		hr = m_pNotify->SetNotificationPositions (m_cpu_sps, m_pNotifyPositions);

		// Init buffer by all zero
		void* pBuff = NULL;
		DWORD size = 0;
		m_pBuffer->Lock (0, m_wfx.nAvgBytesPerSec, &pBuff, &size, NULL, NULL, DSBLOCK_ENTIREBUFFER);
		memset (pBuff, 0, size);
		m_pBuffer->Unlock (pBuff, size, NULL, 0);

		// Play buffer infinite loop
		hr = m_pBuffer->Play (0, 0, DSBPLAY_LOOPING);

		m_bEmulateSoundcard = FALSE;
	}
	catch (CDSTFailQueryInterfaceException)
	{
		InitEvents ();

		m_bEmulateSoundcard = TRUE;
	}

	// Start notify event thread
	m_pNotifyThread = AfxBeginThread ((AFX_THREADPROC)EventThread, (void*)this, THREAD_PRIORITY_NORMAL);
	m_pNotifyThread->m_bAutoDelete = FALSE;

	m_hBufferReadyEvent = ::CreateEvent (NULL, FALSE, FALSE, NULL);
}

CBkSound::~CBkSound()
{
	// Waif for event thread finish
	::SetEvent (m_pNotifyEvents[0]);
	::WaitForSingleObject (*m_pNotifyThread, INFINITE);
	delete m_pNotifyThread;
	
	// Stop playing
	m_pBuffer->Stop ();

	// Release Direct Sound objects
	m_pNotify.Release ();

	for (int i = 0; i < m_cpu_sps + 1; i++) // + 1 Exit event
		::CloseHandle (m_pNotifyEvents[i]);

	::CloseHandle (m_hBufferReadyEvent);
	m_hBufferReadyEvent = NULL;

	delete []m_pNotifyPositions;
	delete []m_pNotifyEvents;

	m_pBuffer.Release ();
	m_pDirectSound.Release ();

	delete []m_pSampleBuffer;
}



//////////////////////////////////////////////////////////////////////
void CBkSound::InitEvents ()
{
	m_pNotifyPositions = new DSBPOSITIONNOTIFY[m_cpu_sps];
	m_pNotifyEvents = new HANDLE[m_cpu_sps + 1]; // + Exit event

	for (int i = 0; i < m_cpu_sps + 1; i++)
		m_pNotifyEvents[i] = ::CreateEvent (NULL, FALSE, FALSE, NULL);
	
	int position = 0;
	int step = BUFFER_FREQUENCY * m_wfx.nBlockAlign / m_cpu_sps;

	for (i = 0; i < m_cpu_sps; i++)
	{
		m_pNotifyPositions[i].dwOffset = position;
		m_pNotifyPositions[i].hEventNotify = m_pNotifyEvents[i + 1];

		position += step;
	}
}



//////////////////////////////////////////////////////////////////////
void CBkSound::EventThread (CBkSound* pThis)
{
	DWORD ds_buffer_size = pThis->m_wfx.nAvgBytesPerSec;
	DWORD buffer_size = ds_buffer_size / pThis->m_cpu_sps;
	ASSERT (buffer_size * pThis->m_cpu_sps == ds_buffer_size);

#ifdef _DEBUG
	int old_pos = ds_buffer_size - buffer_size;
#endif

	CDSTHResult hr;
	
	while (1)
	{
		if (!pThis->m_bEmulateSoundcard)
		{
			int object = ::WaitForMultipleObjects (pThis->m_cpu_sps + 1, pThis->m_pNotifyEvents, FALSE, INFINITE);

			// If exit event
			if (object == WAIT_OBJECT_0)
				break;

			// Get current playing position
			DWORD buffer_pos = (object - WAIT_OBJECT_0 - 1) * buffer_size;

	#ifdef _DEBUG
			//ASSERT ((old_pos + buffer_size) % ds_buffer_size == buffer_pos);
			old_pos = buffer_pos;
	#endif

			// Lock and clear buffer
			int write_pos = (buffer_pos + buffer_size) % ds_buffer_size;

			void* pBuff = NULL;
			DWORD size = 0;

			if (pThis->m_pBuffer->Lock (write_pos, buffer_size, &pBuff, &size, NULL, NULL, 0) == DSERR_BUFFERLOST)
			{
				// If buffer lost
				hr = pThis->m_pBuffer->Restore ();
				hr = pThis->m_pBuffer->Lock (write_pos, buffer_size, &pBuff, &size, NULL, NULL, 0);
			}

			ASSERT (size == buffer_size);
		
			if (pThis->m_bBufferReady)
			{
				CSingleLock lock (&pThis->m_csModifyBuffer, TRUE);
			
				// If data ready copy it to buffer
				// CS is extra protection, but not nessary, couse SendBuffer called direct after "Buffer event"
				memcpy (pBuff, pThis->m_pSampleBuffer, buffer_size);
				
				pThis->m_bBufferReady = FALSE;
			}
			else
			{
				CSingleLock lock (&pThis->m_csModifyBuffer, TRUE);
				// If data not ready fill the buffer all zero
				memset (pBuff, 0, buffer_size);
			}

			
			hr = pThis->m_pBuffer->Unlock (pBuff, size, NULL, 0);
		}
		else
		{
			// If no sound card emulate it
			int delay = 1000 / pThis->m_cpu_sps;
			int currTick = GetTickCount ();
			int waitTime = (currTick + delay - 1) / delay * delay - currTick;
	
			if (::WaitForSingleObject (pThis->m_pNotifyEvents[0], waitTime) == WAIT_OBJECT_0)
				break;
		}

		// Set "Buffer ready" event
		::SetEvent (pThis->m_hBufferReadyEvent);
	}
}



//////////////////////////////////////////////////////////////////////
void CBkSound::WaitAndSendBuffer (BYTE* pBuffer)
{
	if (!m_hBufferReadyEvent)
		return;

	::WaitForSingleObject (m_hBufferReadyEvent, INFINITE);

	// CS is extra protection, but not nessary, couse SendBuffer called direct after "Buffer event"
	CSingleLock lock (&m_csModifyBuffer, TRUE);
	memcpy (m_pSampleBuffer, pBuffer, GetSampleCount ());

	m_bBufferReady = TRUE;
}



//////////////////////////////////////////////////////////////////////
BYTE* CBkSound::GetSampleBuffer ()
{
	return m_pSampleBuffer;
}



//////////////////////////////////////////////////////////////////////
int	CBkSound::GetSampleCount ()
{
	return m_wfx.nAvgBytesPerSec / m_cpu_sps;
}
